"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Simulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const data_type_1 = require("../domain/data-type");
const response_code_1 = require("../domain/response-code");
const automation_simulation_1 = require("./document/automation-simulation");
const command_simulation_1 = require("./document/command-simulation");
class Simulation {
    constructor(document, props) {
        this.document = document;
        this.props = props;
    }
    static ofAutomation(document, props) {
        return new Simulation(document, props);
    }
    static ofCommand(document, props) {
        return new Simulation(document, props);
    }
    /**
     * Synthesize before calling this function!
     * You can use this to Synthesize: SynthUtils.synthesize(stack);
     *
     * Executes the SSM Document in simulation mode.
     * This method DOES NOT result in invocation of SSM APIs.
     * Rather, all steps are executed locally and mimic the behavior of SSM.
     * If any inputs are not provided in this function, the specified defaults for the inputs will be used.
     * @param inputs the inputs to feed into the simulated execution.
     * @returns the outputs of all the executed steps.
     */
    simulate(inputs) {
        this.document.docInputs.forEach(docInput => {
            if (inputs[docInput.name] == undefined) {
                inputs[docInput.name] = docInput.defaultValue;
                if (docInput.defaultValue == undefined) {
                    throw new Error(`Value not provided for ${docInput.name} and no default value was provided`);
                }
            }
        });
        for (let key in inputs) {
            const value = inputs[key];
            const matchedInput = this.document.docInputs.filter(docInput => docInput.name == key)[0];
            if (matchedInput == undefined) {
                throw new Error(`Value provided to simulation {${key}:${value}} is not a supported input for this document`);
            }
            if (!(new data_type_1.DataType(matchedInput.inputType).validateType(value))) {
                throw new Error(`Value provided for ${key} was ${value} which does not match type ${matchedInput.inputType}`);
            }
            if (matchedInput.maxItems && Array.isArray(value) && matchedInput.maxItems < value.length) {
                throw new Error(`Values for input ${key} were ${value} which is greater than maxItems: ${matchedInput.maxItems}`);
            }
            if (matchedInput.minItems && Array.isArray(value) && matchedInput.minItems > value.length) {
                throw new Error(`Values for input ${key} were ${value} which is less than minItems: ${matchedInput.maxItems}`);
            }
            if (matchedInput.allowedPattern && !value.match(matchedInput.allowedPattern)) {
                throw new Error(`Value for input ${key} was ${value} which does not match regex ${matchedInput.allowedPattern}`);
            }
            if (matchedInput.allowedValues && !matchedInput.allowedValues.includes(value)) {
                throw new Error(`Value for input ${key} was ${value} which is not among allowedValues ${matchedInput.allowedValues}`);
            }
        }
        const simulationResult = this.start(inputs);
        var outputValues = [];
        if (simulationResult.responseCode == response_code_1.ResponseCode.SUCCESS) {
            outputValues = this.document.docOutputs.map(docOutput => {
                const valueOfOutput = (simulationResult.outputs ?? {})[docOutput.name];
                if (!new data_type_1.DataType(docOutput.outputType).validateType(valueOfOutput)) {
                    throw new Error(`Document output ${docOutput.name} did not match type ${docOutput.outputType}: ${valueOfOutput}`);
                }
                return valueOfOutput;
            });
        }
        return { ...{ documentOutputs: outputValues }, ...simulationResult };
    }
    /**
     * Delegates the execution of the Document to the subclass (Automation, etc).
     * @param inputs a merge of the defined inputs to the document and the default values if not supplied.
     * @returns the outputs that were emitted from all of the steps.
     */
    start(inputs) {
        switch (this.document.documentType()) {
            case ('Automation'):
                return new automation_simulation_1.AutomationSimulation(this.document, this.props).start(inputs);
            case ('Command'):
                return new command_simulation_1.CommandSimulation(this.document, this.props).start(inputs);
            default:
                throw new Error('No simulation available for document of type: ' + this.document.documentType());
        }
    }
}
exports.Simulation = Simulation;
_a = JSII_RTTI_SYMBOL_1;
Simulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.Simulation", version: "0.0.15" };
//# sourceMappingURL=data:application/json;base64,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