from typing import Tuple

import numpy as np
from numba import jit

from odepy.utils import jit_settings

rk_method_signature = "Tuple((float64[:, ::1], float64[::1], float64[::1], float64[::1]," \
    "float64[::1], float64[:, ::1], b1, int64, int64, int64, int64))()"


@jit(rk_method_signature, **jit_settings)
def BS5() -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray,
                      np.ndarray, np.ndarray, bool, int, int, int, int]:
    """Return Bogacki-Shampine 5(4) tableau and configuration parameters.

    Returns
    -------
    A : ndarray, shape (11, 11)
        Runge-Kutta square matrix. The matrix includes the
        extra-coefficients required for dense-outputs.
    B : ndarray, shape (1,)
        Runge-Kutta weights vector. The vector is empty because BS45 is a FSAL method.
    C : ndarray, shape (11,)
        Runge-Kutta nodes vector. The vector includes the nodes required for dense-outputs.
    E : ndarray, shape (8,)
        Coefficients for the 4th order local error estimation.
    E3 : ndarray, shape (13,)
        Not available for BS45.
    P : ndarray, shape (6, 11)
        Dense-output 5th-order interpolating polynomial coefficients.
    FSAL : boolean
        Specifies whether the RK method is FSAL (First Same As Last).
    order : int
        Runge-Kutta order.
    error_estimation_order : int
        Error estimation order.
    extra_stages : int
        Number of additional stages evaluations requested by the dense-output.

    References
    ----------
    .. [1] RKSuite.cpp codes: <http://www.netlib.org/ode/rksuite/>.
    """
    FSAL = True
    stages = 8

    order = 5
    error_estimation_order = 4

    extra_stages = 1

    B = np.empty((1,))
    E3 = np.zeros((1,))

    A = np.zeros((stages+extra_stages, stages+extra_stages))
    A[1, 0] = 1./6
    A[2, :2] = [2./27, 4/27]
    A[3, :3] = [183./1372, -162/343, 1053/1372]
    A[4, :4] = [68./297, -4/11, 42/143, 1960/3861]
    A[5, :5] = [597./22528, 81/352, 63099/585728, 58653/366080, 4617/20480]
    A[6, :6] = [174197./959244, -30942/79937, 8152137/19744439, 666106/1039181,
                -29421./29068, 482048/414219]
    A[7, :7] = [587./8064, 0, 4440339/15491840, 24353/124800, 387/44800,
                2152/5985, 7267/94080]

    # Extra stages for interpolation
    A[8, :8] = [455./6144, 0, 10256301/35409920, 2307361/17971200,
                -387/102400, 73/5130, -7267/215040, 1/32]

    E = np.array([2479/34992, 0, 123/416, 612941/3411720, 43/1440,
                  2272/6561, 79937/1113912, 3293/556956])
    E -= np.array([587/8064, 0, 4440339/15491840, 24353/124800,
                   387/44800, 2152/5985, 7267/94080, 0])

    C = np.array([0, 1/6, 2/9, 3/7, 2/3, 0.75, 1, 1, 0.5])

    P = np.zeros((5, stages+extra_stages))
    P[0, 1] = 1
    P[1] = [-155/36, 0, 6561/988, 2401/702, 0, 1408/513, 0, -1/2, -8]
    P[2] = [16441/2016, 0, -14727987/774592, -603337/56160,
            -387/2240, -45536/3591, -7267/4704, 4, 32]
    P[3] = [-56689/8064, 0, 60538347/3098368, 2740913/224640, 3483/8960,
            67960/3591, 21801/6272, -15/2, -40]
    P[4] = [757/336, 0, -13321017/1936480, -24353/5200, -1161/5600,
            -17216/1995, -7267/3920, 4, 16]

    return (A, B, C, E, E3, P, FSAL, order, error_estimation_order,
            stages, extra_stages,)


@jit(rk_method_signature, **jit_settings)
def DOP853() -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray,
                      np.ndarray, np.ndarray, bool, int, int, int, int]:
    """Return DOP853 tableau and configuration parameters.

    Returns
    -------
    A : ndarray, shape (16, 16)
        Runge-Kutta square matrix. The matrix includes the
        extra-coefficients required for dense-outputs.
    B : ndarray, shape (1,)
        Runge-Kutta weights vector. The vector is empty because DOP853 is a FSAL method.
    C : ndarray, shape (16,)
        Runge-Kutta nodes vector. The vector includes the nodes required for dense-outputs.
    E5 : ndarray, shape (13,)
        Coefficients for the 5th order local error estimation.
    E3 : ndarray, shape (13,)
        Coeffficients for the 3rd order local error estimation.
    P : ndarray, shape (4, 16)
        Dense-output interpolating polynomial coefficients. Despite the interpolating
        polynomial is of 7th order, only four coefficients are given because the first
        three are computed in a different way.
    FSAL : boolean
        Specifies whether the RK method is FSAL (First Same As Last).
    order : int
        Runge-Kutta order.
    error_estimation_order : int
        Error estimation order.
    reverse_interpolation : boolean
        True if the interpolating polynomial coefficients are defined backwards from the
        end of the step rather than forward as in traditional implementations.
    extra_stages : int
        Number of additional stages evaluations requested by the dense-output.

    References
    ----------
    .. [1] E. Hairer, S. P. Norsett G. Wanner, "Solving Ordinary Differential
           Equations I: Nonstiff Problems"
    .. [2] `Original Fortran code of DOP853
            <http://www.unige.ch/~hairer/software.html>`_.
    """

    FSAL = True
    stages = 13

    order = 8
    error_estimation_order = 7

    extra_stages = 3

    B = np.empty((1,))

    A = np.zeros((stages + extra_stages, stages + extra_stages))
    A[1, 0] = 5.26001519587677318785587544488e-2

    A[2, 0] = 1.97250569845378994544595329183e-2
    A[2, 1] = 5.91751709536136983633785987549e-2

    A[3, 0] = 2.95875854768068491816892993775e-2
    A[3, 2] = 8.87627564304205475450678981324e-2

    A[4, 0] = 2.41365134159266685502369798665e-1
    A[4, 2] = -8.84549479328286085344864962717e-1
    A[4, 3] = 9.24834003261792003115737966543e-1

    A[5, 0] = 3.7037037037037037037037037037e-2
    A[5, 3] = 1.70828608729473871279604482173e-1
    A[5, 4] = 1.25467687566822425016691814123e-1

    A[6, 0] = 3.7109375e-2
    A[6, 3] = 1.70252211019544039314978060272e-1
    A[6, 4] = 6.02165389804559606850219397283e-2
    A[6, 5] = -1.7578125e-2

    A[7, 0] = 3.70920001185047927108779319836e-2
    A[7, 3] = 1.70383925712239993810214054705e-1
    A[7, 4] = 1.07262030446373284651809199168e-1
    A[7, 5] = -1.53194377486244017527936158236e-2
    A[7, 6] = 8.27378916381402288758473766002e-3

    A[8, 0] = 6.24110958716075717114429577812e-1
    A[8, 3] = -3.36089262944694129406857109825
    A[8, 4] = -8.68219346841726006818189891453e-1
    A[8, 5] = 2.75920996994467083049415600797e1
    A[8, 6] = 2.01540675504778934086186788979e1
    A[8, 7] = -4.34898841810699588477366255144e1

    A[9, 0] = 4.77662536438264365890433908527e-1
    A[9, 3] = -2.48811461997166764192642586468
    A[9, 4] = -5.90290826836842996371446475743e-1
    A[9, 5] = 2.12300514481811942347288949897e1
    A[9, 6] = 1.52792336328824235832596922938e1
    A[9, 7] = -3.32882109689848629194453265587e1
    A[9, 8] = -2.03312017085086261358222928593e-2

    A[10, 0] = -9.3714243008598732571704021658e-1
    A[10, 3] = 5.18637242884406370830023853209
    A[10, 4] = 1.09143734899672957818500254654
    A[10, 5] = -8.14978701074692612513997267357
    A[10, 6] = -1.85200656599969598641566180701e1
    A[10, 7] = 2.27394870993505042818970056734e1
    A[10, 8] = 2.49360555267965238987089396762
    A[10, 9] = -3.0467644718982195003823669022

    A[11, 0] = 2.27331014751653820792359768449
    A[11, 3] = -1.05344954667372501984066689879e1
    A[11, 4] = -2.00087205822486249909675718444
    A[11, 5] = -1.79589318631187989172765950534e1
    A[11, 6] = 2.79488845294199600508499808837e1
    A[11, 7] = -2.85899827713502369474065508674
    A[11, 8] = -8.87285693353062954433549289258
    A[11, 9] = 1.23605671757943030647266201528e1
    A[11, 10] = 6.43392746015763530355970484046e-1

    A[12, 0] = 5.42937341165687622380535766363e-2
    A[12, 5] = 4.45031289275240888144113950566
    A[12, 6] = 1.89151789931450038304281599044
    A[12, 7] = -5.8012039600105847814672114227
    A[12, 8] = 3.1116436695781989440891606237e-1
    A[12, 9] = -1.52160949662516078556178806805e-1
    A[12, 10] = 2.01365400804030348374776537501e-1
    A[12, 11] = 4.47106157277725905176885569043e-2

    # Extra stages required
    A[13, 0] = 5.61675022830479523392909219681e-2
    A[13, 6] = 2.53500210216624811088794765333e-1
    A[13, 7] = -2.46239037470802489917441475441e-1
    A[13, 8] = -1.24191423263816360469010140626e-1
    A[13, 9] = 1.5329179827876569731206322685e-1
    A[13, 10] = 8.20105229563468988491666602057e-3
    A[13, 11] = 7.56789766054569976138603589584e-3
    A[13, 12] = -8.298e-3

    A[14, 0] = 3.18346481635021405060768473261e-2
    A[14, 5] = 2.83009096723667755288322961402e-2
    A[14, 6] = 5.35419883074385676223797384372e-2
    A[14, 7] = -5.49237485713909884646569340306e-2
    A[14, 10] = -1.08347328697249322858509316994e-4
    A[14, 11] = 3.82571090835658412954920192323e-4
    A[14, 12] = -3.40465008687404560802977114492e-4
    A[14, 13] = 1.41312443674632500278074618366e-1

    A[15, 0] = -4.28896301583791923408573538692e-1
    A[15, 5] = -4.69762141536116384314449447206
    A[15, 6] = 7.68342119606259904184240953878
    A[15, 7] = 4.06898981839711007970213554331
    A[15, 8] = 3.56727187455281109270669543021e-1
    A[15, 12] = -1.39902416515901462129418009734e-3
    A[15, 13] = 2.9475147891527723389556272149
    A[15, 14] = -9.15095847217987001081870187138

    C = np.zeros((stages + extra_stages,))
    C[1] = 0.526001519587677318785587544488e-01
    C[2] = 0.789002279381515978178381316732e-01
    C[3] = 0.118350341907227396726757197510
    C[4] = 0.281649658092772603273242802490
    C[5] = 0.333333333333333333333333333333
    C[6] = 0.25
    C[7] = 0.307692307692307692307692307692
    C[8] = 0.651282051282051282051282051282
    C[9] = 0.6
    C[10] = 0.857142857142857142857142857142
    C[11] = 1.0
    C[12] = 1.0
    C[13] = 0.1
    C[14] = 0.2
    C[15] = 7 / 9

    E3 = A[stages - 1, :stages].copy()
    E3[0] -= 0.244094488188976377952755905512
    E3[8] -= 0.733846688281611857341361741547
    E3[11] -= 0.220588235294117647058823529412e-1

    E5 = np.zeros((stages,))
    E5[0] = 0.1312004499419488073250102996e-1
    E5[5] = -0.1225156446376204440720569753e1
    E5[6] = -0.4957589496572501915214079952
    E5[7] = 0.1664377182454986536961530415e1
    E5[8] = -0.3503288487499736816886487290
    E5[9] = 0.3341791187130174790297318841
    E5[10] = 0.8192320648511571246570742613e-1
    E5[11] = -0.2235530786388629525884427845e-1

    # The interpolation order is 7, but the first 3 coefficients
    # are evaluated in a different way
    P = np.zeros((4, stages + extra_stages))
    P[0, 0] = -0.84289382761090128651353491142e1
    P[0, 5] = 0.56671495351937776962531783590
    P[0, 6] = -0.30689499459498916912797304727e1
    P[0, 7] = 0.23846676565120698287728149680e1
    P[0, 8] = 0.21170345824450282767155149946e1
    P[0, 9] = -0.87139158377797299206789907490
    P[0, 10] = 0.22404374302607882758541771650e1
    P[0, 11] = 0.63157877876946881815570249290
    P[0, 12] = -0.88990336451333310820698117400e-1
    P[0, 13] = 0.18148505520854727256656404962e2
    P[0, 14] = -0.91946323924783554000451984436e1
    P[0, 15] = -0.44360363875948939664310572000e1

    P[1, 0] = 0.10427508642579134603413151009e2
    P[1, 5] = 0.24228349177525818288430175319e3
    P[1, 6] = 0.16520045171727028198505394887e3
    P[1, 7] = -0.37454675472269020279518312152e3
    P[1, 8] = -0.22113666853125306036270938578e2
    P[1, 9] = 0.77334326684722638389603898808e1
    P[1, 10] = -0.30674084731089398182061213626e2
    P[1, 11] = -0.93321305264302278729567221706e1
    P[1, 12] = 0.15697238121770843886131091075e2
    P[1, 13] = -0.31139403219565177677282850411e2
    P[1, 14] = -0.93529243588444783865713862664e1
    P[1, 15] = 0.35816841486394083752465898540e2

    P[2, 0] = 0.19985053242002433820987653617e2
    P[2, 5] = -0.38703730874935176555105901742e3
    P[2, 6] = -0.18917813819516756882830838328e3
    P[2, 7] = 0.52780815920542364900561016686e3
    P[2, 8] = -0.11573902539959630126141871134e2
    P[2, 9] = 0.68812326946963000169666922661e1
    P[2, 10] = -0.10006050966910838403183860980e1
    P[2, 11] = 0.77771377980534432092869265740
    P[2, 12] = -0.27782057523535084065932004339e1
    P[2, 13] = -0.60196695231264120758267380846e2
    P[2, 14] = 0.84320405506677161018159903784e2
    P[2, 15] = 0.11992291136182789328035130030e2

    P[3, 0] = -0.25693933462703749003312586129e2
    P[3, 5] = -0.15418974869023643374053993627e3
    P[3, 6] = -0.23152937917604549567536039109e3
    P[3, 7] = 0.35763911791061412378285349910e3
    P[3, 8] = 0.93405324183624310003907691704e2
    P[3, 9] = -0.37458323136451633156875139351e2
    P[3, 10] = 0.10409964950896230045147246184e3
    P[3, 11] = 0.29840293426660503123344363579e2
    P[3, 12] = -0.43533456590011143754432175058e2
    P[3, 13] = 0.96324553959188282948394950600e2
    P[3, 14] = -0.39177261675615439165231486172e2
    P[3, 15] = -0.14972683625798562581422125276e3

    return (A, B, C, E5, E3, P, FSAL, order, error_estimation_order,
            stages, extra_stages,)

@jit(rk_method_signature, **jit_settings)
def Vern7() -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray,
                      np.ndarray, np.ndarray, bool, int, int, int, int]:
    """Return Verner's most efficient 7(6) tableau and configuration parameters.

    Returns
    -------
    A : ndarray, shape (13, 13)
        Runge-Kutta square matrix. The matrix includes the
        extra-coefficients required for dense-outputs.
    B : ndarray, shape (10,)
        Runge-Kutta weights vector. The vector is empty because DOP853 is a FSAL method.
    C : ndarray, shape (13,)
        Runge-Kutta nodes vector. The vector includes the nodes required for dense-outputs.
    E : ndarray, shape (10,)
        Coefficients for the local error estimation.
    E3 : ndarray, shape (1,)
        Not available for Vern7.
    P : ndarray, shape (6, 13)
        Dense-output interpolating 6th order polynomial coefficients.
    FSAL : boolean
        Specifies whether the RK method is FSAL (First Same As Last).
    order : int
        Runge-Kutta order.
    error_estimation_order : int
        Error estimation order.
    extra_stages : int
        Number of additional stages evaluations requested by the dense-output.

    Notes
    -----
    This code implements a 6th order dense-output interpolant, which requires 3 extra
    stages evaluations. A 7th order interpolant is also available, requiring 6 extra
    stages evaluations. The latter has not been implemented here.

    References
    ----------
    .. [1] Jim Verner's Refuge for Runge-Kutta pairs website:
        <https://www.sfu.ca/~jverner/>
    """

    FSAL = False
    stages = 10

    order = 7
    error_estimation_order = 6

    extra_stages = 3

    A = np.zeros((stages + extra_stages, stages + extra_stages))
    A[1, 0] = 0.005

    A[2, 0] = -1.076790123456790123456790123456790123457
    A[2, 1] = 1.185679012345679012345679012345679012346

    A[3, 0] = 0.04083333333333333333333333333333333333333
    A[3, 2] = 0.1225

    A[4, 0] = 0.6389139236255726780508121615993336109954
    A[4, 2] = -2.455672638223656809662640566430653894211
    A[4, 3] = 2.272258714598084131611828404831320283215

    A[5, 0] = -2.661577375018757131119259297861818119279
    A[5, 2] = 10.80451388645613769565396655365532838482
    A[5, 3] = -8.353914657396199411968048547819291691541
    A[5, 4] = 0.8204875949566569791420417341743839209619

    A[6, 0] = 6.067741434696770992718360183877276714679
    A[6, 2] = -24.71127363591108579734203485290746001803
    A[6, 3] = 20.42751793078889394045773111748346612697
    A[6, 4] = -1.906157978816647150624096784352757010879
    A[6, 5] = 1.006172249242068014790040335899474187268

    A[7, 0] = 12.05467007625320299509109452892778311648
    A[7, 2] = -49.75478495046898932807257615331444758322
    A[7, 3] = 41.14288863860467663259698416710157354209
    A[7, 4] = -4.461760149974004185641911603484815375051
    A[7, 5] =  2.042334822239174959821717077708608543738
    A[7, 6] = -0.09834843665406107379530801693870224403537

    A[8, 0] = 10.13814652288180787641845141981689030769
    A[8, 2] = -42.64113603171750214622846006736635730625
    A[8, 3] = 35.76384003992257007135021178023160054034
    A[8, 4] = -4.348022840392907653340370296908245943710
    A[8, 5] = 2.009862268377035895441943593011827554771
    A[8, 6] = 0.3487490460338272405953822853053145879140
    A[8, 7] = -0.2714390051048312842371587140910297407572

    A[9, 0] = -45.03007203429867712435322405073769635151
    A[9, 2] = 187.3272437654588840752418206154201997384
    A[9, 3] = -154.0288236935018690596728621034510402582
    A[9, 4] = 18.56465306347536233859492332958439136765
    A[9, 5] = -7.141809679295078854925420496823551192821
    A[9, 6] = 1.308808578161378625114762706007696696508

    # Extra stages required
    A[10, 0] = 0.04715561848627222170431765108838175679569
    A[10, 3] = 0.2575056429843415189596436101037687580986
    A[10, 4] = 0.2621665397741262047713863095764527711129
    A[10, 5] = 0.1521609265673855740323133199165117535523
    A[10, 6] = 0.4939969170032484246907175893227876844296
    A[10, 7] = -0.2943031171403250441557244744092703429139
    A[10, 8] = 0.08131747232495109999734599440136761892478

    A[11, 0] = 0.05232227691599689815470932256735029887614
    A[11, 3] = 0.2249586182670571550244187743667190903405
    A[11, 4] = 0.01744370924877637539031751304611402542578
    A[11, 5] = -0.007669379876829393188009028209348812321417
    A[11, 6] = 0.03435896044073284645684381456417912794447
    A[11, 7] = -0.04102097230093949839125144540100346681769
    A[11, 8] = 0.02565113300520561655297104906598973655221
    A[11, 10] = -0.160443457

    A[12, 0] = 0.05305334125785908638834747243817578898946
    A[12, 3] = 0.1219530101140188607092225622195251463666
    A[12, 4] = 0.01774684073760249704011573985936092552347
    A[12, 5] = -0.0005928372667681494328907467430302313286925
    A[12, 6] = 0.008381833970853750873624781948796072714855
    A[12, 7] = -0.01293369259698611956700998079778496462996
    A[12, 8] = 0.009412056815253860804791356641605087829772
    A[12, 10] = -0.005353253107275676032399320754008272222345
    A[12, 10] = -0.06666729992455811078380186481263955324311

    C = np.zeros((stages + extra_stages,))
    C[1] = 0.005
    C[2] = 0.1088888888888888888888888888888888888889
    C[3] = 0.1633333333333333333333333333333333333333
    C[4] = 0.4555
    C[5] = 0.6095094489978381317087004421486024949638
    C[6] = 0.884
    C[7] = 0.925
    C[8] = 1.
    C[9] = 1.
    C[10] = 1.
    C[11] = 0.29
    C[12] = 0.125

    B = np.zeros((stages,))
    B[0] = 0.04715561848627222170431765108838175679569
    B[3] = 0.2575056429843415189596436101037687580986
    B[4] = 0.2621665397741262047713863095764527711129
    B[5] = 0.1521609265673855740323133199165117535523
    B[6] = 0.4939969170032484246907175893227876844296
    B[7] = -0.2943031171403250441557244744092703429139
    B[8] = 0.08131747232495109999734599440136761892478

    E3 = np.zeros((1,))

    E = np.zeros((stages,))
    E[0] = 0.002547011879931045
    E[3] = -0.00965839487279575
    E[4] = 0.04206470975639691
    E[5] = -0.0666822437469301
    E[6] = 0.2650097464621281
    E[7] = -0.29430311714032503
    E[8] = 0.08131747232495111
    E[9] = -0.02029518466335628


    P = np.zeros((6, stages + extra_stages))

    P[0, 0] = 1.
    P[1, 0] = -7.579486522562013856370387489358761917136
    P[2, 0] = 24.84859042701758998305254381572114767883
    P[3, 0] = -38.85067748922540724491934058373805286459
    P[4, 0] = 28.75646349856373329638850558427800090199
    P[5, 0] = -8.127734295307629956447003675813952042307

    P[1, 3] =  4.551232240400547830289289496617659221741
    P[2, 3] = -41.74306197989451542904654379577378740985
    P[3, 3] = 125.9208614672431114278470271124781332255
    P[4, 3] =  -143.2724320984988187759537191634829285599
    P[5, 3] = 54.80090601373401646582358996026469228056

    P[1, 4] = 2.347388766837311148481149416205748182220
    P[2, 4] = -21.42965001437327943727021057869980940196
    P[3, 4] = 64.01102018753913070410229000503296635821
    P[4, 4] = -71.54964616066291046169022808133078061275
    P[5, 4] = 26.88305376043387425114838554836832824539

    P[1, 5] = .6629628602943922170984684021172226061984
    P[2, 5] = -5.991791209485099663334522931412659054138
    P[3, 5] = 17.51358806151491209369363907599542681959
    P[4, 5] = -18.79068837634778062158370304672269638025
    P[5, 5] = 6.758089590590961548158431819939217762152

    P[1, 6] = -1.894931483197030289667134961741273285056
    P[2, 6] = 17.84551891193619310038245879582238106213
    P[3, 6] = -56.77440614878830815775145830865327266897
    P[4, 6] = 70.55596299657564872116838561274122139834
    P[5, 6] = -29.23814735952325494944153354884626882202

    P[1, 7] = 1.489077233496668566079506992903531187388
    P[2, 7] = -13.95068086791358924071990604089950620891
    P[3, 7] = 43.93573738643896559384998172064294274652
    P[4, 7] = -53.74155980596578799479262013665711367333
    P[5, 7] = 21.97312293680341803142731298960087560542

    P[1, 8] = -.5934749977615343231780548830714052210176
    P[2, 8] = 5.532214575131257019966932786240153623467
    P[3, 8] = -17.25057884887540359301582894972755244557
    P[4, 8] = 20.76631879735288001882715503942847061836
    P[5, 8] = -8.373162053522248022602857998468298956311

    P[1, 10] = .5229705513661462700547131555553062786922
    P[2, 10] = -4.935085895387111016017254136967884797119
    P[3, 10] = 15.77686659404020360951757145807650520227
    P[4, 10] = -19.84035770738365925120223312747058112796
    P[5, 10] = 8.475606457364420387647202650806654444113

    P[1, 11] = -8.546914399459162196852655523910775929846
    P[2, 11] = 71.42491845557421929764191001202432467195
    P[3, 11] = -182.3743002138117003976306073059834582214
    P[4, 11] = 184.6615026587373916897461066715370461463
    P[5, 11] = -65.16520650104074839290475385366713666699

    P[1, 12] = 9.041175750584674634065105394682748876817
    P[2, 12] = -31.60097240260566461465540792605436016441
    P[3, 12] = 28.09188900392449596430672577587636184837
    P[4, 12] = 2.454436197629303379092350647679361289224
    P[5, 12] = -7.986528549532809362808773892184111850000


    return (A, B, C, E, E3, P, FSAL, order, error_estimation_order,
            stages, extra_stages,)


@jit(rk_method_signature, **jit_settings)
def Vern8() -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray,
                      np.ndarray, np.ndarray, bool, int, int, int, int]:
    """Return Verner's most efficient 8(7) tableau and configuration parameters.

    Returns
    -------
    A : ndarray, shape (17, 17)
        Runge-Kutta square matrix. The matrix includes the
        extra-coefficients required for dense-outputs.
    B : ndarray, shape (13,)
        Runge-Kutta weights vector. The vector is empty because DOP853 is a FSAL method.
    C : ndarray, shape (17,)
        Runge-Kutta nodes vector. The vector includes the nodes required for dense-outputs.
    E : ndarray, shape (13,)
        Coefficients for the local error estimation.
    E3 : ndarray, shape (1,)
        Not available for Vern8.
    P : ndarray, shape (7, 17)
        Dense-output interpolating 7th order polynomial coefficients.
    FSAL : boolean
        Specifies whether the RK method is FSAL (First Same As Last).
    order : int
        Runge-Kutta order.
    error_estimation_order : int
        Error estimation order.
    extra_stages : int
        Number of additional stages evaluations requested by the dense-output.

    Notes
    -----
    This code implements a 7th order dense-output interpolant, which requires 4 extra
    stages evaluations. An 8th order interpolant is also available, requiring 8 extra
    stages evaluations. The latter has not been implemented here.

    References
    ----------
    .. [1] Jim Verner's Refuge for Runge-Kutta pairs website:
        <https://www.sfu.ca/~jverner/>
    """

    FSAL = False
    stages = 13

    order = 8
    error_estimation_order = 7

    extra_stages = 4

    A = np.zeros((stages+extra_stages,stages+extra_stages))
    A[1, 0] = 0.05

    A[2, 0] = -0.69931640625e-2
    A[2, 1] = 0.1135556640625

    A[3, 0] = 0.399609375e-1
    A[3, 2] = 0.1198828125

    A[4, 0] = 0.3613975628004575124052940721184028345129
    A[4, 2] = -1.341524066700492771819987788202715834917
    A[4, 3] = 1.370126503900035259414693716084313000404

    A[5, 0] = 0.490472027972027972027972027972027972028e-1
    A[5, 3] = 0.2350972042214404739862988335493427143122
    A[5, 4] = 0.1808555929813567288109039636534544884850

    A[6, 0] = 0.6169289044289044289044289044289044289044e-1
    A[6, 3] = 0.1123656831464027662262557035130015442303
    A[6, 4] = -0.3885046071451366767049048108111244567456e-1
    A[6, 5] = 0.1979188712522045855379188712522045855379e-1

    A[7, 0] = -1.767630240222326875735597119572145586714
    A[7, 3] = -62.5
    A[7, 4] = -6.061889377376669100821361459659331999758
    A[7, 5] = 5.650823198222763138561298030600840174201
    A[7, 6] = 65.62169641937623283799566054863063741227

    A[8, 0] = -1.180945066554970799825116282628297957882
    A[8, 3] = -41.50473441114320841606641502701994225874
    A[8, 4] = -4.434438319103725011225169229846100211776
    A[8, 5] = 4.260408188586133024812193710744693240761
    A[8, 6] = 43.75364022446171584987676829438379303004
    A[8, 7] = 0.7871425489912310687446475044226307550860e-2

    A[9, 0] = -1.281405999441488405459510291182054246266
    A[9, 3] = -45.04713996013986630220754257136007322267
    A[9, 4] = -4.731362069449576477311464265491282810943
    A[9, 5] = 4.514967016593807841185851584597240996214
    A[9, 6] = 47.44909557172985134869022392235929015114
    A[9, 7] = 0.1059228297111661135687393955516542875228e-1
    A[9, 8] = -0.5746842263844616254432318478286296232021e-2

    A[10, 0] = -1.724470134262485191756709817484481861731
    A[10, 3] = -60.92349008483054016518434619253765246063
    A[10, 4] = -5.951518376222392455202832767061854868290
    A[10, 5] = 5.556523730698456235979791650843592496839
    A[10, 6] = 63.98301198033305336837536378635995939281
    A[10, 7] = 0.1464202825041496159275921391759452676003e-1
    A[10, 8] = 0.6460408772358203603621865144977650714892e-1
    A[10, 9] = -0.7930323169008878984024452548693373291447e-1

    A[11, 0] = -3.301622667747079016353994789790983625569
    A[11, 3] = -118.0112723597525085666923303957898868510
    A[11, 4] = -10.14142238845611248642783916034510897595
    A[11, 5] = 9.139311332232057923544012273556827000619
    A[11, 6] = 123.3759428284042683684847180986501894364
    A[11, 7] = 4.623244378874580474839807625067630924792
    A[11, 8] = -3.383277738068201923652550971536811240814
    A[11, 9] = 4.527592100324618189451265339351129035325
    A[11, 10] = -5.828495485811622963193088019162985703755

    A[12, 0] = -3.039515033766309030040102851821200251056
    A[12, 3] = -109.2608680894176254686444192322164623352
    A[12, 4] = -9.290642497400293449717665542656897549158
    A[12, 5] = 8.430504981764911142134299253836167803454
    A[12, 6] = 114.2010010378331313557424041095523427476
    A[12, 7] = -0.9637271342145479358162375658987901652762
    A[12, 8] = -5.034884088802189791198680336183332323118
    A[12, 9] = 5.958130824002923177540402165388172072794

    # Extra stages
    A[13, 0] = .4427989419007951074716746668098518862111e-1
    A[13, 5] = 0.3541049391724448744815552028733568354121
    A[13, 6] = 0.2479692154956437828667629415370663023884
    A[13, 7] = -15.69420203883808405099207034271191213468
    A[13, 8] = 25.08406496555856261343930031237186278518
    A[13, 9] = -31.73836778626027646833156112007297739997
    A[13, 10] = 22.93828327398878395231483560344797018313
    A[13, 11] = -0.2361324633071542145259900641263517600737

    A[14, 0] = 0.4620700646754963101730413150238116432863e-1
    A[14, 5] = 0.4503904160842480866828520384400679697151e-1
    A[14, 6] = 0.2336816697713424410788701065340221126565
    A[14, 7] = 37.83901368421067410780338220861855254153
    A[14, 8] = -15.94911328945424610266139490307397370835
    A[14, 9] = 23.02836835181610285142510596329590091940
    A[14, 10] = -44.85578507769412524816130998016948002745
    A[14, 11] = -0.6379858768647444009509067402330140781326e-1
    A[14, 13] = -0.1259503554386166268241032464519842162533e-1

    A[15, 0] = 0.5037946855482040993065158747220696112586e-1
    A[15, 5] = 0.4109836131046079339916530614028848248545e-1
    A[15, 6] = .1718054153348195783296309209549424619697
    A[15, 7] = 4.61410531998151886974342237185977124648
    A[15, 8] = -1.791667883085396449712744996746836471721
    A[15, 9] = 2.531658930485041408462243518792913614971
    A[15, 10] = -5.32497786020573071925718815977276269909
    A[15, 11] = -.3065532595385634734924449496356513113607e-1
    A[15, 13] = -.5254479979429613570549519094377878106127e-2
    A[15, 14] = -.8399194644224792997538653464258058697156e-1

    A[16, 0] = .4082897132997079620207118756242653796386e-1
    A[16, 5] = 0.4244479514247632218892086657732332485609
    A[16, 6] = 0.2326091531275234539465100096964845486081
    A[16, 7] = 2.677982520711806062780528871014035962908
    A[16, 8] = 0.7420826657338945216477607044022963622057
    A[16, 9] = 0.1460377847941461193920992339731312296021
    A[16, 10] = -3.579344509890565218033356743825917680543
    A[16, 11] = 0.1138844389600173704531638716149985665239
    A[16, 13] = 0.1267790651033190047378693537615687232109e-1
    A[16, 14] = -0.7443436349946674429752785032561552478382e-1
    A[16, 15] = 0.4782748079757851554575511473876987663388e-1

    C = np.zeros((stages+extra_stages,))
    C[1] = 0.05
    C[2] = 0.1065625
    C[3] = 0.15984375
    C[4] = 0.39
    C[5] = 0.465
    C[6] = 0.155
    C[7] = 0.943
    C[8] = 0.9018020417358569582597079406783721499560
    C[9] = 0.909
    C[10] = 0.940
    C[11] = 1.
    C[12] = 1.
    C[13] = 1.
    C[14] = 0.3110177634953863863927417318829099695921
    C[15] = 0.1725
    C[16] = 0.7846

    B = np.zeros((stages,))
    B[0] = 0.4427989419007951074716746668098518862111e-1
    B[5] = 0.3541049391724448744815552028733568354121
    B[6] = 0.2479692154956437828667629415370663023884
    B[7] = -15.69420203883808405099207034271191213468
    B[8] = 25.08406496555856261343930031237186278518
    B[9] = -31.73836778626027646833156112007297739997
    B[10] = 22.93828327398878395231483560344797018313
    B[11] = -0.2361324633071542145259900641263517600737
    B[12] = 0.

    E = np.zeros((stages,))
    E[0] = -0.0000327210390102875003925930741388583556
    E[5] = -0.0005046250618777614604937298281583935022
    E[6] = 0.0001211723589784885479758713699993677437
    E[7] = -20.1423367713138681267537322128191590309143
    E[8] = 5.2371785994398294405982596799731254577637
    E[9] = -8.1567444087946583408665901515632867813110
    E[10] = 22.9382832739887838613412895938381552696228
    E[11] = -0.2361324633071542056228508954518474638462
    E[12] = 0.3601679437289775354003040774841792881489

    E3 = np.zeros((1,))

    P = np.zeros((7, stages + extra_stages))

    P[0, 0] = 1.
    P[1, 0] = -7.238550783576432811855355839508646327161
    P[2, 0] = 26.00913483254676138219215542805486438340
    P[3, 0] = -50.23684777762566731759165474184543812128
    P[4, 0] = 52.12072084601022449485077581012685809554
    P[5, 0] = -27.06472451211777193118825764262673140465
    P[6, 0] = 5.454547288952965694339504452480078562780

    P[1, 5] = 11.15330887588935170976376962782446833855
    P[2, 5] = -91.7609656398961659890179437322816238711
    P[3, 5] = 291.7074241722059450113911477530513089255
    P[4, 5] = -430.4096692910862817449451677633631387823
    P[5, 5] = 299.4531188198997479843407054776900024282
    P[6, 5] = -79.78911199784015209705095616004766020335

    P[1, 6] = 2.34875229807309355640904629061136935335
    P[2, 6] = -11.6724894172018429369093778842231443146
    P[3, 6] = -3.339139076505928386509206543237093540
    P[4, 6] = 94.885262249720610030798242337479596095
    P[5, 6] = -143.071126583012024456409244370652716962
    P[6, 6] =  61.0967097444217359754873031115590556707

    P[1, 7] = -1027.321675339240679090464776362465090654
    P[2, 7] = 9198.71432360760879019681406218311101879
    P[3, 7] = -33189.78048157363822223641020734287802492
    P[4, 7] = 57750.0831348887181073584126028277545727
    P[5, 7] = -47698.93315706261990169947144294597707756
    P[6, 7] = 14951.54365344033382142012769129774268946

    P[1, 8] = 1568.546608927281956416687915664731868885
    P[2, 8] = -13995.38852541600542155322174511897930298
    P[3, 8] = 50256.2124698102445419491620666726469821
    P[4, 8] = -86974.5128036219909523950692144595063700
    P[5, 8] = 71494.7977095997701213661747332399327008
    P[6, 8] = -22324.57139433374168317029445568645401598

    P[1, 9] = -2000.882061921041961546811133479107090218
    P[2, 9] = 17864.36380347691630038038755096765127729
    P[3, 9] = -64205.1907515562863000297926577113695108
    P[4, 9] = 111224.8489930378077126420609392735999202
    P[5, 9] = -91509.3392102130338542605593697286718077
    P[6, 9] = 28594.46085938937782634638310955782423389

    P[1, 10] = 1496.620400693446268810344884971434468267
    P[2, 10] = -13397.55405171476021512904990709508924800
    P[3, 10] = 48323.5602199437493999696912750109765015
    P[4, 10] = -84051.4283423393032636942266780744607468
    P[5, 10] = 69399.8582111570893316100585838633124312
    P[6, 10] = -21748.11815446623273761450332307272543593

    P[1, 11] = -16.41320775560933621675902845723196069900
    P[2, 11] = 147.6097045407002371315249807692915435608
    P[3, 11] = -535.719963714732106447158760197417632645
    P[4, 11] = 938.286247077820650371318861625025573381
    P[5, 11] = -779.438309639349328345148153897689081893
    P[6, 11] = 245.4393970278627292916961100938952065362

    P[1, 13] = -4.29672443178246482824254064733546854251
    P[2, 13] = 38.6444746111678092366406218271498656093
    P[3, 13] = -140.3503471762808981414524290552248895548
    P[4, 13] = 246.3954669697502467443139611011701827640
    P[5, 13] = -205.8341686964167118696204191085878165880
    P[6, 13] = 65.44129872356201885836080588282812631205

    P[1, 14] = -20.41628069294821485579834313809132051248
    P[2, 14] = 153.5213232524836445391962375168798263930
    P[3, 14] = -436.5502610211220460266289847121377276100
    P[4, 14] = 598.214644262650861959065070073603792110
    P[5, 14] = -398.7823950071290897160364203878571043995
    P[6, 14] = 104.0129692060648441002024406476025340187

    P[1, 15] = 16.53007184264271512356106095760699278945
    P[2, 15] = -96.6861433615782065041742809436987893361
    P[3, 15] = 268.959934219531723149495873437076657635
    P[4, 15] = -428.681909788964647271837835032326719249
    P[5, 15] = 354.578231152433375494079868740183658991
    P[6, 15] = -114.7001840640649599911246871588418008302

    P[1, 16] = -18.63064171313429626683549958846959067803
    P[2, 16] = 164.1994112280183092456176460821337125030
    P[3, 16] = -579.272256249540441494196462569641132906
    P[4, 16] = 980.198255708866731505258442280896479501
    P[5, 16] = -786.224179015513894176220583239056456901
    P[6, 16] = 239.7294100413035911863764570341369884827

    return (A, B, C, E, E3, P, FSAL, order, error_estimation_order,
            stages, extra_stages,)


@jit(rk_method_signature, **jit_settings)
def Vern9() -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray,
                      np.ndarray, np.ndarray, bool, int, int, int, int]:
    """Return Verner's most efficient 9(8) tableau and configuration parameters.

    Returns
    -------
    A : ndarray, shape (21, 21)
        Runge-Kutta square matrix. The matrix includes the
        extra-coefficients required for dense-outputs.
    B : ndarray, shape (16,)
        Runge-Kutta weights vector. The vector is empty because DOP853 is a FSAL method.
    C : ndarray, shape (21,)
        Runge-Kutta nodes vector. The vector includes the nodes required for dense-outputs.
    E : ndarray, shape (16,)
        Coefficients for the local error estimation.
    E3 : ndarray, shape (1,)
        Not available for Vern9.
    P : ndarray, shape (8, 21)
        Dense-output interpolating 8th order polynomial coefficients.
    FSAL : boolean
        Specifies whether the RK method is FSAL (First Same As Last).
    order : int
        Runge-Kutta order.
    error_estimation_order : int
        Error estimation order.
    extra_stages : int
        Number of additional stages evaluations requested by the dense-output.

    Notes
    -----
    This code implements a 8th order dense-output interpolant, which requires 5 extra
    stages evaluations. An 9th order interpolant is also available, requiring 10 extra
    stages evaluations. The latter has not been implemented here.

    References
    ----------
    .. [1] Jim Verner's Refuge for Runge-Kutta pairs website:
        <https://www.sfu.ca/~jverner/>
    """

    FSAL = False
    stages = 16

    order = 9
    error_estimation_order = 8

    extra_stages = 5

    A = np.zeros((stages+extra_stages,stages+extra_stages))
    A[1, 0] = 0.03462

    A[2, 0] = -0.0389335438857287327017042687229284478532
    A[2, 1] = 0.1359578945245091786499878854939346230295

    A[3, 0] = 0.03638413148954266723060635628912731569111
    A[3, 2] = 0.1091523944686280016918190688673819470733

    A[4, 0] = 2.025763914393969636805657604282571047511
    A[4, 2] = -7.638023836496292020387602153091964592952
    A[4, 3] = 6.173259922102322383581944548809393545442

    A[5, 0] = 0.05112275589406060872792270881648288397197
    A[5, 3] = 0.1770823794555021537929910813839068684087
    A[5, 4] = 0.00080277624092225014536138698108025283759

    A[6, 0] = 0.1316006357975216279279871693164256985334
    A[6, 3] = -0.2957276252669636417685183174672273730699
    A[6, 4] = 0.0878137803564295237421124704053886667082
    A[6, 5] = 0.6213052975225274774321435005639430026100

    A[7, 0] = 0.07166666666666666666666666666666666666667
    A[7, 5] = 0.3305533578915319409260346730051472207728
    A[7, 6] = 0.2427799754418013924072986603281861125606

    A[8, 0] = 0.071806640625
    A[8, 5] = 0.3294380283228177160744825466257672816401
    A[8, 6] = 0.1165190029271822839255174533742327183599
    A[8, 7] = -0.034013671875

    A[9, 0] = 0.04836757646340646986611287718844085773549
    A[9, 5] = 0.03928989925676163974333190042057047002852
    A[9, 6] = 0.1054740945890344608263649267140088017604
    A[9, 7] = -0.02143865284648312665982642293830533996214
    A[9, 8] = -0.1041229174627194437759832813847147895623

    A[10, 0] = -0.02664561487201478635337289243849737340534
    A[10, 5] = 0.03333333333333333333333333333333333333333
    A[10, 6] = -0.1631072244872467239162704487554706387141
    A[10, 7] = 0.03396081684127761199487954930015522928244
    A[10, 8] = 0.1572319413814626097110769806810024118077
    A[10, 9] = 0.2152267478031879552303534778794770376960

    A[11, 0] = 0.03689009248708622334786359863227633989718
    A[11, 5] = -0.1465181576725542928653609891758501156785
    A[11, 6] = 0.2242577768172024345345469822625833796001
    A[11, 7] = 0.02294405717066072637090897902753790803034
    A[11, 8] = -0.0035850052905728761357394424889330334334
    A[11, 9] = 0.08669223316444385506869203619044453906053
    A[11, 10] = 0.4383840651968337846196219974168630120572

    A[12, 0] = -0.4866012215113340846662212357570395295088
    A[12, 5] = -6.304602650282852990657772792012007122988
    A[12, 6] = -0.281245618289472564778284183790118418111
    A[12, 7] = -2.679019236219849057687906597489223155566
    A[12, 8] = 0.518815663924157511565311164615012522024
    A[12, 9] = 1.365353187603341710683633635235238678626
    A[12, 10] = 5.885091088503946585721274891680604830712
    A[12, 11] = 2.802808786272062889819965117517532194812

    A[13, 0] = 0.4185367457753471441471025246471931649633
    A[13, 5] = 6.724547581906459363100870806514855026676
    A[13, 6] = -0.425444280164611790606983409697113064616
    A[13, 7] = 3.343279153001265577811816947557982637749
    A[13, 8] = 0.617081663117537759528421117507709784737
    A[13, 9] = -0.929966123939932833937749523988800852013
    A[13, 10] = -6.099948804751010722472962837945508844846
    A[13, 11] = -3.002206187889399044804158084895173690015
    A[13, 12] = 0.2553202529443445472336424602988558373637

    A[14, 0] = -0.779374086122884664644623040843840506343
    A[14, 5] = -13.93734253810777678786523664804936051203
    A[14, 6] = 1.252048853379357320949735183924200895136
    A[14, 7] = -14.69150040801686878191527989293072091588
    A[14, 8] = -0.494705058533141685655191992136962873577
    A[14, 9] = 2.242974909146236657906984549543692874755
    A[14, 10] = 13.36789380382864375813864978592679139881
    A[14, 11] = 14.39665048665068644512236935340272139005
    A[14, 12] = -0.7975813331776800379127866056663258667437
    A[14, 13] = 0.4409353709534277758753793068298041158235

    A[15, 0] = 2.058051337466886442151242368989994043993
    A[15, 5] = 22.35793772796803295519317565842520212899
    A[15, 6] = 0.90949810997556332745009198137971890783
    A[15, 7] = 35.89110098240264104710550686568482456493
    A[15, 8] = -3.442515027624453437985000403608480262211
    A[15, 9] = -4.865481358036368826566013387928704014496
    A[15, 10] = -18.90980381354342625688427480879773032857
    A[15, 11] = -34.26354448030451782929251177395134170515
    A[15, 12] = 1.264756521695642578827783499806516664686

    A[16, 0] = .1461197685842315252051541915018784713459e-1
    A[16, 7] = -.3915211862331339089410228267288242030810
    A[16, 8] = .2310932500289506415909675644868993669908
    A[16, 9] = .1274766769992852382560589467488989175618
    A[16, 10] = .2246434176204157731566981937082069688984
    A[16, 11] = .5684352689748512932705226972873692126743
    A[16, 12] = .5825871557215827200814768021863420902155e-1
    A[16, 13] = .1364317403482215641609022744494239843327
    A[16, 14] = .3057013983082797397721005067920369646664e-1

    A[17, 0] = .1549973668189559302279946863304789372788e-1
    A[17, 7] = .3355153219059635054403439303177105512242
    A[17, 8] = .2003613944191860651552622660712101217322
    A[17, 9] = .1252060659283549312946162355194540994211
    A[17, 10] = .2298676393184206750544046308957155868736
    A[17, 11] = -.2020250653476181447824906889122391003637
    A[17, 12] = .5917103230665456601422111997583025339897e-1
    A[17, 13] = -.2651834783047638681693835956996437528251e-1
    A[17, 14] = -.2384094602130971415278110567256446033405e-1
    A[17, 16] = .2718171570208501807097257892166705118335e-1

    A[18, 0] = .1302453943114338366054520296881099431474e-1
    A[18, 7] = -.7452850902413112085299330666038981625179
    A[18, 8] = .2643867896429300961465132150322749722129
    A[18, 9] = .1313961758372753932588328082078842388890
    A[18, 10] = .2167253815122927263092467187957410643315
    A[18, 11] = .8734117564076052559016338094938888451419
    A[18, 12] = .1185905643935776688228545787724340848142e-1
    A[18, 13] = .5876002941689550612992712203494447529933e-1
    A[18, 14] = .3266518630202087866399279690939423159022e-2
    A[18, 16] = -.8959308648417929824525368306101792182274e-2
    A[18, 17] = .6941415157202692219907482080827253287034e-1

    A[19, 0] = .1397089996925942721283716334050740168797e-1
    A[19, 7] = -.4665765335957674596054673402956853940520
    A[19, 8] = .2416372787216257077935214889875485248580
    A[19, 9] = .1290363341345674735721677437066933999929
    A[19, 10] = .2216700671735105311080225734522323922813
    A[19, 11] = .6257275123364644931771253383573999863003
    A[19, 12] = .4355312415679284117869124964829805160429e-1
    A[19, 13] = .1011962491667290833450024852274278874501
    A[19, 14] = .1808582254679721049279369742685497400353e-1
    A[19, 16] = -.2079875587689169691156509689282083267654e-1
    A[19, 17] = -.9022232517086218976198252891464664868640e-1
    A[19, 18] = -.1212796735622254216011467740438097427634

    A[20, 0] = .1604638888318112738641232352800290501904e-1
    A[20, 7] = .9517712399458336651642257453589397190702e-1
    A[20, 8] = .1359187264655317806136927180199100622471
    A[20, 9] = .1237765280959854006935081364365637515893
    A[20, 10] = .2335656264102966047058755123098072346246
    A[20, 11] = -.9051508172625873314662090873741762206189e-1
    A[20, 12] = -.2537574270006131028513276914038326155331e-1
    A[20, 13] = -.1359631696887162048002744757083947500478
    A[20, 14] = -.4679214284145113075088049469061349990847e-1
    A[20, 16] = .5177958859391748239949773879090325427473e-1
    A[20, 17] = .9672595677476773313884172931875718705561e-1
    A[20, 18] = .1477312690340742769720989417101989769314
    A[20, 19] = -.1150750712958503934434410263732282100773

    C = np.zeros((stages+extra_stages,))
    C[1] = 0.03426
    C[2] = 0.09702435063878044594828361677100617517633
    C[3] = 0.1455365259581706689224254251565092627645
    C[4] = 0.561
    C[5] = 0.2290079115904850126662751771814700052182
    C[6] = 0.5449920884095149873337248228185299947818
    C[7] = 0.645
    C[8] = 0.48375
    C[9] = 0.06757
    C[10] = 0.25
    C[11] = 0.6590650618730998549405331618649220295334
    C[12] = 0.8206
    C[13] = 0.9012
    C[14] = 1.
    C[15] = 1.
    C[16] = 1.
    C[17] = 0.7404185470631561083004100761798676215811
    C[18] = 0.888
    C[19] = 0.696
    C[20] = 0.487

    B = np.zeros((stages,))
    B[0] = 0.01461197685842315252051541915018784713459
    B[7] = -0.3915211862331339089410228267288242030810
    B[8] = 0.2310932500289506415909675644868993669908
    B[9] = 0.1274766769992852382560589467488989175618
    B[10] = 0.2246434176204157731566981937082069688984
    B[11] = 0.5684352689748512932705226972873692126743
    B[12] = 0.05825871557215827200814768021863420902155
    B[13] = 0.1364317403482215641609022744494239843327
    B[14] = 0.03057013983082797397721005067920369646664

    E = np.zeros((stages,))
    E[0] = -0.005357988290444578
    E[7] = -2.583020491182464
    E[8] = 0.14252253154686625
    E[9] = 0.013420653512688676
    E[10] = -0.02867296291409493
    E[11] = 2.624999655215792
    E[12] = -0.2825509643291537
    E[13] = 0.1364317403482215641609022744494239843327
    E[14] = 0.03057013983082797397721005067920369646664
    E[15] = -0.04834231373823958

    E3 = np.zeros((1,))

    P = np.zeros((8, stages + extra_stages))

    P[0, 0] = 1.
    P[1, 0] = -12.74966541771576102412726774271024205003
    P[2, 0] = 68.53080766672322360449931519438662010864
    P[3, 0] = -194.8119745354184115708662255742713867869
    P[4, 0] = 317.8426371352858465721400485549409045261
    P[5, 0] = -299.7155409593396451474764978718840789438
    P[6, 0] = 152.1119186420412302044289062135608459907
    P[7, 0] = -32.19357055471805948607776335487247499763

    P[1, 7] = 141.0696092533712605710078918334708622849
    P[2, 7] = -1283.768450646593611469556198945305487267
    P[3, 7] = 4630.280061766681019456906394425884505766
    P[4, 7] = -8648.500976100316056391914391739346002386
    P[5, 7] = 8890.812161067019054072477301499300549282
    P[6, 7] = -4787.949212676938594146631544372229445471
    P[7, 7] = 1057.665286150543793998769524471496193587

    P[1, 8] = -51.75101323451538200013913368910238945675
    P[2, 8] = 486.0412507312931363219502660627455126442
    P[3, 8] = -1777.475586368523869187924524440189207548
    P[4, 8] = 3345.498238649791484065923838383080369426
    P[5, 8] = -3455.762480007366110892744664331997981283
    P[6, 8] = 1867.081161290311461863227123612293652933
    P[7, 8] = -413.4004778109617695287019380323430573493

    P[1, 9] = 16.32082008695896537661978157425676142665
    P[2, 9] = -118.7072740966746629137122793258036781419
    P[3, 9] = 379.8980653294655976666790006724113822620
    P[4, 9] = -659.1980179681815547557635890342282178072
    P[5, 9] = 645.0127094968870314571906430728856355521
    P[6, 9] = -335.3923630294779850989153429752631332406
    P[7, 9] = 72.19353685802189350615784496249014886653

    P[1, 10] = -5.897787927512737700004074981094546266733
    P[2, 10] = 89.61427156602281202549839530979318529349
    P[3, 10] = -381.3887877052766946040448187450031021400
    P[4, 10] = 773.0964199867749581865429091250468401092
    P[5, 10] = -834.1212536283262120951621382995179788432
    P[6, 10] = 463.6209151933600203046608830911748528029
    P[7, 10] = -104.6991340674217303443344573066910439868

    P[1, 11] = -211.5753578293942678386576895806496541353
    P[2, 11] = 1922.149747207956657441031349510100069933
    P[3, 11] = -6927.544647063403206822988172864704814066
    P[4, 11] = 12933.89131149183768334045388734576967421
    P[5, 11] = -13292.72252236109238830196845558874371599
    P[6, 11] = 7157.200591588665684047481512110694194478
    P[7, 11] = -1580.830687765595310572081908235178385210

    P[1, 12] = -29.64315154973386448483853503830022916124
    P[2, 12] = 265.6161452668802722379486185155415065660
    P[3, 12] = -951.3146379152784513383019821793424939005
    P[4, 12] = 1769.876627954108227401999032086988914455
    P[5, 12] = -1814.926158732241586204906382926227921564
    P[6, 12] = 975.7252379518513874523768974027019911763
    P[7, 12] = -215.2758042600138267922695001811431333620

    P[1, 13] = -78.71890822220322199073537821460790272065
    P[2, 13] = 702.2030963698086240509004430267820179684
    P[3, 13] = -2509.783618527791392508610051909561977746
    P[4, 13] = 4663.884687403353676982783161174879377253
    P[5, 13] = -4779.049763533834849959594190430001691071
    P[6, 13] = 2567.969360375736223208476377322945879725
    P[7, 13] = -566.3684221247208382190594586959862794242

    P[1, 14] = -20.19281566680438095870219832836548040387
    P[2, 14] = 179.3637257949118586258743524266532193814
    P[3, 14] = -639.8126260732690700781771881696950701907
    P[4, 14] = 1187.623035597440146280793419574292438396
    P[5, 14] = -1216.083146478910882267036576696259876826
    P[6, 14] = 653.1305166034912224190607555907831343139
    P[7, 14] = -143.9981196370280660478353543467291609746

    P[1, 16] = 22.25529172937004228561680254541914022993
    P[2, 16] = -202.2768882656563899475956661732714767089
    P[3, 16] = 741.4504035959390672891852702617386096470
    P[4, 16] = -1420.171375819646417700236265030810075574
    P[5, 16] = 1506.826759283909640808006726351811877831
    P[6, 16] = -842.0883145405746016477682230643054253648
    P[7, 16] = 194.0041240166586589127913551094173499391

    P[1, 17] = 14.87763987360506280784259993181818867611
    P[2, 17] = -312.2840512785346745116969727542541022617
    P[3, 17] = 1842.506617216153294661664671292748129746
    P[4, 17] = -4868.899648863078786274115894868122667808
    P[5, 17] = 6508.346688203267682481628418765155233374
    P[6, 17] = -4307.866481530869189076138573916573603059
    P[7, 17] = 1123.319236379456609910815751549228821331

    P[1, 18] = 94.98456650525377646383263451778568838647
    P[2, 18] = -820.5775654295434684704012573738365180124
    P[3, 18] = 2818.347256898649110692476937532304801453
    P[4, 18] = -4992.649725157522569698401937223054737557
    P[5, 18] = 4835.804647435804986125480300928971220935
    P[6, 18] = -2434.068718877444022356652060555528975267
    P[7, 18] = 498.1595386248021872436653821733585200624

    P[1, 19] = 63.46082596146032232234944838983857033697
    P[2, 19] = -387.6244695782448432215454472613264482576
    P[3, 19] = 652.6008727039380999132826268449332985693
    P[4, 19] = 222.0020976000799700370477743563371275858
    P[5, 19] = -1696.555363345186459306426642315756771818
    P[6, 19] = 1674.058335196842891022210000150369785868
    P[7, 19] = -527.9422985388899807669177601643955622843

    P[1, 20] = 57.55994643786018616993511878224123285359
    P[2, 20] = -588.2803453083489337731949182122044212454
    P[3, 20] = 2317.048600678134906430718062852747324934
    P[4, 20] = -4624.295311909926608047251992705773944827
    P[5, 20] = 5002.133263559409739230532157842265499369
    P[6, 20] = -2803.532946186995728195816710610623754886
    P[7, 20] = 639.3667927298664381850782820513480638022

    return (A, B, C, E, E3, P, FSAL, order, error_estimation_order,
            stages, extra_stages,)

