from pathlib import Path

import yaml

from . import exceptions


class Config:
    FILENAME = 'serverless.yaml'
    VERSION = 'V4'

    def __init__(self):
        self.version = None
        self.base_dir = None
        self.cloud_type = None
        self.cloud_project = None
        self.region = None
        self.endpoint_api_title = None
        self.endpoint_api_description = None
        self.endpoint_deploy_version = 0

    def load(self) -> bool:
        if not Path(Config.FILENAME).exists():
            return False
        with open(Config.FILENAME) as f:
            data = yaml.safe_load(f)
            self.version = data.get('version', '')
            self.base_dir = data.get('base_dir', '.')
            self.cloud = data.get('cloud')
            self.project = data.get('project')
            self.region = data.get('region')
            self.endpoint_api_title = data.get('endpoints', {}).get('api_title')
            self.endpoint_api_description = data.get('endpoints', {}).get('api_description')
            self.endpoint_deploy_version = int(data.get('endpoints', {}).get('deploy_version', 0))

        if str(self.version).strip() != self.VERSION:
            raise exceptions.ConfigInvalidVersion()

        return True

    def save(self):
        with open(Config.FILENAME, 'w') as f:
            yaml.dump(self.dict(), f)

    def dict(self):
        return {
            'version': self.VERSION,
            'base_dir': self.base_dir or '.',
            'cloud': self.cloud,
            'project': self.project,
            'region': self.region,
            'endpoints': {
                'api_title': self.endpoint_api_title,
                'api_description': self.endpoint_api_description,
                'deploy_version': self.endpoint_deploy_version,
            },
        }
