import re
import os
import ignorem
from ignorem import sources

GITIGNORE_PATH = os.path.join(os.getcwd(), ".gitignore")


class GitIgnore:
	_HEADER_BEGIN = "# BEGIN ignorem auto-gen"
	_HEADER_END = "# END ignorem auto-gen"
	_HEADER_INSTALLED = "# ignorem installed gitignores: "

	def __init__(self, text=""):
		self.__text = text
		self.__ignores = ""
		self.__loose_ignore = ""
		self.__installed = {}

		self.parse(text)

	def parse(self, text):
		# Get the list of ignorem installed gitignores
		installed = re.search(rf"(?<={GitIgnore._HEADER_INSTALLED})([A-Za-z0-1_ ,+-]+)", text)
		if installed:
			installed = re.split(r"[ ,]+", installed.group(0))
		else:
			installed = {}

		for i in installed:
			self.__installed[i.lower()] = ""

		# Isolate loose ignores
		auto_gen_ignores = re.findall(rf"{GitIgnore._HEADER_BEGIN}[\s\S]*?{GitIgnore._HEADER_END}", text)
		for i in auto_gen_ignores:
			text = text.replace(i, "")
		text = re.sub(r"(?<=(\r|\n){2})(\r|\n)", "", text)
		self.__loose_ignore = text

	def add(self, gitignore):
		self.__installed[gitignore] = ""

	def remove(self, gitignore):
		if gitignore in self.__installed:
			del self.__installed[gitignore]
			return True
		else:
			return False

	def get_installed(self):
		return list(self.__installed.keys())

	def is_has_loose_gitignore(self):
		return len(self.__loose_ignore) > 0

	def get_complete_gitignore(self, verbose=True):
		ignore_text = ""

		for i in self.__installed:
			ignore_text += "# BEGIN " + i + "\n"
			ignore_text += sources.fetch_ignore(i) + "\n"

		ignore_text = ignore_text.rstrip("\n")

		text = f"""{GitIgnore._HEADER_BEGIN}
# This gitingore was generated by {ignorem.PROGRAM_NAME} version {ignorem.PROGRAM_VERSION}
# {ignorem.PROGRAM_URL}
# DO NOT EDIT ANY TEXT INSIDE THE {ignorem.PROGRAM_NAME} auto-gen tags!
# It will be overwritten next time ignorem is ran.
# Following this header are the loose ignores, or ignores added externally.
# After that, there are the ignores generated by {ignorem.PROGRAM_NAME}.
{GitIgnore._HEADER_END}

{self.__loose_ignore.lstrip().rstrip()}

{GitIgnore._HEADER_BEGIN}
# Beginning of {ignorem.PROGRAM_NAME} generated ignores.
{GitIgnore._HEADER_INSTALLED + ", ".join(self.__installed)}

{ignore_text.lstrip().rstrip()}

{GitIgnore._HEADER_END}
"""
		return text

	def __str__(self):
		return self.get_complete_gitignore(False)


class GitIgnoreManager:
	@staticmethod
	def read(file_path=GITIGNORE_PATH):
		# Create gitignore if it doesn't exist
		if not os.path.exists(file_path):
			open(file_path, "w").close()

		# Read gitignore
		gitignore = open(file_path, "r")
		data = gitignore.read()
		gitignore.close()
		return GitIgnore(data)

	@staticmethod
	def write(gitignore: GitIgnore, file_path=GITIGNORE_PATH):
		if not isinstance(gitignore, GitIgnore):
			raise TypeError("GitIgnoreManager.write requires first parameter to be type GitIgnore!")

		file = open(file_path, "w")
		file.write(gitignore.get_complete_gitignore())
		file.close()
