"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationAccount = exports.OrganizationAccountProvider = exports.OrganizationOU = exports.OrganizationOUProvider = exports.processOUObj = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
;
;
/**
 * @function processOUObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} ouProvider The custom resource provider for managing OUs
 * @param {custom_resources.Provider} accountProvider The custom resource provider for managing accounts
 * @param {OUObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
function processOUObj(ouProvider, accountProvider, obj, parent) {
    var _e, _f;
    const parentStr = parent instanceof OrganizationOU ? parent.resource.ref : parent;
    let id = (_e = obj.id) !== null && _e !== void 0 ? _e : obj.properties.name;
    const ou = new OrganizationOU(this, id, {
        provider: ouProvider,
        parent: parentStr,
        name: obj.properties.name,
        importOnDuplicate: obj.properties.importOnDuplicate,
        allowRecreateOnUpdate: obj.properties.allowRecreateOnUpdate,
    });
    (_f = obj.accounts) === null || _f === void 0 ? void 0 : _f.forEach((account) => {
        new OrganizationAccount(this, `${account.name}-${account.email.replace(/[^a-zA-Z ]/g, '')}`, {
            provider: accountProvider,
            parent: ou,
            name: account.name,
            email: account.email,
            importOnDuplicate: account.importOnDuplicate,
            allowMove: account.allowMove,
            disableDelete: account.disableDelete,
        });
    });
    obj.children.forEach(child => {
        processOUObj.call(this, ouProvider, accountProvider, child, ou);
    });
}
exports.processOUObj = processOUObj;
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
class OrganizationOUProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:UpdateOrganizationalUnit',
                            'organizations:DeleteOrganizationalUnit',
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:CreateOrganizationalUnit',
                            'organizations:DescribeAccount',
                            'organizations:DescribeCreateAccountStatus',
                            'organizations:CreateAccount',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/ou'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(10),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationOUProvider = OrganizationOUProvider;
_a = JSII_RTTI_SYMBOL_1;
OrganizationOUProvider[_a] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOUProvider", version: "0.4.2" };
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
class OrganizationOU extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id);
        const importOnDuplicate = (_e = props.importOnDuplicate) !== null && _e !== void 0 ? _e : false;
        const allowRecreateOnUpdate = (_f = props.allowRecreateOnUpdate) !== null && _f !== void 0 ? _f : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowRecreateOnUpdate: allowRecreateOnUpdate,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationOU = OrganizationOU;
_b = JSII_RTTI_SYMBOL_1;
OrganizationOU[_b] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOU", version: "0.4.2" };
/**
 * The provider for account custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating accounts.
 *
 * @stability stable
 */
class OrganizationAccountProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:ListAccountsForParent',
                            'organizations:ListRoots',
                            'organizations:MoveAccount',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/account'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(300),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationAccountProvider = OrganizationAccountProvider;
_c = JSII_RTTI_SYMBOL_1;
OrganizationAccountProvider[_c] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationAccountProvider", version: "0.4.2" };
/**
 * The construct to create or update an Organization account.
 *
 * This relies on the custom resource provider OrganizationAccountProvider.
 *
 * @stability stable
 */
class OrganizationAccount extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f, _g;
        super(scope, id);
        const importOnDuplicate = (_e = props.importOnDuplicate) !== null && _e !== void 0 ? _e : false;
        const allowMove = (_f = props.allowMove) !== null && _f !== void 0 ? _f : false;
        const disableDelete = (_g = props.disableDelete) !== null && _g !== void 0 ? _g : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowMove: allowMove,
                DisableDelete: disableDelete,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationAccount = OrganizationAccount;
_d = JSII_RTTI_SYMBOL_1;
OrganizationAccount[_d] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationAccount", version: "0.4.2" };
//# sourceMappingURL=data:application/json;base64,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