"""Main module describes code generation for Swagger OpenAPI specs"""
import keyword
import os

import inflection

from dekogen.base_codegen import CodeGenerator
from dekogen.file_writer import NEW_LINE, FileWriter

BASE_CLASS = 'BaseBodyBuilder'


class PythonSpecGenerator(CodeGenerator):
    """Main utility class for .py code generation"""

    def __init__(self, spec_data, file_path=''):
        """Initialize dekogen properties for future assign"""
        self.file_writer = None
        self.file_path = file_path
        self.spec_data = spec_data
        self.classes = []

    def create_file_with_header(self, is_requests=False):
        """
        Create or override .py file. Set file headers.
         Note: Change value of class member 'file_writer'
        """
        module_doc = '"""Autogenerated module describes {} definitions via python code"""{new_line}'
        module_warning = '# WARNING: AUTOGENERATED CODE. DO NOT EDIT.{new_line}'

        if is_requests:
            file_name = '{}_requests.py'.format(self.spec_data.get("info").get("title")).replace(' ', '_').lower()
            methods = list()
            methods.append(self.gen_method(method_name='__init__', method_body=["self.body = {}", NEW_LINE]))
            methods.append(self.gen_method(method_name='clear', method_body=["self.body = {}", "return self"]))
            self.classes.append(self.gen_class(class_name=BASE_CLASS, inner_methods=methods))
        else:
            file_name = '{}_responses.py'.format(self.spec_data.get("info").get("title")).replace(' ', '_').lower()
        file_path = os.path.join(self.file_path, file_name)
        file_writer = FileWriter(file_path)
        file_writer.file.write(module_doc.format(self.spec_data.get("info").get("title"), new_line=NEW_LINE))
        file_writer.add_empty_lines(count=2)
        file_writer.file.write(module_warning.format(new_line=NEW_LINE))
        file_writer.add_empty_lines(2)
        self.file_writer = file_writer

    def write_result_code(self):
        """
        Write result code from 'self.classes' using 'self.file_writer'
        """
        for item in self.classes:
            self.file_writer.file.write(item)
            empty_line_count = 0 if item == self.classes[-1] else 2
            self.file_writer.add_empty_lines(empty_line_count)


class PythonResponsesGenerator(PythonSpecGenerator):
    """Responses specific methods"""

    def generate_responses_by_yaml_data(self):
        """
        Generate python file content according to YAML
         Note: Change value of class member 'responses_classes'
        """
        for def_name, def_value in self.spec_data.get("definitions").items():
            properties = def_value.get("properties")
            method_body = list()
            methods = list()
            method_body.append("self.orig_value = dict_structure")
            if properties:
                self.add_code_line_response(definition_value=def_value, method_body=method_body)
            methods.append(self.gen_method(method_name='__init__', method_body=method_body,
                                           parameters=['dict_structure']))
            self.classes.append(self.gen_class(class_name=def_name, inner_methods=methods))

    @staticmethod
    def add_code_line_response(definition_value, method_body):
        """
        Utility function to set value according to property description
        :param definition_value: Property definition
        :param method_body: List of lines of the body content
         Doesn't return anything but update existing method_body list
        """
        class_init = "self.{0} = {1}(dict_structure.get(\"{2}\", dict()))"
        class_init_comprehension = "self.{0} = [{1}(item) for item in dict_structure.get(\"{2}\", dict())]"
        list_init = "self.{0} = [item for item in dict_structure.get(\"{1}\")]"
        value_init = "self.{0} = dict_structure.get(\"{1}\")"

        for prop_name, prop_value in definition_value["properties"].items():
            python_style_prop_name = inflection.underscore(prop_name)
            value_type = prop_value.get("type")
            reference = prop_value.get('$ref')
            items = prop_value.get("items")
            if reference:
                method_body.append(class_init.format(python_style_prop_name, reference.split("/")[-1], prop_name))
            elif items:
                items_type = items.get("type")
                items_reference = items.get('$ref')
                if items_reference:
                    method_body.append(class_init_comprehension.format(python_style_prop_name,
                                                                       items_reference.split("/")[-1], prop_name))
                elif items_type:
                    method_body.append(list_init.format(python_style_prop_name, prop_name))
            elif value_type:
                method_body.append(value_init.format(python_style_prop_name, prop_name))

    def generate(self):
        """Generate *_responses.py file"""
        self.create_file_with_header()
        self.generate_responses_by_yaml_data()
        self.write_result_code()


class PythonRequestsGenerator(PythonSpecGenerator):
    """Requests specific methods"""

    def add_request_method(self, definition_value, definition_name, methods):
        """
        Create methods and store in passed list
        :param definition_value: Property definition
        :param definition_name: Property (class) name
        :param methods:  List of methods'
         Doesn't return anything but update existing methods list
        """
        scalar_value = 'self.body["{0}"] = {1}{new_line}return self'
        iterable_value = 'self.body["{0}"] = list({1}){new_line}return self'

        for prop_name, prop_value in definition_value["properties"].items():
            if prop_name in keyword.kwlist or prop_name in __builtins__.keys():
                python_style_prop_name = '{0}_{1}'.format(inflection.underscore(definition_name),
                                                          inflection.underscore(prop_name))
            else:
                python_style_prop_name = inflection.underscore(prop_name)

            if prop_value.get("items"):
                method_body = iterable_value.format(prop_name, python_style_prop_name, new_line=NEW_LINE)
                parameter_template = '*{0}'
            else:
                method_body = scalar_value.format(prop_name, python_style_prop_name, new_line=NEW_LINE)
                parameter_template = '{0}'
            methods.append(self.gen_method(method_name=python_style_prop_name, method_body=method_body.split(NEW_LINE),
                                           parameters=[parameter_template.format(python_style_prop_name)]))

    def create_enum_class(self, class_name, definition):
        """
        Create class containing string constants
        :param class_name: Name of the class
        :param definition: Property definition
        :return: Class string
        """
        enum_values = definition.get("enum")
        inner_properties = ['{0} = "{1}"'.format(value.replace(' ', '_').replace('-', '_'), value) for value in
                            enum_values]
        return self.gen_class(class_name=class_name, inner_methods='', inner_properties=inner_properties)

    def generate_requests_by_yaml_data(self):
        """
        Generate python file content according to YAML
         Note: Change value of class member 'requests_classes'
        """
        for def_name, def_value in self.spec_data.get("definitions").items():
            methods = list()
            if def_value.get("properties"):
                self.add_request_method(definition_value=def_value, definition_name=def_name, methods=methods)
                self.classes.append(self.gen_class(class_name=def_name, inner_methods=methods, base_class=BASE_CLASS))
            elif def_value.get("enum"):
                self.classes.append(self.create_enum_class(class_name=def_name, definition=def_value))

    def generate(self):
        """Generate *_requests.py file"""
        self.create_file_with_header(is_requests=True)
        self.generate_requests_by_yaml_data()
        self.write_result_code()
