# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vocode',
 'vocode.streaming',
 'vocode.streaming.agent',
 'vocode.streaming.input_device',
 'vocode.streaming.models',
 'vocode.streaming.output_device',
 'vocode.streaming.synthesizer',
 'vocode.streaming.telephony',
 'vocode.streaming.telephony.config_manager',
 'vocode.streaming.telephony.conversation',
 'vocode.streaming.telephony.hosted',
 'vocode.streaming.telephony.server',
 'vocode.streaming.telephony.server.router',
 'vocode.streaming.transcriber',
 'vocode.streaming.user_implemented_agent',
 'vocode.streaming.utils',
 'vocode.turn_based',
 'vocode.turn_based.agent',
 'vocode.turn_based.input_device',
 'vocode.turn_based.output_device',
 'vocode.turn_based.synthesizer',
 'vocode.turn_based.transcriber']

package_data = \
{'': ['*'],
 'vocode.streaming.synthesizer': ['filler_audio/*'],
 'vocode.streaming.telephony': ['templates/*'],
 'vocode.streaming.utils': ['goodbye_embeddings/.gitkeep']}

install_requires = \
['aiohttp==3.8.4',
 'aiosignal==1.3.1',
 'anyio==3.6.2',
 'async-timeout==4.0.2',
 'attrs==22.2.0',
 'azure-cognitiveservices-speech==1.25.0',
 'cachetools==5.3.0',
 'certifi==2022.12.7',
 'cffi==1.15.1',
 'charset-normalizer==3.0.1',
 'click==8.1.3',
 'dataclasses-json==0.5.7',
 'decorator==5.1.1',
 'fastapi==0.92.0',
 'frozenlist==1.3.3',
 'google-api-core==2.11.0',
 'google-auth==2.16.3',
 'google-cloud-speech==2.17.3',
 'google-cloud-texttospeech==2.14.1',
 'googleapis-common-protos==1.59.0',
 'grpcio-status==1.51.3',
 'grpcio==1.51.3',
 'h11==0.14.0',
 'idna==3.4',
 'jinja2==3.1.2',
 'joblib==1.2.0',
 'langchain==0.0.117',
 'markupsafe==2.1.2',
 'marshmallow-enum==1.5.1',
 'marshmallow==3.19.0',
 'mccabe==0.7.0',
 'multidict==6.0.4',
 'mypy-extensions==1.0.0',
 'nltk==3.8.1',
 'numpy==1.24.2',
 'openai==0.27.2',
 'packaging==23.0',
 'pathspec==0.11.0',
 'platformdirs==3.1.0',
 'ply==3.11',
 'proto-plus==1.22.2',
 'protobuf==4.22.1',
 'pyasn1-modules==0.2.8',
 'pyasn1==0.4.8',
 'pycodestyle==2.10.0',
 'pycparser==2.21',
 'pydantic>=1.9.0',
 'pyflakes>=2.5.0',
 'pyjwt==2.6.0',
 'python-dotenv==0.21.1',
 'python-multipart==0.0.6',
 'pytz==2022.7.1',
 'pyyaml==6.0',
 'redis==4.5.3',
 'regex==2023.3.23',
 'requests==2.28.2',
 'rsa==4.9',
 'six==1.16.0',
 'sniffio==1.3.0',
 'sqlalchemy==1.4.47',
 'starlette==0.25.0',
 'tenacity==8.2.2',
 'tomli==2.0.1',
 'tqdm==4.65.0',
 'twilio==7.17.0',
 'typing-extensions>=3.10.0.2',
 'typing-inspect==0.8.0',
 'urllib3==1.26.14',
 'uvicorn==0.20.0',
 'websockets==10.4',
 'yarl==1.8.2']

extras_require = \
{'io': ['sounddevice==0.4.6', 'pyaudio==0.2.13', 'pydub==0.25.1']}

setup_kwargs = {
    'name': 'vocode',
    'version': '0.1.67',
    'description': 'The all-in-one voice SDK',
    'long_description': '<div align="center">\n\n![Hero](https://user-images.githubusercontent.com/6234599/228337850-e32bb01d-3701-47ef-a433-3221c9e0e56e.png)\n\n    \n[![Twitter](https://img.shields.io/twitter/url/https/twitter.com/vocodehq.svg?style=social&label=Follow%20%40vocodehq)](https://twitter.com/vocodehq) [![GitHub Repo stars](https://img.shields.io/github/stars/vocodedev/vocode-python?style=social)](https://github.com/vocodedev/vocode-python)\n\n[Community](https://discord.gg/NaU4mMgcnC) | [Docs](https://docs.vocode.dev) | [Dashboard](https://app.vocode.dev)\n</div>\n\n# <span><img style=\'vertical-align:middle; display:inline;\' src="https://user-images.githubusercontent.com/6234599/228339858-95a0873a-2d40-4542-963a-6358d19086f5.svg"  width="5%" height="5%">&nbsp; vocode</span>\n\n### **Build voice-based LLM apps in minutes**\n\nVocode is an open source library that makes it easy to build voice-based LLM apps. Using Vocode, you can build real-time streaming conversations with LLMs and deploy them to phone calls, Zoom meetings, and more. You can also build personal assistants or apps like voice-based chess. Vocode provides easy abstractions and integrations so that everything you need is in a single library.\n\n# ⭐️ Features\n- 🗣 [Spin up a conversation with your system audio](https://docs.vocode.dev/python-quickstart)\n- ➡️ 📞 [Set up a phone number that responds with a LLM-based agent](https://docs.vocode.dev/telephony#inbound-calls)\n- 📞 ➡️ [Send out phone calls from your phone number managed by an LLM-based agent](https://docs.vocode.dev/telephony#outbound-calls)\n- 🧑\u200d💻 [Dial into a Zoom call](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/telephony/hosted/zoom_dial_in.py)\n- Out of the box integrations with:\n  - Transcription services, including:\n    - [Deepgram](https://deepgram.com/)\n    - [AssemblyAI](https://www.assemblyai.com/)\n    - [Google Cloud](https://cloud.google.com/speech-to-text)\n    - [Whisper](https://openai.com/blog/introducing-chatgpt-and-whisper-apis)\n  - LLMs, including:\n    - [ChatGPT](https://openai.com/blog/chatgpt)\n    - [GPT-4](https://platform.openai.com/docs/models/gpt-4)\n    - [Anthropic](https://www.anthropic.com/) - coming soon!\n  - Synthesis services, including:\n    - [Microsoft Azure](https://azure.microsoft.com/en-us/products/cognitive-services/text-to-speech/)\n    - [Google Cloud](https://cloud.google.com/text-to-speech)\n    - [Eleven Labs](https://elevenlabs.io/) \n\nCheck out our React SDK [here](https://github.com/vocodedev/vocode-react-sdk)! \n\n# 🫂 Contribution\n\nWe\'d love for you all to build on top of our abstractions to enable new and better LLM voice applications!\n\nYou can extend our [`BaseAgent`](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/agent/base_agent.py), [`BaseTranscriber`](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/transcriber/base_transcriber.py), and [`BaseSynthesizer`](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/synthesizer/base_synthesizer.py) abstractions to integrate with new LLM APIs, speech recognition and speech synthesis providers. More detail [here](https://docs.vocode.dev/create-your-own-agent#self-hosted).\n\nYou can also work with our [`BaseInputDevice`](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/input_device/base_input_device.py) and [`BaseOutputDevice`](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/output_device/base_output_device.py) abstractions to set up voice applications on new surfaces/platforms. More guides for this coming soon!\n\nBecause our [`StreamingConversation`](https://github.com/vocodedev/vocode-python/blob/main/vocode/streaming/streaming_conversation.py) runs locally, it\'s relatively quick to develop on! Feel free to fork and create a PR and we will get it merged as soon as possible. And we\'d love to talk to you on [Discord](https://discord.gg/NaU4mMgcnC)!\n\n# 🚀 Quickstart (Self-hosted)\n\n```bash\npip install \'vocode[io]\'\n```\n\n```python\nimport asyncio\nimport signal\n\nimport vocode\nfrom vocode.streaming.streaming_conversation import StreamingConversation\nfrom vocode.helpers import create_microphone_input_and_speaker_output\nfrom vocode.streaming.models.transcriber import (\n    DeepgramTranscriberConfig,\n    PunctuationEndpointingConfig,\n)\nfrom vocode.streaming.agent.chat_gpt_agent import ChatGPTAgent\nfrom vocode.streaming.models.agent import ChatGPTAgentConfig\nfrom vocode.streaming.models.message import BaseMessage\nfrom vocode.streaming.models.synthesizer import AzureSynthesizerConfig\nfrom vocode.streaming.synthesizer.azure_synthesizer import AzureSynthesizer\nfrom vocode.streaming.transcriber.deepgram_transcriber import DeepgramTranscriber\n\n# these can also be set as environment variables\nvocode.setenv(\n    OPENAI_API_KEY="<your OpenAI key>",\n    DEEPGRAM_API_KEY="<your Deepgram key>",\n    AZURE_SPEECH_KEY="<your Azure key>",\n    AZURE_SPEECH_REGION="<your Azure region>",\n)\n\n\nasync def main():\n    microphone_input, speaker_output = create_microphone_input_and_speaker_output(\n        streaming=True, use_default_devices=False\n    )\n\n    conversation = StreamingConversation(\n        output_device=speaker_output,\n        transcriber=DeepgramTranscriber(\n            DeepgramTranscriberConfig.from_input_device(\n                microphone_input, endpointing_config=PunctuationEndpointingConfig()\n            )\n        ),\n        agent=ChatGPTAgent(\n            ChatGPTAgentConfig(\n                initial_message=BaseMessage(text="Hello!"),\n                prompt_preamble="Have a pleasant conversation about life",\n            ),\n        ),\n        synthesizer=AzureSynthesizer(\n            AzureSynthesizerConfig.from_output_device(speaker_output)\n        ),\n    )\n    await conversation.start()\n    print("Conversation started, press Ctrl+C to end")\n    signal.signal(signal.SIGINT, lambda _0, _1: conversation.terminate())\n    while conversation.is_active():\n        chunk = microphone_input.get_audio()\n        if chunk:\n            conversation.receive_audio(chunk)\n        await asyncio.sleep(0)\n\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\n# ☁️ Quickstart (Hosted)\n\nFirst, get a *free* API key from our [dashboard](https://app.vocode.dev).\n\n```bash\npip install \'vocode[io]\'\n```\n\n```python\nimport asyncio\nimport signal\n\nimport vocode\nfrom vocode.streaming.hosted_streaming_conversation import HostedStreamingConversation\nfrom vocode.streaming.streaming_conversation import StreamingConversation\nfrom vocode.helpers import create_microphone_input_and_speaker_output\nfrom vocode.streaming.models.transcriber import (\n    DeepgramTranscriberConfig,\n    PunctuationEndpointingConfig,\n)\nfrom vocode.streaming.models.agent import ChatGPTAgentConfig\nfrom vocode.streaming.models.message import BaseMessage\nfrom vocode.streaming.models.synthesizer import AzureSynthesizerConfig\n\nvocode.api_key = "<your API key>"\n\n\nif __name__ == "__main__":\n    microphone_input, speaker_output = create_microphone_input_and_speaker_output(\n        streaming=True, use_default_devices=False\n    )\n\n    conversation = HostedStreamingConversation(\n        input_device=microphone_input,\n        output_device=speaker_output,\n        transcriber_config=DeepgramTranscriberConfig.from_input_device(\n            microphone_input,\n            endpointing_config=PunctuationEndpointingConfig(),\n        ),\n        agent_config=ChatGPTAgentConfig(\n            initial_message=BaseMessage(text="Hello!"),\n            prompt_preamble="Have a pleasant conversation about life",\n        ),\n        synthesizer_config=AzureSynthesizerConfig.from_output_device(speaker_output),\n    )\n    signal.signal(signal.SIGINT, lambda _0, _1: conversation.deactivate())\n    asyncio.run(conversation.start())\n```\n\n# 📞 Phone call quickstarts\n\n- [Inbound calls - Hosted](https://docs.vocode.dev/telephony#inbound-calls)\n- [Outbound calls - Hosted](https://docs.vocode.dev/telephony#outbound-calls)\n- [Telephony Server - Self-hosted](https://github.com/vocodedev/vocode-python/blob/main/examples/telephony_app.py)\n\n\n\n# 🌱 Documentation\n\n[docs.vocode.dev](https://docs.vocode.dev/)\n',
    'author': 'Ajay Raj',
    'author_email': 'ajay@vocode.dev',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/vocodedev/vocode-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
