"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.trustedPrincipals = [];
        const stack = core_1.Stack.of(this);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        this.creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.CompositePrincipal(...provider.roles.map(x => new iam.ArnPrincipal(x.roleArn))),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                Config: props,
                AssumeRoleArn: this.creationRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.creationRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    /**
     * Grants `trustedRole` permissions to assume the creation role.
     */
    addTrustedRole(trustedRole) {
        if (this.trustedPrincipals.includes(trustedRole.roleArn)) {
            return;
        }
        if (!this.creationRole.assumeRolePolicy) {
            throw new Error('unexpected: cluster creation role must have trust policy');
        }
        this.creationRole.assumeRolePolicy.addStatements(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            principals: [new iam.ArnPrincipal(trustedRole.roleArn)],
        }));
        this.trustedPrincipals.push(trustedRole.roleArn);
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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