"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("assert");
const lib_1 = require("../lib");
test('boolean type', () => {
    assert_1.equal(lib_1.Schema.BOOLEAN.inputString, 'boolean');
    assert_1.equal(lib_1.Schema.BOOLEAN.isPrimitive, true);
});
test('binary type', () => {
    assert_1.equal(lib_1.Schema.BINARY.inputString, 'binary');
    assert_1.equal(lib_1.Schema.BINARY.isPrimitive, true);
});
test('bigint type', () => {
    assert_1.equal(lib_1.Schema.BIG_INT.inputString, 'bigint');
    assert_1.equal(lib_1.Schema.BIG_INT.isPrimitive, true);
});
test('double type', () => {
    assert_1.equal(lib_1.Schema.DOUBLE.inputString, 'double');
    assert_1.equal(lib_1.Schema.DOUBLE.isPrimitive, true);
});
test('float type', () => {
    assert_1.equal(lib_1.Schema.FLOAT.inputString, 'float');
    assert_1.equal(lib_1.Schema.FLOAT.isPrimitive, true);
});
test('integer type', () => {
    assert_1.equal(lib_1.Schema.INTEGER.inputString, 'int');
    assert_1.equal(lib_1.Schema.INTEGER.isPrimitive, true);
});
test('smallint type', () => {
    assert_1.equal(lib_1.Schema.SMALL_INT.inputString, 'smallint');
    assert_1.equal(lib_1.Schema.SMALL_INT.isPrimitive, true);
});
test('tinyint type', () => {
    assert_1.equal(lib_1.Schema.TINY_INT.inputString, 'tinyint');
    assert_1.equal(lib_1.Schema.TINY_INT.isPrimitive, true);
});
test('decimal type', () => {
    assert_1.equal(lib_1.Schema.decimal(16).inputString, 'decimal(16)');
    assert_1.equal(lib_1.Schema.decimal(16, 1).inputString, 'decimal(16,1)');
    assert_1.equal(lib_1.Schema.decimal(16).isPrimitive, true);
    assert_1.equal(lib_1.Schema.decimal(16, 1).isPrimitive, true);
});
// TODO: decimal bounds
test('date type', () => {
    assert_1.equal(lib_1.Schema.DATE.inputString, 'date');
    assert_1.equal(lib_1.Schema.DATE.isPrimitive, true);
});
test('timestamp type', () => {
    assert_1.equal(lib_1.Schema.TIMESTAMP.inputString, 'timestamp');
    assert_1.equal(lib_1.Schema.TIMESTAMP.isPrimitive, true);
});
test('string type', () => {
    assert_1.equal(lib_1.Schema.STRING.inputString, 'string');
    assert_1.equal(lib_1.Schema.STRING.isPrimitive, true);
});
test('char type', () => {
    assert_1.equal(lib_1.Schema.char(1).inputString, 'char(1)');
    assert_1.equal(lib_1.Schema.char(1).isPrimitive, true);
});
test('char length must be test(at least 1', () => {
    assert_1.doesNotThrow(() => lib_1.Schema.char(1));
    assert_1.throws(() => lib_1.Schema.char(0));
    assert_1.throws(() => lib_1.Schema.char(-1));
});
test('char length must test(be <= 255', () => {
    assert_1.doesNotThrow(() => lib_1.Schema.char(255));
    assert_1.throws(() => lib_1.Schema.char(256));
});
test('varchar type', () => {
    assert_1.equal(lib_1.Schema.varchar(1).inputString, 'varchar(1)');
    assert_1.equal(lib_1.Schema.varchar(1).isPrimitive, true);
});
test('varchar length must be test(at least 1', () => {
    assert_1.doesNotThrow(() => lib_1.Schema.varchar(1));
    assert_1.throws(() => lib_1.Schema.varchar(0));
    assert_1.throws(() => lib_1.Schema.varchar(-1));
});
test('varchar length must test(be <= 65535', () => {
    assert_1.doesNotThrow(() => lib_1.Schema.varchar(65535));
    assert_1.throws(() => lib_1.Schema.varchar(65536));
});
test('test(array<string>', () => {
    const type = lib_1.Schema.array(lib_1.Schema.STRING);
    assert_1.equal(type.inputString, 'array<string>');
    assert_1.equal(type.isPrimitive, false);
});
test('array<test(char(1)>', () => {
    const type = lib_1.Schema.array(lib_1.Schema.char(1));
    assert_1.equal(type.inputString, 'array<char(1)>');
    assert_1.equal(type.isPrimitive, false);
});
test('test(array<array>', () => {
    const type = lib_1.Schema.array(lib_1.Schema.array(lib_1.Schema.STRING));
    assert_1.equal(type.inputString, 'array<array<string>>');
    assert_1.equal(type.isPrimitive, false);
});
test('test(array<map>', () => {
    const type = lib_1.Schema.array(lib_1.Schema.map(lib_1.Schema.STRING, lib_1.Schema.STRING));
    assert_1.equal(type.inputString, 'array<map<string,string>>');
    assert_1.equal(type.isPrimitive, false);
});
test('test(array<struct>', () => {
    const type = lib_1.Schema.array(lib_1.Schema.struct([{
            name: 'key',
            type: lib_1.Schema.STRING,
        }]));
    assert_1.equal(type.inputString, 'array<struct<key:string>>');
    assert_1.equal(type.isPrimitive, false);
});
test('map<test(string,string>', () => {
    const type = lib_1.Schema.map(lib_1.Schema.STRING, lib_1.Schema.STRING);
    assert_1.equal(type.inputString, 'map<string,string>');
    assert_1.equal(type.isPrimitive, false);
});
test('map<test(int,string>', () => {
    const type = lib_1.Schema.map(lib_1.Schema.INTEGER, lib_1.Schema.STRING);
    assert_1.equal(type.inputString, 'map<int,string>');
    assert_1.equal(type.isPrimitive, false);
});
test('map<char(1),test(char(1)>', () => {
    const type = lib_1.Schema.map(lib_1.Schema.char(1), lib_1.Schema.char(1));
    assert_1.equal(type.inputString, 'map<char(1),char(1)>');
    assert_1.equal(type.isPrimitive, false);
});
test('map<test(string,array>', () => {
    const type = lib_1.Schema.map(lib_1.Schema.char(1), lib_1.Schema.array(lib_1.Schema.STRING));
    assert_1.equal(type.inputString, 'map<char(1),array<string>>');
    assert_1.equal(type.isPrimitive, false);
});
test('map<test(string,map>', () => {
    const type = lib_1.Schema.map(lib_1.Schema.char(1), lib_1.Schema.map(lib_1.Schema.STRING, lib_1.Schema.STRING));
    assert_1.equal(type.inputString, 'map<char(1),map<string,string>>');
    assert_1.equal(type.isPrimitive, false);
});
test('map<test(string,struct>', () => {
    const type = lib_1.Schema.map(lib_1.Schema.char(1), lib_1.Schema.struct([{
            name: 'key',
            type: lib_1.Schema.STRING,
        }]));
    assert_1.equal(type.inputString, 'map<char(1),struct<key:string>>');
    assert_1.equal(type.isPrimitive, false);
});
test('map throws if keyType is test(non-primitive', () => {
    assert_1.throws(() => lib_1.Schema.map(lib_1.Schema.array(lib_1.Schema.STRING), lib_1.Schema.STRING));
    assert_1.throws(() => lib_1.Schema.map(lib_1.Schema.map(lib_1.Schema.STRING, lib_1.Schema.STRING), lib_1.Schema.STRING));
    assert_1.throws(() => lib_1.Schema.map(lib_1.Schema.struct([{
            name: 'key',
            type: lib_1.Schema.STRING,
        }]), lib_1.Schema.STRING));
});
test('struct type', () => {
    const type = lib_1.Schema.struct([{
            name: 'primitive',
            type: lib_1.Schema.STRING,
        }, {
            name: 'with_comment',
            type: lib_1.Schema.STRING,
            comment: 'this has a comment',
        }, {
            name: 'array',
            type: lib_1.Schema.array(lib_1.Schema.STRING),
        }, {
            name: 'map',
            type: lib_1.Schema.map(lib_1.Schema.STRING, lib_1.Schema.STRING),
        }, {
            name: 'nested_struct',
            type: lib_1.Schema.struct([{
                    name: 'nested',
                    type: lib_1.Schema.STRING,
                    comment: 'nested comment',
                }]),
        }]);
    assert_1.equal(type.isPrimitive, false);
    assert_1.equal(type.inputString, 
    // eslint-disable-next-line max-len
    'struct<primitive:string,with_comment:string COMMENT \'this has a comment\',array:array<string>,map:map<string,string>,nested_struct:struct<nested:string COMMENT \'nested comment\'>>');
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZW1hLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzY2hlbWEudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdDQUE4QjtBQUM5QixtQ0FBcUQ7QUFDckQsZ0NBQWdDO0FBRWhDLElBQUksQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFO0lBQ3hCLGNBQUssQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUM3QyxjQUFLLENBQUMsWUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDMUMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsYUFBYSxFQUFFLEdBQUcsRUFBRTtJQUN2QixjQUFLLENBQUMsWUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDM0MsY0FBSyxDQUFDLFlBQU0sQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQ3pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGFBQWEsRUFBRSxHQUFHLEVBQUU7SUFDdkIsY0FBSyxDQUFDLFlBQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzVDLGNBQUssQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUMxQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxhQUFhLEVBQUUsR0FBRyxFQUFFO0lBQ3ZCLGNBQUssQ0FBQyxZQUFNLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUMzQyxjQUFLLENBQUMsWUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDekMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsWUFBWSxFQUFFLEdBQUcsRUFBRTtJQUN0QixjQUFLLENBQUMsWUFBTSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDekMsY0FBSyxDQUFDLFlBQU0sQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQ3hDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7SUFDeEIsY0FBSyxDQUFDLFlBQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3pDLGNBQUssQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUMxQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxlQUFlLEVBQUUsR0FBRyxFQUFFO0lBQ3pCLGNBQUssQ0FBQyxZQUFNLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztJQUNoRCxjQUFLLENBQUMsWUFBTSxDQUFDLFNBQVMsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDNUMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsY0FBYyxFQUFFLEdBQUcsRUFBRTtJQUN4QixjQUFLLENBQUMsWUFBTSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDOUMsY0FBSyxDQUFDLFlBQU0sQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQzNDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7SUFDeEIsY0FBSyxDQUFDLFlBQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsV0FBVyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQ3JELGNBQUssQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDMUQsY0FBSyxDQUFDLFlBQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQzVDLGNBQUssQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDakQsQ0FBQyxDQUFDLENBQUM7QUFDSCx1QkFBdUI7QUFFdkIsSUFBSSxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUU7SUFDckIsY0FBSyxDQUFDLFlBQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ3ZDLGNBQUssQ0FBQyxZQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUN2QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUU7SUFDMUIsY0FBSyxDQUFDLFlBQU0sQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBQ2pELGNBQUssQ0FBQyxZQUFNLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUM1QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxhQUFhLEVBQUUsR0FBRyxFQUFFO0lBQ3ZCLGNBQUssQ0FBQyxZQUFNLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRSxRQUFRLENBQUMsQ0FBQztJQUMzQyxjQUFLLENBQUMsWUFBTSxDQUFDLE1BQU0sQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDekMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsV0FBVyxFQUFFLEdBQUcsRUFBRTtJQUNyQixjQUFLLENBQUMsWUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDN0MsY0FBSyxDQUFDLFlBQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQzFDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxxQkFBWSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNuQyxlQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzdCLGVBQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxZQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUNoQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxpQ0FBaUMsRUFBRSxHQUFHLEVBQUU7SUFDM0MscUJBQVksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxZQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDckMsZUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztBQUNqQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFO0lBQ3hCLGNBQUssQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxZQUFZLENBQUMsQ0FBQztJQUNuRCxjQUFLLENBQUMsWUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLENBQUM7QUFDN0MsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0NBQXdDLEVBQUUsR0FBRyxFQUFFO0lBQ2xELHFCQUFZLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3RDLGVBQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxZQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDaEMsZUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ25DLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxxQkFBWSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUMxQyxlQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0FBQ3RDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLG9CQUFvQixFQUFFLEdBQUcsRUFBRTtJQUM5QixNQUFNLElBQUksR0FBRyxZQUFNLENBQUMsS0FBSyxDQUFDLFlBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUN6QyxjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxlQUFlLENBQUMsQ0FBQztJQUN6QyxjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNqQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7SUFDL0IsTUFBTSxJQUFJLEdBQUcsWUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDMUMsY0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUMxQyxjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNqQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7SUFDN0IsTUFBTSxJQUFJLEdBQUcsWUFBTSxDQUFDLEtBQUssQ0FDdkIsWUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUMvQixjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO0lBQ2hELGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlCQUFpQixFQUFFLEdBQUcsRUFBRTtJQUMzQixNQUFNLElBQUksR0FBRyxZQUFNLENBQUMsS0FBSyxDQUN2QixZQUFNLENBQUMsR0FBRyxDQUFDLFlBQU0sQ0FBQyxNQUFNLEVBQUUsWUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7SUFDNUMsY0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsMkJBQTJCLENBQUMsQ0FBQztJQUNyRCxjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNqQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvQkFBb0IsRUFBRSxHQUFHLEVBQUU7SUFDOUIsTUFBTSxJQUFJLEdBQUcsWUFBTSxDQUFDLEtBQUssQ0FDdkIsWUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBQ2IsSUFBSSxFQUFFLEtBQUs7WUFDWCxJQUFJLEVBQUUsWUFBTSxDQUFDLE1BQU07U0FDcEIsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNQLGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLDJCQUEyQixDQUFDLENBQUM7SUFDckQsY0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDakMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMseUJBQXlCLEVBQUUsR0FBRyxFQUFFO0lBQ25DLE1BQU0sSUFBSSxHQUFHLFlBQU0sQ0FBQyxHQUFHLENBQ3JCLFlBQU0sQ0FBQyxNQUFNLEVBQ2IsWUFBTSxDQUFDLE1BQU0sQ0FDZCxDQUFDO0lBQ0YsY0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztJQUM5QyxjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUNqQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQkFBc0IsRUFBRSxHQUFHLEVBQUU7SUFDaEMsTUFBTSxJQUFJLEdBQUcsWUFBTSxDQUFDLEdBQUcsQ0FDckIsWUFBTSxDQUFDLE9BQU8sRUFDZCxZQUFNLENBQUMsTUFBTSxDQUNkLENBQUM7SUFDRixjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBQzNDLGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJCQUEyQixFQUFFLEdBQUcsRUFBRTtJQUNyQyxNQUFNLElBQUksR0FBRyxZQUFNLENBQUMsR0FBRyxDQUNyQixZQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUNkLFlBQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQ2YsQ0FBQztJQUNGLGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLHNCQUFzQixDQUFDLENBQUM7SUFDaEQsY0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDakMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO0lBQ2xDLE1BQU0sSUFBSSxHQUFHLFlBQU0sQ0FBQyxHQUFHLENBQ3JCLFlBQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQ2QsWUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFNLENBQUMsTUFBTSxDQUFDLENBQzVCLENBQUM7SUFDRixjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSw0QkFBNEIsQ0FBQyxDQUFDO0lBQ3RELGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNCQUFzQixFQUFFLEdBQUcsRUFBRTtJQUNoQyxNQUFNLElBQUksR0FBRyxZQUFNLENBQUMsR0FBRyxDQUNyQixZQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUNkLFlBQU0sQ0FBQyxHQUFHLENBQ1IsWUFBTSxDQUFDLE1BQU0sRUFDYixZQUFNLENBQUMsTUFBTSxDQUFDLENBQ2pCLENBQUM7SUFDRixjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxpQ0FBaUMsQ0FBQyxDQUFDO0lBQzNELGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlCQUF5QixFQUFFLEdBQUcsRUFBRTtJQUNuQyxNQUFNLElBQUksR0FBRyxZQUFNLENBQUMsR0FBRyxDQUNyQixZQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUNkLFlBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNiLElBQUksRUFBRSxLQUFLO1lBQ1gsSUFBSSxFQUFFLFlBQU0sQ0FBQyxNQUFNO1NBQ3BCLENBQUMsQ0FBQyxDQUNKLENBQUM7SUFDRixjQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxpQ0FBaUMsQ0FBQyxDQUFDO0lBQzNELGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDZDQUE2QyxFQUFFLEdBQUcsRUFBRTtJQUN2RCxlQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsWUFBTSxDQUFDLEdBQUcsQ0FDckIsWUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFNLENBQUMsTUFBTSxDQUFDLEVBQzNCLFlBQU0sQ0FBQyxNQUFNLENBQ2QsQ0FBQyxDQUFDO0lBQ0gsZUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQU0sQ0FBQyxHQUFHLENBQ3JCLFlBQU0sQ0FBQyxHQUFHLENBQUMsWUFBTSxDQUFDLE1BQU0sRUFBRSxZQUFNLENBQUMsTUFBTSxDQUFDLEVBQ3hDLFlBQU0sQ0FBQyxNQUFNLENBQ2QsQ0FBQyxDQUFDO0lBQ0gsZUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQU0sQ0FBQyxHQUFHLENBQ3JCLFlBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNiLElBQUksRUFBRSxLQUFLO1lBQ1gsSUFBSSxFQUFFLFlBQU0sQ0FBQyxNQUFNO1NBQ3BCLENBQUMsQ0FBQyxFQUNILFlBQU0sQ0FBQyxNQUFNLENBQ2QsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsYUFBYSxFQUFFLEdBQUcsRUFBRTtJQUN2QixNQUFNLElBQUksR0FBRyxZQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDMUIsSUFBSSxFQUFFLFdBQVc7WUFDakIsSUFBSSxFQUFFLFlBQU0sQ0FBQyxNQUFNO1NBQ3BCLEVBQUU7WUFDRCxJQUFJLEVBQUUsY0FBYztZQUNwQixJQUFJLEVBQUUsWUFBTSxDQUFDLE1BQU07WUFDbkIsT0FBTyxFQUFFLG9CQUFvQjtTQUM5QixFQUFFO1lBQ0QsSUFBSSxFQUFFLE9BQU87WUFDYixJQUFJLEVBQUUsWUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFNLENBQUMsTUFBTSxDQUFDO1NBQ2xDLEVBQUU7WUFDRCxJQUFJLEVBQUUsS0FBSztZQUNYLElBQUksRUFBRSxZQUFNLENBQUMsR0FBRyxDQUFDLFlBQU0sQ0FBQyxNQUFNLEVBQUUsWUFBTSxDQUFDLE1BQU0sQ0FBQztTQUMvQyxFQUFFO1lBQ0QsSUFBSSxFQUFFLGVBQWU7WUFDckIsSUFBSSxFQUFFLFlBQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztvQkFDbkIsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsSUFBSSxFQUFFLFlBQU0sQ0FBQyxNQUFNO29CQUNuQixPQUFPLEVBQUUsZ0JBQWdCO2lCQUMxQixDQUFDLENBQUM7U0FDSixDQUFDLENBQUMsQ0FBQztJQUVKLGNBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQy9CLGNBQUssQ0FDSCxJQUFJLENBQUMsV0FBVztJQUNoQixtQ0FBbUM7SUFDbkMsdUxBQXVMLENBQUMsQ0FBQztBQUM3TCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgZG9lc05vdFRocm93LCBlcXVhbCwgdGhyb3dzIH0gZnJvbSAnYXNzZXJ0JztcbmltcG9ydCB7IFNjaGVtYSB9IGZyb20gJy4uL2xpYic7XG5cbnRlc3QoJ2Jvb2xlYW4gdHlwZScsICgpID0+IHtcbiAgZXF1YWwoU2NoZW1hLkJPT0xFQU4uaW5wdXRTdHJpbmcsICdib29sZWFuJyk7XG4gIGVxdWFsKFNjaGVtYS5CT09MRUFOLmlzUHJpbWl0aXZlLCB0cnVlKTtcbn0pO1xuXG50ZXN0KCdiaW5hcnkgdHlwZScsICgpID0+IHtcbiAgZXF1YWwoU2NoZW1hLkJJTkFSWS5pbnB1dFN0cmluZywgJ2JpbmFyeScpO1xuICBlcXVhbChTY2hlbWEuQklOQVJZLmlzUHJpbWl0aXZlLCB0cnVlKTtcbn0pO1xuXG50ZXN0KCdiaWdpbnQgdHlwZScsICgpID0+IHtcbiAgZXF1YWwoU2NoZW1hLkJJR19JTlQuaW5wdXRTdHJpbmcsICdiaWdpbnQnKTtcbiAgZXF1YWwoU2NoZW1hLkJJR19JTlQuaXNQcmltaXRpdmUsIHRydWUpO1xufSk7XG5cbnRlc3QoJ2RvdWJsZSB0eXBlJywgKCkgPT4ge1xuICBlcXVhbChTY2hlbWEuRE9VQkxFLmlucHV0U3RyaW5nLCAnZG91YmxlJyk7XG4gIGVxdWFsKFNjaGVtYS5ET1VCTEUuaXNQcmltaXRpdmUsIHRydWUpO1xufSk7XG5cbnRlc3QoJ2Zsb2F0IHR5cGUnLCAoKSA9PiB7XG4gIGVxdWFsKFNjaGVtYS5GTE9BVC5pbnB1dFN0cmluZywgJ2Zsb2F0Jyk7XG4gIGVxdWFsKFNjaGVtYS5GTE9BVC5pc1ByaW1pdGl2ZSwgdHJ1ZSk7XG59KTtcblxudGVzdCgnaW50ZWdlciB0eXBlJywgKCkgPT4ge1xuICBlcXVhbChTY2hlbWEuSU5URUdFUi5pbnB1dFN0cmluZywgJ2ludCcpO1xuICBlcXVhbChTY2hlbWEuSU5URUdFUi5pc1ByaW1pdGl2ZSwgdHJ1ZSk7XG59KTtcblxudGVzdCgnc21hbGxpbnQgdHlwZScsICgpID0+IHtcbiAgZXF1YWwoU2NoZW1hLlNNQUxMX0lOVC5pbnB1dFN0cmluZywgJ3NtYWxsaW50Jyk7XG4gIGVxdWFsKFNjaGVtYS5TTUFMTF9JTlQuaXNQcmltaXRpdmUsIHRydWUpO1xufSk7XG5cbnRlc3QoJ3RpbnlpbnQgdHlwZScsICgpID0+IHtcbiAgZXF1YWwoU2NoZW1hLlRJTllfSU5ULmlucHV0U3RyaW5nLCAndGlueWludCcpO1xuICBlcXVhbChTY2hlbWEuVElOWV9JTlQuaXNQcmltaXRpdmUsIHRydWUpO1xufSk7XG5cbnRlc3QoJ2RlY2ltYWwgdHlwZScsICgpID0+IHtcbiAgZXF1YWwoU2NoZW1hLmRlY2ltYWwoMTYpLmlucHV0U3RyaW5nLCAnZGVjaW1hbCgxNiknKTtcbiAgZXF1YWwoU2NoZW1hLmRlY2ltYWwoMTYsIDEpLmlucHV0U3RyaW5nLCAnZGVjaW1hbCgxNiwxKScpO1xuICBlcXVhbChTY2hlbWEuZGVjaW1hbCgxNikuaXNQcmltaXRpdmUsIHRydWUpO1xuICBlcXVhbChTY2hlbWEuZGVjaW1hbCgxNiwgMSkuaXNQcmltaXRpdmUsIHRydWUpO1xufSk7XG4vLyBUT0RPOiBkZWNpbWFsIGJvdW5kc1xuXG50ZXN0KCdkYXRlIHR5cGUnLCAoKSA9PiB7XG4gIGVxdWFsKFNjaGVtYS5EQVRFLmlucHV0U3RyaW5nLCAnZGF0ZScpO1xuICBlcXVhbChTY2hlbWEuREFURS5pc1ByaW1pdGl2ZSwgdHJ1ZSk7XG59KTtcblxudGVzdCgndGltZXN0YW1wIHR5cGUnLCAoKSA9PiB7XG4gIGVxdWFsKFNjaGVtYS5USU1FU1RBTVAuaW5wdXRTdHJpbmcsICd0aW1lc3RhbXAnKTtcbiAgZXF1YWwoU2NoZW1hLlRJTUVTVEFNUC5pc1ByaW1pdGl2ZSwgdHJ1ZSk7XG59KTtcblxudGVzdCgnc3RyaW5nIHR5cGUnLCAoKSA9PiB7XG4gIGVxdWFsKFNjaGVtYS5TVFJJTkcuaW5wdXRTdHJpbmcsICdzdHJpbmcnKTtcbiAgZXF1YWwoU2NoZW1hLlNUUklORy5pc1ByaW1pdGl2ZSwgdHJ1ZSk7XG59KTtcblxudGVzdCgnY2hhciB0eXBlJywgKCkgPT4ge1xuICBlcXVhbChTY2hlbWEuY2hhcigxKS5pbnB1dFN0cmluZywgJ2NoYXIoMSknKTtcbiAgZXF1YWwoU2NoZW1hLmNoYXIoMSkuaXNQcmltaXRpdmUsIHRydWUpO1xufSk7XG5cbnRlc3QoJ2NoYXIgbGVuZ3RoIG11c3QgYmUgdGVzdChhdCBsZWFzdCAxJywgKCkgPT4ge1xuICBkb2VzTm90VGhyb3coKCkgPT4gU2NoZW1hLmNoYXIoMSkpO1xuICB0aHJvd3MoKCkgPT4gU2NoZW1hLmNoYXIoMCkpO1xuICB0aHJvd3MoKCkgPT4gU2NoZW1hLmNoYXIoLTEpKTtcbn0pO1xuXG50ZXN0KCdjaGFyIGxlbmd0aCBtdXN0IHRlc3QoYmUgPD0gMjU1JywgKCkgPT4ge1xuICBkb2VzTm90VGhyb3coKCkgPT4gU2NoZW1hLmNoYXIoMjU1KSk7XG4gIHRocm93cygoKSA9PiBTY2hlbWEuY2hhcigyNTYpKTtcbn0pO1xuXG50ZXN0KCd2YXJjaGFyIHR5cGUnLCAoKSA9PiB7XG4gIGVxdWFsKFNjaGVtYS52YXJjaGFyKDEpLmlucHV0U3RyaW5nLCAndmFyY2hhcigxKScpO1xuICBlcXVhbChTY2hlbWEudmFyY2hhcigxKS5pc1ByaW1pdGl2ZSwgdHJ1ZSk7XG59KTtcblxudGVzdCgndmFyY2hhciBsZW5ndGggbXVzdCBiZSB0ZXN0KGF0IGxlYXN0IDEnLCAoKSA9PiB7XG4gIGRvZXNOb3RUaHJvdygoKSA9PiBTY2hlbWEudmFyY2hhcigxKSk7XG4gIHRocm93cygoKSA9PiBTY2hlbWEudmFyY2hhcigwKSk7XG4gIHRocm93cygoKSA9PiBTY2hlbWEudmFyY2hhcigtMSkpO1xufSk7XG5cbnRlc3QoJ3ZhcmNoYXIgbGVuZ3RoIG11c3QgdGVzdChiZSA8PSA2NTUzNScsICgpID0+IHtcbiAgZG9lc05vdFRocm93KCgpID0+IFNjaGVtYS52YXJjaGFyKDY1NTM1KSk7XG4gIHRocm93cygoKSA9PiBTY2hlbWEudmFyY2hhcig2NTUzNikpO1xufSk7XG5cbnRlc3QoJ3Rlc3QoYXJyYXk8c3RyaW5nPicsICgpID0+IHtcbiAgY29uc3QgdHlwZSA9IFNjaGVtYS5hcnJheShTY2hlbWEuU1RSSU5HKTtcbiAgZXF1YWwodHlwZS5pbnB1dFN0cmluZywgJ2FycmF5PHN0cmluZz4nKTtcbiAgZXF1YWwodHlwZS5pc1ByaW1pdGl2ZSwgZmFsc2UpO1xufSk7XG5cbnRlc3QoJ2FycmF5PHRlc3QoY2hhcigxKT4nLCAoKSA9PiB7XG4gIGNvbnN0IHR5cGUgPSBTY2hlbWEuYXJyYXkoU2NoZW1hLmNoYXIoMSkpO1xuICBlcXVhbCh0eXBlLmlucHV0U3RyaW5nLCAnYXJyYXk8Y2hhcigxKT4nKTtcbiAgZXF1YWwodHlwZS5pc1ByaW1pdGl2ZSwgZmFsc2UpO1xufSk7XG5cbnRlc3QoJ3Rlc3QoYXJyYXk8YXJyYXk+JywgKCkgPT4ge1xuICBjb25zdCB0eXBlID0gU2NoZW1hLmFycmF5KFxuICAgIFNjaGVtYS5hcnJheShTY2hlbWEuU1RSSU5HKSk7XG4gIGVxdWFsKHR5cGUuaW5wdXRTdHJpbmcsICdhcnJheTxhcnJheTxzdHJpbmc+PicpO1xuICBlcXVhbCh0eXBlLmlzUHJpbWl0aXZlLCBmYWxzZSk7XG59KTtcblxudGVzdCgndGVzdChhcnJheTxtYXA+JywgKCkgPT4ge1xuICBjb25zdCB0eXBlID0gU2NoZW1hLmFycmF5KFxuICAgIFNjaGVtYS5tYXAoU2NoZW1hLlNUUklORywgU2NoZW1hLlNUUklORykpO1xuICBlcXVhbCh0eXBlLmlucHV0U3RyaW5nLCAnYXJyYXk8bWFwPHN0cmluZyxzdHJpbmc+PicpO1xuICBlcXVhbCh0eXBlLmlzUHJpbWl0aXZlLCBmYWxzZSk7XG59KTtcblxudGVzdCgndGVzdChhcnJheTxzdHJ1Y3Q+JywgKCkgPT4ge1xuICBjb25zdCB0eXBlID0gU2NoZW1hLmFycmF5KFxuICAgIFNjaGVtYS5zdHJ1Y3QoW3tcbiAgICAgIG5hbWU6ICdrZXknLFxuICAgICAgdHlwZTogU2NoZW1hLlNUUklORyxcbiAgICB9XSkpO1xuICBlcXVhbCh0eXBlLmlucHV0U3RyaW5nLCAnYXJyYXk8c3RydWN0PGtleTpzdHJpbmc+PicpO1xuICBlcXVhbCh0eXBlLmlzUHJpbWl0aXZlLCBmYWxzZSk7XG59KTtcblxudGVzdCgnbWFwPHRlc3Qoc3RyaW5nLHN0cmluZz4nLCAoKSA9PiB7XG4gIGNvbnN0IHR5cGUgPSBTY2hlbWEubWFwKFxuICAgIFNjaGVtYS5TVFJJTkcsXG4gICAgU2NoZW1hLlNUUklORyxcbiAgKTtcbiAgZXF1YWwodHlwZS5pbnB1dFN0cmluZywgJ21hcDxzdHJpbmcsc3RyaW5nPicpO1xuICBlcXVhbCh0eXBlLmlzUHJpbWl0aXZlLCBmYWxzZSk7XG59KTtcblxudGVzdCgnbWFwPHRlc3QoaW50LHN0cmluZz4nLCAoKSA9PiB7XG4gIGNvbnN0IHR5cGUgPSBTY2hlbWEubWFwKFxuICAgIFNjaGVtYS5JTlRFR0VSLFxuICAgIFNjaGVtYS5TVFJJTkcsXG4gICk7XG4gIGVxdWFsKHR5cGUuaW5wdXRTdHJpbmcsICdtYXA8aW50LHN0cmluZz4nKTtcbiAgZXF1YWwodHlwZS5pc1ByaW1pdGl2ZSwgZmFsc2UpO1xufSk7XG5cbnRlc3QoJ21hcDxjaGFyKDEpLHRlc3QoY2hhcigxKT4nLCAoKSA9PiB7XG4gIGNvbnN0IHR5cGUgPSBTY2hlbWEubWFwKFxuICAgIFNjaGVtYS5jaGFyKDEpLFxuICAgIFNjaGVtYS5jaGFyKDEpLFxuICApO1xuICBlcXVhbCh0eXBlLmlucHV0U3RyaW5nLCAnbWFwPGNoYXIoMSksY2hhcigxKT4nKTtcbiAgZXF1YWwodHlwZS5pc1ByaW1pdGl2ZSwgZmFsc2UpO1xufSk7XG5cbnRlc3QoJ21hcDx0ZXN0KHN0cmluZyxhcnJheT4nLCAoKSA9PiB7XG4gIGNvbnN0IHR5cGUgPSBTY2hlbWEubWFwKFxuICAgIFNjaGVtYS5jaGFyKDEpLFxuICAgIFNjaGVtYS5hcnJheShTY2hlbWEuU1RSSU5HKSxcbiAgKTtcbiAgZXF1YWwodHlwZS5pbnB1dFN0cmluZywgJ21hcDxjaGFyKDEpLGFycmF5PHN0cmluZz4+Jyk7XG4gIGVxdWFsKHR5cGUuaXNQcmltaXRpdmUsIGZhbHNlKTtcbn0pO1xuXG50ZXN0KCdtYXA8dGVzdChzdHJpbmcsbWFwPicsICgpID0+IHtcbiAgY29uc3QgdHlwZSA9IFNjaGVtYS5tYXAoXG4gICAgU2NoZW1hLmNoYXIoMSksXG4gICAgU2NoZW1hLm1hcChcbiAgICAgIFNjaGVtYS5TVFJJTkcsXG4gICAgICBTY2hlbWEuU1RSSU5HKSxcbiAgKTtcbiAgZXF1YWwodHlwZS5pbnB1dFN0cmluZywgJ21hcDxjaGFyKDEpLG1hcDxzdHJpbmcsc3RyaW5nPj4nKTtcbiAgZXF1YWwodHlwZS5pc1ByaW1pdGl2ZSwgZmFsc2UpO1xufSk7XG5cbnRlc3QoJ21hcDx0ZXN0KHN0cmluZyxzdHJ1Y3Q+JywgKCkgPT4ge1xuICBjb25zdCB0eXBlID0gU2NoZW1hLm1hcChcbiAgICBTY2hlbWEuY2hhcigxKSxcbiAgICBTY2hlbWEuc3RydWN0KFt7XG4gICAgICBuYW1lOiAna2V5JyxcbiAgICAgIHR5cGU6IFNjaGVtYS5TVFJJTkcsXG4gICAgfV0pLFxuICApO1xuICBlcXVhbCh0eXBlLmlucHV0U3RyaW5nLCAnbWFwPGNoYXIoMSksc3RydWN0PGtleTpzdHJpbmc+PicpO1xuICBlcXVhbCh0eXBlLmlzUHJpbWl0aXZlLCBmYWxzZSk7XG59KTtcblxudGVzdCgnbWFwIHRocm93cyBpZiBrZXlUeXBlIGlzIHRlc3Qobm9uLXByaW1pdGl2ZScsICgpID0+IHtcbiAgdGhyb3dzKCgpID0+IFNjaGVtYS5tYXAoXG4gICAgU2NoZW1hLmFycmF5KFNjaGVtYS5TVFJJTkcpLFxuICAgIFNjaGVtYS5TVFJJTkcsXG4gICkpO1xuICB0aHJvd3MoKCkgPT4gU2NoZW1hLm1hcChcbiAgICBTY2hlbWEubWFwKFNjaGVtYS5TVFJJTkcsIFNjaGVtYS5TVFJJTkcpLFxuICAgIFNjaGVtYS5TVFJJTkcsXG4gICkpO1xuICB0aHJvd3MoKCkgPT4gU2NoZW1hLm1hcChcbiAgICBTY2hlbWEuc3RydWN0KFt7XG4gICAgICBuYW1lOiAna2V5JyxcbiAgICAgIHR5cGU6IFNjaGVtYS5TVFJJTkcsXG4gICAgfV0pLFxuICAgIFNjaGVtYS5TVFJJTkcsXG4gICkpO1xufSk7XG5cbnRlc3QoJ3N0cnVjdCB0eXBlJywgKCkgPT4ge1xuICBjb25zdCB0eXBlID0gU2NoZW1hLnN0cnVjdChbe1xuICAgIG5hbWU6ICdwcmltaXRpdmUnLFxuICAgIHR5cGU6IFNjaGVtYS5TVFJJTkcsXG4gIH0sIHtcbiAgICBuYW1lOiAnd2l0aF9jb21tZW50JyxcbiAgICB0eXBlOiBTY2hlbWEuU1RSSU5HLFxuICAgIGNvbW1lbnQ6ICd0aGlzIGhhcyBhIGNvbW1lbnQnLFxuICB9LCB7XG4gICAgbmFtZTogJ2FycmF5JyxcbiAgICB0eXBlOiBTY2hlbWEuYXJyYXkoU2NoZW1hLlNUUklORyksXG4gIH0sIHtcbiAgICBuYW1lOiAnbWFwJyxcbiAgICB0eXBlOiBTY2hlbWEubWFwKFNjaGVtYS5TVFJJTkcsIFNjaGVtYS5TVFJJTkcpLFxuICB9LCB7XG4gICAgbmFtZTogJ25lc3RlZF9zdHJ1Y3QnLFxuICAgIHR5cGU6IFNjaGVtYS5zdHJ1Y3QoW3tcbiAgICAgIG5hbWU6ICduZXN0ZWQnLFxuICAgICAgdHlwZTogU2NoZW1hLlNUUklORyxcbiAgICAgIGNvbW1lbnQ6ICduZXN0ZWQgY29tbWVudCcsXG4gICAgfV0pLFxuICB9XSk7XG5cbiAgZXF1YWwodHlwZS5pc1ByaW1pdGl2ZSwgZmFsc2UpO1xuICBlcXVhbChcbiAgICB0eXBlLmlucHV0U3RyaW5nLFxuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgJ3N0cnVjdDxwcmltaXRpdmU6c3RyaW5nLHdpdGhfY29tbWVudDpzdHJpbmcgQ09NTUVOVCBcXCd0aGlzIGhhcyBhIGNvbW1lbnRcXCcsYXJyYXk6YXJyYXk8c3RyaW5nPixtYXA6bWFwPHN0cmluZyxzdHJpbmc+LG5lc3RlZF9zdHJ1Y3Q6c3RydWN0PG5lc3RlZDpzdHJpbmcgQ09NTUVOVCBcXCduZXN0ZWQgY29tbWVudFxcJz4+Jyk7XG59KTsiXX0=