from typing import List, Tuple, Callable, Dict

from tunepy2.interfaces import AbstractGenomeFactory, AbstractRandomNumberGenerator
from tunepy2.internal import Genome
import numpy as np


class HillClimberGenomeFactory(AbstractGenomeFactory):
    def __init__(
            self,
            dimensions: Tuple,
            rng: AbstractRandomNumberGenerator,
            fitness_func: Callable[..., float],
            *args,
            **kwargs):
        """
        Creates a new HillClimberGenomeFactory

        :param dimensions: dimensions of bitstrings of Genome objects generated by the factory
        :param rng: random number generator
        :param fitness_func: fitness function passed into new Genome objects that accepts a bitstring
        :param args: will be passed into fitness_func
        :param kwargs: will be passed into fitness_func
        """
        self._dimensions = dimensions
        self._rng = rng
        self._fitness_func = fitness_func
        self._bit_index = 0
        self._args = args
        self._kwargs = kwargs

    def build(self, prior_genomes: List[Genome]) -> Genome:
        """ Builds a new Genome from a list of priors

        :param prior_genomes: list of prior Genomes
        :return: a new Genome object
        """
        def bit_flip(bit_value):
            if bit_value == 0:
                return 1
            else:
                return 0

        selected_genome = prior_genomes[self._rng.random_int(0, len(prior_genomes))]
        new_bitstring = np.array(selected_genome.bitstring)
        new_bitstring.ravel()

        self._bit_index %= len(new_bitstring)

        new_bitstring[self._bit_index] = bit_flip(new_bitstring[self._bit_index])
        new_bitstring = new_bitstring.reshape(shape=self._dimensions)

        self._bit_index += 1

        return Genome(self._fitness_func, new_bitstring, *self._args, **self._kwargs)

    @property
    def dimensions(self):
        """

        :return: the dimensions of Genome objects returned by this factory
        """
        return self._dimensions
