# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['flask_paseto_extended']

package_data = \
{'': ['*']}

install_requires = \
['Flask-Login==0.5.0',
 'Flask>=2.0.2,<3.0.0',
 'Werkzeug==2.0.3',
 'pyseto>=1.5.0,<2.0.0']

extras_require = \
{':extra == "docs"': ['Sphinx[docs]>=4.3.1,<6.0.0',
                      'sphinx-rtd-theme[docs]>=1.0.0,<2.0.0',
                      'sphinx-autodoc-typehints[docs]==1.12.0']}

setup_kwargs = {
    'name': 'flask-paseto-extended',
    'version': '0.4.2',
    'description': 'PASETO (Platform-Agnostic Security Tokens) for Flask applications.',
    'long_description': '# Flask PASETO Extended\n\n![Flask PASETO Extended](https://github.com/dajiaji/flask-paseto-extended/wiki/images/flask_paseto_extended_header.png)\n\n[![PyPI version](https://badge.fury.io/py/flask-paseto-extended.svg)](https://badge.fury.io/py/flask-paseto-extended)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/flask-paseto-extended)\n[![Documentation Status](https://readthedocs.org/projects/flask-paseto-extended/badge/?version=latest)](https://flask-paseto-extended.readthedocs.io/en/latest/?badge=latest)\n![Github CI](https://github.com/dajiaji/flask-paseto-extended/actions/workflows/python-package.yml/badge.svg)\n[![codecov](https://codecov.io/gh/dajiaji/flask-paseto-extended/branch/main/graph/badge.svg?token=QN8GXEYEP3)](https://codecov.io/gh/dajiaji/flask-paseto-extended)\n\nFlask-PASETO-Extended provides following four classes to use [PASETO (Platform-Agnostic Security Tokens)](https://paseto.io/) for Flask applications:\n\n- **PasetoIssuer**\n  - This class can be used for issuing `public` (signed) PASETO. It is suitable for using PASETO as API tokens.\n- **PasetoVerifier**\n  - This class can be used for verifying `public` (signed) PASETO. It is suitable for using PASETO as API tokens.\n- **PasetoCookieSessionInterface**\n  - Flask (`Flask.sessions`) stores session information as a Cookie value. By using this class, you can serialize the session information as a `local` (encrypted and then MACed) PASETO.\n- **PasetoLoginManager**\n  - By using this class together with [Flask-Login](https://github.com/maxcountryman/flask-login), you can use a `local` PASETO for remember-me tokens which is also encoded into a Cookie value.\n\nFor encoding/decoding PASETO, we have adopted [PySETO](https://github.com/dajiaji/pyseto),\nwhich is a PASETO/PASERK implementation supporting all of PASETO versions (\n[v4](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version4.md),\n[v3](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version3.md),\n[v2](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version2.md) and\n[v1](https://github.com/paseto-standard/paseto-spec/blob/master/docs/01-Protocol-Versions/Version1.md)) and purposes (`local` and `public`).\n\n## Index\n- [Installation](#installation)\n- [Basic Usage](#basic-usage)\n  - [PasetoIssuer/PasetoVerifier](#pasetoissuerverifier)\n  - [PasetoCookieSessionInterface](#pasetocookiesessioninterface)\n  - [PasetoLoginManager](#pasetologinmanager)\n- [API Reference](#api-reference)\n- [Tests](#tests)\n- [Contributing](#contributing)\n\n## Installation\n\nYou can install Flask-PASETO-Extended with pip:\n\n```py\n$ pip install flask-paseto-extended\n```\n\n## Basic Usage\n\nFlask-PASETO-Extended provides three classes for each purpose.\n\n### PasetoIssuer/Verifier\n\n`PasetoIssuer` and `PasetoVerifier` can be used for issuing and verifying `public` (signed) PASETO tokens.\n\nBy using `PasetoIssuer`, you can easily implement the endpoint issuing PASETO tokens as follows:\n\n```py\nimport flask\n\nfrom flask_paseto_extended import PasetoIssuer\n\n# Mock user database.\nusers = {"foo@bar.example": {"password": "mysecret"}}\n\n\napp = flask.Flask(__name__)\n\napp.config["PASETO_ISS"] = "https://issuer.example"\napp.config["PASETO_PRIVATE_KEYS"] = [\n    {\n        "version": 4,\n        "key": "-----BEGIN PRIVATE KEY-----\\nMC4CAQAwBQYDK2VwBCIEILTL+0PfTOIQcn2VPkpxMwf6Gbt9n4UEFDjZ4RuUKjd0\\n-----END PRIVATE KEY-----",\n    },\n    # PASERK can also be used (RECOMMENDED).\n    # {\n    #     "paserk": "k4.secret.tMv7Q99M4hByfZU-SnEzB_oZu32fhQQUONnhG5QqN3Qeudu7vAR8A_1wYE4AcfCYfhayi3VyJcEfAEFdDiCxog",\n    # },\n]\nissuer = PasetoIssuer(app)\n\n\n@app.route("/login", methods=["POST"])\ndef login():\n    email = flask.request.form["email"]\n    if flask.request.form["password"] != users[email]["password"]:\n        return "Bad login"\n\n    token = issuer.issue(payload={"user": {"email": email}})\n    resp = flask.redirect(flask.url_for("protected_me"))\n    resp.set_cookie("paseto", token, httponly=True)  # Note: MUST add secure=True in production\n    return resp\n```\n\nOn the other hand, by using `PasetoVerifier`, you can easily implement the endpoint verifying PASETO tokens. You can enable PASETO token verification in your APIs by simply adding `@paseto_required` decorator to the API definitions. In the APIs, you can refer to the veified PASETO token with `current_paseto`.\n\n```py\nimport flask\nfrom flask import jsonify, make_response\n\nfrom flask_paseto_extended import PasetoVerifier, current_paseto, paseto_required\n\n# Mock user database.\nusers = {"foo@bar.example": {"password": "mysecret"}}\n\napp = flask.Flask(__name__)\n\n# Configurations for PasetoVerifier.\napp.config["PASETO_PUBLIC_KEYS"] = [\n    {\n        "iss": "https://issuer.exmaple",\n        "version": 4,\n        "key": "-----BEGIN PUBLIC KEY-----\\nMCowBQYDK2VwAyEAHrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI=\\n-----END PUBLIC KEY-----",\n    },\n    # PASERK can also be used (RECOMMENDED).\n    # {\n    #     "iss": "https://issuer.exmaple",\n    #     "paserk": "k4.public.Hrnbu7wEfAP9cGBOAHHwmH4Wsot1ciXBHwBBXQ4gsaI",\n    # },\n]\nverifier = PasetoVerifier(app)\n\n\n@verifier.token_loader\ndef token_loader(req: flask.Request):\n    # You must implement a callback func to extract a PASETO token from each request.\n    return req.cookies.get("paseto", None)\n\n\n@verifier.verification_error_handler\ndef verification_error_handler():\n    # You must also implement a callback func to handle token verification errors..\n    resp = make_response("Unauthorized")\n    resp.delete_cookie("paseto", httponly=True)\n    return resp\n\n\n@app.route("/protected/me")\n@paseto_required()\ndef protected_me():\n    return jsonify(current_paseto.payload["user"])\n```\n\nSee [examples/issuer_and_verifier.py](https://github.com/dajiaji/flask-paseto-extended/blob/main/examples/issuer_and_verifier.py) for a sample code that actually works.\n\n\n### PasetoCookieSessionInterface\n\nFlask (`Flask.sessions`) stores session information as a Cookie value. By using this class, you can serialize the session information as an encrypted (and then MACed) PASETO.\n\nThis class can be used as follows:\n\n```py\nimport flask\nfrom flask_paseto_extended import PasetoCookieSessionInterface\n\napp = flask.Flask(__name__)\napp.secret_key = "super secret string"\n\n# Use PASETO("v4" by default) for cookie sessions.\napp.session_interface = PasetoCookieSessionInterface()\n```\n\nSee [examples/cookie_session.py](https://github.com/dajiaji/flask-paseto-extended/blob/main/examples/cookie_session.py) for a sample code that actually works.\n\n### PasetoLoginManager\n\nBy using this class together with [Flask-Login](https://github.com/maxcountryman/flask-login), you can use PASETO for remember-me tokens which is also encoded into a Cookie value.\n\nThis class can be used as follows:\n\n```py\nimport flask\nimport flask_login\n# Import PasetoLoginManager instead of flask_login.LoginManager.\nfrom flask_paseto_extended import PasetoLoginManager\n\napp = flask.Flask(__name__)\napp.secret_key = "super secret string"\n\nlogin_manager = PasetoLoginManager(app)\n```\n\nSee [examples/login_manager.py](https://github.com/dajiaji/flask-paseto-extended/blob/main/examples/login_manager.py) for a sample code that actually works.\n\n## API Reference\n\nSee [Documentation](https://flask-paseto-extended.readthedocs.io/en/stable/api.html).\n\n\n## Tests\n\nYou can run tests from the project root after cloning with:\n\n```sh\n$ tox\n```\n\n## Contributing\n\nWe welcome all kind of contributions, filing issues, suggesting new features or sending PRs.\n',
    'author': 'Ajitomi Daisuke',
    'author_email': 'dajiaji@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dajiaji/flask-paseto-extended',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
