## BIP-0039 library

The BIP-0039 library allows to:
- Generate mnemonics from words number or entropy bytes
- Validate a mnemonic
- Get back the entropy bytes from a mnemonic
- Generate the seed from a mnemonic

### Mnemonic generation

A mnemonic phrase can be generated by specifying the words number (in this case a random entropy will be used) or directly by the entropy bytes.

Supported words number:

|Words number|Enum|
|---|---|
|12|*Bip39WordsNum.WORDS_NUM_12*|
|15|*Bip39WordsNum.WORDS_NUM_15*|
|18|*Bip39WordsNum.WORDS_NUM_18*|
|21|*Bip39WordsNum.WORDS_NUM_21*|
|24|*Bip39WordsNum.WORDS_NUM_24*|

Supported entropy bits:

|Entropy bits|Enum|
|---|---|
|128|*Bip39EntropyBitLen.BIT_LEN_128*|
|160|*Bip39EntropyBitLen.BIT_LEN_160*|
|192|*Bip39EntropyBitLen.BIT_LEN_192*|
|224|*Bip39EntropyBitLen.BIT_LEN_224*|
|256|*Bip39EntropyBitLen.BIT_LEN_256*|

Supported languages:

|Language|Enum|
|---|---|
|Chinese (simplified)|*Bip39Languages.CHINESE_SIMPLIFIED*|
|Chinese (traditional)|*Bip39Languages.CHINESE_TRADITIONAL*|
|Czech|*Bip39Languages.CZECH*|
|English|*Bip39Languages.ENGLISH*|
|French|*Bip39Languages.FRENCH*|
|Italian|*Bip39Languages.ITALIAN*|
|Korean|*Bip39Languages.KOREAN*|
|Portuguese|*Bip39Languages.PORTUGUESE*|
|Spanish|*Bip39Languages.SPANISH*|

**Code example**

    import binascii
    from bip_utils import (
        Bip39EntropyBitLen, Bip39EntropyGenerator, Bip39WordsNum, Bip39Languages, Bip39MnemonicGenerator, Bip39MnemonicEncoder
    )
    
    # Generate a random mnemonic string of 12 words with default language (English)
    # A Mnemonic object will be returned
    mnemonic = Bip39MnemonicGenerator().FromWordsNumber(Bip39WordsNum.WORDS_NUM_12)
    
    # Get words count
    print(mnemonic.WordsCount())
    # Get as string
    print(mnemonic.ToStr())
    print(str(mnemonic))
    # Get as list of strings
    print(mnemonic.ToList())
    
    # Generate a random mnemonic string of 15 words by specifying the language
    mnemonic = Bip39MnemonicGenerator(Bip39Languages.ITALIAN).FromWordsNumber(Bip39WordsNum.WORDS_NUM_15)
    
    # Generate the mnemonic string from entropy bytes
    entropy_bytes = binascii.unhexlify(b"00000000000000000000000000000000")
    mnemonic = Bip39MnemonicGenerator().FromEntropy(entropy_bytes)
    mnemonic = Bip39MnemonicGenerator(Bip39Languages.FRENCH).FromEntropy(entropy_bytes)
    
    # Generate mnemonic from random 192-bit entropy
    entropy_bytes = Bip39EntropyGenerator(Bip39EntropyBitLen.BIT_LEN_192).Generate()
    mnemonic = Bip39MnemonicGenerator().FromEntropy(entropy_bytes)
    
    # Alternatively, the mnemonic can be generated from entropy using the encoder
    mnemonic = Bip39MnemonicEncoder(Bip39Languages.ENGLISH).Encode(entropy_bytes)

### Mnemonic validation

A mnemonic string can be validated by verifying its language and checksum. Moreover, it is also possible to get back the entropy bytes from a mnemonic.\
When validating, the language can be either specified or automatically detected.\
Automatic detection takes more time, so if the mnemonic language is known in advance it'll be better to specify it at construction.

**Code example**

    from bip_utils import (
        MnemonicChecksumError, Bip39Languages, Bip39WordsNum, Bip39Mnemonic,
        Bip39MnemonicGenerator, Bip39MnemonicValidator, Bip39MnemonicDecoder
    )

    # Mnemonic can be generated with Bip39MnemonicGenerator
    mnemonic = Bip39MnemonicGenerator().FromWordsNumber(Bip39WordsNum.WORDS_NUM_15)
    # Or it can be a string
    mnemonic = "abandon abandon abandon abandon abandon abandon abandon abandon abandon abandon abandon about"
    # Or from a list
    mnemonic = Bip39Mnemonic.FromList(mnemonic.split())


    # Get if a mnemonic is valid, return bool
    is_valid = Bip39MnemonicValidator().IsValid(mnemonic)
    # Validate a mnemonic, raise exceptions
    try:
        Bip39MnemonicValidator().Validate(mnemonic)
        # Valid...
    except MnemonicChecksumError:
        # Invalid checksum...
        pass
    except ValueError:
        # Invalid length or language...
        pass

    # Use Bip39MnemonicDecoder to get back the entropy bytes from a mnemonic, specifying the language
    entropy_bytes = Bip39MnemonicDecoder(Bip39Languages.ENGLISH).Decode(mnemonic)
    # Like before with automatic language detection
    entropy_bytes = Bip39MnemonicDecoder().Decode(mnemonic)

    # Alternatively, it's possible to get back the entropy bytes with the computed checksum
    entropy_chksum_bytes = Bip39MnemonicDecoder(Bip39Languages.ENGLISH).DecodeWithChecksum(mnemonic)

### Seed generation

A secure 64-byte seed is generated from a mnemonic and can be protected by a passphrase.\
This seed can be used to construct a Bip class using the *FromSeed* method (e.g. *Bip44.FromSeed*).\
Also in this case, the language can be specified or automatically detected.

**Code example**

    from bip_utils import Bip39Languages, Bip39WordsNum, Bip39MnemonicGenerator, Bip39SeedGenerator

    # Mnemonic can be generated with Bip39MnemonicGenerator
    mnemonic = Bip39MnemonicGenerator().FromWordsNumber(Bip39WordsNum.WORDS_NUM_24)
    # Or it can be a string
    mnemonic = "branka dorost klam slanina omezit cuketa kazeta cizost rozchod tvaroh majetek kyvadlo"

    # Generate with automatic language detection and passphrase (empty)
    seed_bytes = Bip39SeedGenerator(mnemonic).Generate()
    # Generate with automatic language detection and custom passphrase
    seed_bytes = Bip39SeedGenerator(mnemonic).Generate("my_passphrase")
    # Generate specifying the language
    seed_bytes = Bip39SeedGenerator(mnemonic, Bip39Languages.CZECH).Generate()

### Substrate seed generation

Polkadot introduced a variant for generating seed, which computes the seed directly from the mnemonic entropy instead of the mnemonic string.\
Reference: [Substrate seed generation](https://wiki.polkadot.network/docs/learn-accounts#seed-generation)\
For this purpose, the class *SubstrateBip39SeedGenerator* can be used, which has the same usage of *Bip39SeedGenerator*.\
The seed can be used to construct a *Substrate* class using the *Substrate.FromSeed* method.

    from bip_utils import Bip39Languages, Bip39WordsNum, Bip39MnemonicGenerator, SubstrateBip39SeedGenerator

    # Mnemonic can be generated with Bip39MnemonicGenerator
    mnemonic = Bip39MnemonicGenerator().FromWordsNumber(Bip39WordsNum.WORDS_NUM_12)
    # Or it can be a string
    mnemonic = "branka dorost klam slanina omezit cuketa kazeta cizost rozchod tvaroh majetek kyvadlo"

    # Generate with automatic language detection and passphrase (empty)
    seed_bytes = SubstrateBip39SeedGenerator(mnemonic).Generate()
    # Generate with automatic language detection and custom passphrase
    seed_bytes = SubstrateBip39SeedGenerator(mnemonic).Generate("my_passphrase")
    # Generate specifying the language
    seed_bytes = SubstrateBip39SeedGenerator(mnemonic, Bip39Languages.CZECH).Generate()

Please note that this is not used by all wallets supporting Polkadot. For example, TrustWallet or Ledger still use the standard BIP39 seed generation for Polkadot.
