"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: props.tableName ?? cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: props.sortStyle ?? this.getDefaultSortStyle(props.tableColumns),
                tableComment: props.tableComment,
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.65.0-alpha.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
//# sourceMappingURL=data:application/json;base64,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