"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployAssert = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const assertions_1 = require("../assertions");
const hash_1 = require("../private/hash");
const sdk_1 = require("../sdk");
const DEPLOY_ASSERT_SYMBOL = Symbol.for('@aws-cdk/integ-tests.DeployAssert');
/**
 * Construct that allows for registering a list of assertions
 * that should be performed on a construct
 */
class DeployAssert extends constructs_1.Construct {
    constructor(scope) {
        super(scope, 'Default');
        this.scope = new aws_cdk_lib_1.Stack(scope, 'DeployAssert');
        Object.defineProperty(this, DEPLOY_ASSERT_SYMBOL, { value: true });
    }
    /**
     * Returns whether the construct is a DeployAssert construct
     */
    static isDeployAssert(x) {
        return x !== null && typeof (x) === 'object' && DEPLOY_ASSERT_SYMBOL in x;
    }
    /**
     * Finds a DeployAssert construct in the given scope
     */
    static of(construct) {
        const scopes = constructs_1.Node.of(constructs_1.Node.of(construct).root).findAll();
        const deployAssert = scopes.find(s => DeployAssert.isDeployAssert(s));
        if (!deployAssert) {
            throw new Error('No DeployAssert construct found in scopes');
        }
        return deployAssert;
    }
    awsApiCall(service, api, parameters) {
        return new sdk_1.AwsApiCall(this.scope, `AwsApiCall${service}${api}`, {
            api,
            service,
            parameters,
        });
    }
    invokeFunction(props) {
        const hash = hash_1.md5hash(this.scope.resolve(props));
        return new sdk_1.LambdaInvokeFunction(this.scope, `LambdaInvoke${hash}`, props);
    }
    expect(id, expected, actual) {
        new assertions_1.EqualsAssertion(this.scope, `EqualsAssertion${id}`, {
            expected,
            actual,
        });
    }
}
exports.DeployAssert = DeployAssert;
//# sourceMappingURL=data:application/json;base64,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