"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 * @stability stable
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     * @stability stable
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack.
     *
     * @stability stable
     */
    static resourceServer(server, scope) {
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_IUserPoolResourceServer(server);
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_ResourceServerScope(scope);
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
}
exports.OAuthScope = OAuthScope;
_a = JSII_RTTI_SYMBOL_1;
OAuthScope[_a] = { fqn: "@aws-cdk/aws-cognito.OAuthScope", version: "1.134.0" };
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client.
 *
 * @stability stable
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
 *
 * @stability stable
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient.
 *
 * @stability stable
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     * @stability stable
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
_b = JSII_RTTI_SYMBOL_1;
UserPoolClientIdentityProvider[_b] = { fqn: "@aws-cdk/aws-cognito.UserPoolClientIdentityProvider", version: "1.134.0" };
/**
 * Allow users to sign in using 'Sign In With Apple'.
 *
 * A `UserPoolIdentityProviderApple` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.APPLE = new UserPoolClientIdentityProvider('SignInWithApple');
/**
 * Allow users to sign in using 'Facebook Login'.
 *
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 *
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 *
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client.
 *
 * @stability stable
 */
class UserPoolClient extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolClientProps(props);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_e = (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.flows) !== null && _e !== void 0 ? _e : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_f = props.oAuth) === null || _f === void 0 ? void 0 : _f.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_g = props.oAuth) === null || _g === void 0 ? void 0 : _g.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
            readAttributes: (_h = props.readAttributes) === null || _h === void 0 ? void 0 : _h.attributes(),
            writeAttributes: (_j = props.writeAttributes) === null || _j === void 0 ? void 0 : _j.attributes(),
            enableTokenRevocation: props.enableTokenRevocation,
        });
        this.configureTokenValidity(resource, props);
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     *
     * @stability stable
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     *
     * @stability stable
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _d;
        const scopes = (_d = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _d !== void 0 ? _d : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
    configureTokenValidity(resource, props) {
        this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.idTokenValidity);
        this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.accessTokenValidity);
        this.validateDuration('refreshTokenValidity', core_1.Duration.minutes(60), core_1.Duration.days(10 * 365), props.refreshTokenValidity);
        if (props.refreshTokenValidity) {
            this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.idTokenValidity);
            this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.accessTokenValidity);
        }
        if (props.accessTokenValidity || props.idTokenValidity || props.refreshTokenValidity) {
            resource.tokenValidityUnits = {
                idToken: props.idTokenValidity ? 'minutes' : undefined,
                accessToken: props.accessTokenValidity ? 'minutes' : undefined,
                refreshToken: props.refreshTokenValidity ? 'minutes' : undefined,
            };
        }
        ;
        resource.idTokenValidity = props.idTokenValidity ? props.idTokenValidity.toMinutes() : undefined;
        resource.refreshTokenValidity = props.refreshTokenValidity ? props.refreshTokenValidity.toMinutes() : undefined;
        resource.accessTokenValidity = props.accessTokenValidity ? props.accessTokenValidity.toMinutes() : undefined;
    }
    validateDuration(name, min, max, value) {
        if (value === undefined) {
            return;
        }
        if (value.toMilliseconds() < min.toMilliseconds() || value.toMilliseconds() > max.toMilliseconds()) {
            throw new Error(`${name}: Must be a duration between ${min.toHumanString()} and ${max.toHumanString()} (inclusive); received ${value.toHumanString()}.`);
        }
    }
}
exports.UserPoolClient = UserPoolClient;
_c = JSII_RTTI_SYMBOL_1;
UserPoolClient[_c] = { fqn: "@aws-cdk/aws-cognito.UserPoolClient", version: "1.134.0" };
//# sourceMappingURL=data:application/json;base64,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