"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeType = exports.ElasticacheAutoMonitor = void 0;
const path = require("path");
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sns_sub = require("@aws-cdk/aws-sns-subscriptions");
const actions = require("@aws-cdk/aws-cloudwatch-actions");
const cpu_monitor_1 = require("./cpu-monitor");
const swap_usage_monitor_1 = require("./swap-usage-monitor");
const evictions_monitor_1 = require("./evictions-monitor");
const current_connections_monitor_1 = require("./current-connections-monitor");
const freeable_memory_monitor_1 = require("./freeable-memory-monitor");
/**
 * ElasticacheAutoMonitor allows you to send email, sms, slack, or trigger aws sns topic when an alarm occurs.
 * You will get the following monitoring:
 *  1. Cpu Monitor: Should be less than 90%. (See below reference)
 *  2. SwapUsage Monitor: Should be less than 50M.
 *  3. Evictions Monitor: Should not have evictions value.
 *  4. CurrConnections Monitor: According to your business needs, default every 1 vcup is equal to 200 connections.
 *  5. FreeableMemory Monitor: Not less than 10%
 *
 * Reference: https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/CacheMetrics.WhichShouldIMonitor.html
 */
class ElasticacheAutoMonitor extends cdk.Construct {
    static setUpWithLambda(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-lambda');
        topic.addSubscription(new sns_sub.LambdaSubscription(props.lambda));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSlack(scope, cacheClusterId, props) {
        let channel = props.channel || 'cloudwatch-alarm';
        let username = props.username || 'Webhookbot';
        let icon_emoji = props.iconEmoji || ':scream:';
        const fn = new lambda.Function(scope, 'alarm-' + cacheClusterId + '-to-slack', {
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset(path.join(__dirname, 'asset/lambda')),
            environment: {
                SLACK_WEBHOOK_URL: props.webHookUrl,
                CHANNEL: channel,
                USERNAME: username,
                ICON_EMOJI: icon_emoji,
            },
        });
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-slack');
        topic.addSubscription(new sns_sub.LambdaSubscription(fn));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSms(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-with-sms-topic');
        for (let phone in props.phones) {
            topic.addSubscription(new sns_sub.SmsSubscription(phone));
        }
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setup(scope, cacheClusterId, topic, props) {
        // bind default and props
        const nodeType = props.nodeType || NodeType.DEFAULT;
        const _props = {
            nodeType,
            currConnectionsPeriod: cdk.Duration.minutes(1),
            currConnectionsThreshold: nodeType.vcupCount * 200,
            ...props,
        };
        cpu_monitor_1.CpuMonitor.setup(scope, cacheClusterId, _props.nodeType).addAlarmAction(new actions.SnsAction(topic));
        swap_usage_monitor_1.SwapUsageMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        evictions_monitor_1.EvictionsMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        current_connections_monitor_1.CurrConnectionsMomiton.setup(scope, cacheClusterId, _props.currConnectionsPeriod, _props.currConnectionsThreshold).addAlarmAction(new actions.SnsAction(topic));
        freeable_memory_monitor_1.FreeableMemoryMomiton.setup(scope, cacheClusterId, _props.nodeType).addAlarmAction(new actions.SnsAction(topic));
    }
}
exports.ElasticacheAutoMonitor = ElasticacheAutoMonitor;
class NodeType {
    constructor(name, vcupCount, memory) {
        this.name = name;
        this.vcupCount = vcupCount;
        this.memory = memory;
    }
}
exports.NodeType = NodeType;
NodeType.T2_MIRCO = new NodeType('cache.t2.mirco', 1, 0.555 * 1024 * 1024 * 1024);
NodeType.T2_SMALL = new NodeType('cache.t2.small', 1, 1.55 * 1024 * 1024 * 1024);
NodeType.T2_MEDIUM = new NodeType('cache.t2.medium', 2, 3.22 * 1024 * 1024 * 1024);
NodeType.T3_MICRO = new NodeType('cache.t3.micro', 2, 0.5 * 1024 * 1024 * 1024);
NodeType.T3_SMALL = new NodeType('cache.t3.small', 2, 1.37 * 1024 * 1024 * 1024);
NodeType.T3_MEDIUM = new NodeType('cache.t3.medium', 2, 3.09 * 1024 * 1024 * 1024);
NodeType.M4_LARGE = new NodeType('cache.m4.large', 2, 6.42 * 1024 * 1024 * 1024);
NodeType.M4_XLARGE = new NodeType('cache.m4.xlarge', 4, 14.28 * 1024 * 1024 * 1024);
NodeType.M4_2XLARGE = new NodeType('cache.m4.2xlarge', 8, 29.70 * 1024 * 1024 * 1024);
NodeType.M4_4XLARGE = new NodeType('cache.m4.4xlarge', 16, 60.78 * 1024 * 1024 * 1024);
NodeType.M4_10XLARGE = new NodeType('cache.m4.10xlarge', 40, 154.64 * 1024 * 1024 * 1024);
NodeType.M5_LARGE = new NodeType('cache.m5.large', 2, 6.38 * 1024 * 1024 * 1024);
NodeType.M5_XLARGE = new NodeType('cache.m5.xlarge', 4, 12.93 * 1024 * 1024 * 1024);
NodeType.M5_2XLARGE = new NodeType('cache.m5.2xlarge', 8, 26.04 * 1024 * 1024 * 1024);
NodeType.M5_4XLARGE = new NodeType('cache.m5.4xlarge', 16, 52.26 * 1024 * 1024 * 1024);
NodeType.M5_12XLARGE = new NodeType('cache.m5.12xlarge', 48, 157.12 * 1024 * 1024 * 1024);
NodeType.M5_24XLARGE = new NodeType('cache.m5.12xlarge', 96, 314.32 * 1024 * 1024 * 1024);
NodeType.R4_LARGE = new NodeType('cache.r4.large', 2, 12.3 * 1024 * 1024 * 1024);
NodeType.R4_XLARGE = new NodeType('cache.r4.xlarge', 4, 25.05 * 1024 * 1024 * 1024);
NodeType.R4_2XLARGE = new NodeType('cache.r4.2xlarge', 8, 50.47 * 1024 * 1024 * 1024);
NodeType.R4_4XLARGE = new NodeType('cache.r4.4xlarge', 16, 101.38 * 1024 * 1024 * 1024);
NodeType.R4_8XLARGE = new NodeType('cache.r4.8xlarge', 32, 203.26 * 1024 * 1024 * 1024);
NodeType.R4_16XLARGE = new NodeType('cache.r4.16xlarge', 64, 407 * 1024 * 1024 * 1024);
NodeType.R5_LARGE = new NodeType('cache.r5.large', 2, 13.07 * 1024 * 1024 * 1024);
NodeType.R5_XLARGE = new NodeType('cache.r5.xlarge', 4, 26.32 * 1024 * 1024 * 1024);
NodeType.R5_2XLARGE = new NodeType('cache.r5.2xlarge', 8, 52.82 * 1024 * 1024 * 1024);
NodeType.R5_4XLARGE = new NodeType('cache.r5.4xlarge', 16, 105.81 * 1024 * 1024 * 1024);
NodeType.R5_12XLARGE = new NodeType('cache.r5.12xlarge', 48, 317.77 * 1024 * 1024 * 1024);
NodeType.R5_24LARGE = new NodeType('cache.r5.12xlarge', 96, 635.61 * 1024 * 1024 * 1024);
NodeType.DEFAULT = NodeType.M5_LARGE;
//# sourceMappingURL=data:application/json;base64,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