#!/usr/bin/env python
# coding: utf-8

from trivoreid.models.misc import Meta

class Subscription(object):
    '''
    Class that represents a subscription object.
    '''

    def __init__(self, data={}):
        '''
        Args:
            data (dict): subscription fields
        Dictionary keys:
            'id' (str) Readonly.
            'subscriberId' (str) subscriber User ID. Cannot be changed after
                                 Subscription creation.
            'nsCode' (str) namespace code. Value is autogenerated based on
                           subscriber. Readonly.
            'externalId' (str) external ID for subscription. For example,
                               Order ID.
            'type' (str) subscription type
            'customFields' (dict)
            'periods' (list)
            'terminated' (str) subscription termination time.
            'active' (boolean) is subscription active. Depends on termination
                               time.
            'meta' (dict) meta data
        '''
        self.id = data.pop('id', None)
        self.subscriberId = data.pop('subscriberId', None)
        self.nsCode = data.pop('nsCode', None)
        self.externalId = data.pop('externalId', None)
        self.type = data.pop('type', None)
        self.customFields = data.pop('customFields', {})
        self.terminated = data.pop('terminated', None)
        self.active = data.pop('active', None)
        self.meta = Meta(data.pop('meta', {}))

        self.periods = []
        for p in data.pop('periods', []):
            self.periods.append(Period(p))

    def serialize(self):
        '''
        Return JSON serializable object.
        '''
        periods = []
        for p in self.periods:
            periods.append(p.serialize())

        return {
            'id'            : self.id,
            'subscriberId'  : self.subscriberId,
            'nsCode'        : self.nsCode,
            'externalId'    : self.externalId,
            'type'          : self.type,
            'customFields'  : self.customFields,
            'terminated'    : self.terminated,
            'active'        : self.active,
            'periods'       : periods,
            'meta'          : self.meta.serialize()
        }

class Period(object):
    '''
    Class that represents a subscription's period object.
    '''

    def __init__(self, data={}):
        '''
        Args:
            data (dict): period fields
        Dictionary keys:
            'id' (str) Readonly.
            'created' (str) Time when period info was added.
            'externalId' (str) External ID for period.
            'from' (str) Period start time.
            'until' (str) Period end time.
            'customFields' (dict)
        '''
        self.id = data.pop('id', None)
        self.created = data.pop('created', None)
        self.externalId = data.pop('externalId', None)
        self.startTime = data.pop('from', None)
        self.until = data.pop('until', None)
        self.customFields = data.pop('customFields', {})

    def serialize(self):
        '''
        Return JSON serializable object.
        '''
        return {
            'id'            : self.id,
            'created'       : self.created,
            'externalId'    : self.externalId,
            'from'          : self.startTime,
            'until'         : self.until,
            'customFields'  : self.customFields
        }
