from io import StringIO

import pytest

from protfetch.processor import ProcessedProtein, process_fasta_stream

MOCK_FASTA_CONTENT_PROCESSOR = """>sp|P01111|RAS_HUMAN HRAS proto-oncogene OS=Homo sapiens OX=9606 GN=HRAS PE=1 SV=1
MTEYKLVVVGAGGVGKSALTIQLIQNHFVDEYDPTIEDSYRKQVVIDGETCLLDILDTAGQEEYSAMRDQYMRTGEGFLCVFAINNTKSFEDIHQYREQIKRVKDSDDVPMVLVGNKCDLAARTVESRQAQDLARSYGIPYIETSAKTRQGVEDAFYTLVREIRQHKLRKLNPPDESGPGCMNCKCVLS
>tr|A0A023|IDENTICAL_TO_P01111 Identical sequence to P01111 OS=Homo sapiens
MTEYKLVVVGAGGVGKSALTIQLIQNHFVDEYDPTIEDSYRKQVVIDGETCLLDILDTAGQEEYSAMRDQYMRTGEGFLCVFAINNTKSFEDIHQYREQIKRVKDSDDVPMVLVGNKCDLAARTVESRQAQDLARSYGIPYIETSAKTRQGVEDAFYTLVREIRQHKLRKLNPPDESGPGCMNCKCVLS
>sp|P01112|RAS_MOUSE HRAS proto-oncogene, GTPase OS=Mus musculus OX=10090 GN=Hras PE=1 SV=1
MTEYKLVVVGAGGVGKSALTIQLIQNHFVDEYDPTIEDSYRKQVVIDGETCLLDILDTAGQEEYSAMRDQYMRTGEGFLCVFAINNTKSFEDIHQYREQIKRVKDSDDVPMVLVGNKCDLPSRTVESRQAQDLARSYGIPYIETSAKTRQGVEDAFYTLVREIRQHKLRKLNPPDESGPGCMNCKCVLS
>my|FRAG01|FRAGMENT_OF_P01111 A fragment of P01111 OS=Homo sapiens
SAMRDQYMRTGEGFLCVFAINNTKSFEDIHQYREQIKRVKDSDDVPMVLVGNKCDLAARTVESRQAQDLARSYGIPYIETSAKTRQGVEDAFYTLVREIRQH
>my|UNRELATED|UNRELATED_SEQ Completely different sequence OS=Bacillus subtilis
MKLSIVAVAGAGNVGKSSIVNIQEIHYSNTRNFIDKYDARTVKNLIVGETLLLDVLDTAGQEEYAEIRNWYITSKGFLCVYSVNSTKSFDELLTKYKELIKKKSKTPVFIIGNKIDLRENDIRKFLIAYNPDEKLKGLKILK
"""


def test_process_fasta_stream_filters():
    fasta_stream = StringIO(MOCK_FASTA_CONTENT_PROCESSOR)
    gene_symbol = "TESTGENE"
    max_dist = 2

    proteins, stats = process_fasta_stream(fasta_stream, gene_symbol, max_dist)
    accessions_kept = {p.accession for p in proteins}

    assert "A0A023" in accessions_kept
    assert "P01111" not in accessions_kept
    assert "P01112" not in accessions_kept
    assert "FRAG01" not in accessions_kept
    assert "my|UNRELATED|UNRELATED_SEQ" in accessions_kept

    assert stats["initial_unique_sequences_parsed"] == 5
    assert stats["removed_identical_sequences"] == 1
    assert stats["removed_near_identical_sequences"] == 1
    assert stats["removed_fragment_sequences"] == 1
    assert stats["final_sequences_kept"] == 2
