"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
const crypto = require("crypto");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tag.add(this, tagKey, tagValue);
        constructs.forEach(construct => core_1.Tag.add(construct, tagKey, tagValue));
        return result;
    }
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tag.add(this, tagKey, tagValue);
        constructs.forEach(construct => core_1.Tag.add(construct, tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(construct.node.uniqueId));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 */
class Volume extends VolumeBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_b = props.enableMultiAttach) !== null && _b !== void 0 ? _b : false,
            size: (_c = props.size) === null || _c === void 0 ? void 0 : _c.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_e = grant.principalStatement) === null || _e === void 0 ? void 0 : _e.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    validateProps(props) {
        var _a;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.iops) {
            if (props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
            }
            if (props.iops < 100 || props.iops > 64000) {
                throw new Error('`iops` must be in the range 100 to 64,000, inclusive.');
            }
            if (props.size && (props.iops > 50 * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error('`iops` has a maximum ratio of 50 IOPS/GiB.');
            }
        }
        if (props.enableMultiAttach && props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
            throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce maximum volume size:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1000 };
            const volumeType = (_a = props.volumeType) !== null && _a !== void 0 ? _a : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
//# sourceMappingURL=data:application/json;base64,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