"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHostLinux = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
/**
 * This creates a linux bastion host you can use to connect to other instances or services in your VPC.
 * The recommended way to connect to the bastion host is by using AWS Systems Manager Session Manager.
 *
 * The operating system is Amazon Linux 2 with the latest SSM agent installed
 *
 * You can also configure this bastion host to allow connections via SSH
 *
 * @experimental
 */
class BastionHostLinux extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.instance = new instance_1.Instance(this, 'Resource', {
            vpc: props.vpc,
            availabilityZone: props.availabilityZone,
            securityGroup: props.securityGroup,
            instanceName: (_a = props.instanceName) !== null && _a !== void 0 ? _a : 'BastionHost',
            instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : _1.InstanceType.of(_1.InstanceClass.T3, _1.InstanceSize.NANO),
            machineImage: (_c = props.machineImage) !== null && _c !== void 0 ? _c : machine_image_1.MachineImage.latestAmazonLinux({ generation: _1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            vpcSubnets: (_d = props.subnetSelection) !== null && _d !== void 0 ? _d : {},
            blockDevices: (_e = props.blockDevices) !== null && _e !== void 0 ? _e : undefined,
        });
        this.instance.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssmmessages:*',
                'ssm:UpdateInstanceInformation',
                'ec2messages:*',
            ],
            resources: ['*'],
        }));
        this.instance.addUserData('yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm');
        this.connections = this.instance.connections;
        this.role = this.instance.role;
        this.grantPrincipal = this.instance.role;
        this.instanceId = this.instance.instanceId;
        this.instancePrivateIp = this.instance.instancePrivateIp;
        this.instanceAvailabilityZone = this.instance.instanceAvailabilityZone;
        this.instancePrivateDnsName = this.instance.instancePrivateDnsName;
        this.instancePublicIp = this.instance.instancePublicIp;
        this.instancePublicDnsName = this.instance.instancePublicDnsName;
        new core_1.CfnOutput(this, 'BastionHostId', {
            description: 'Instance ID of the bastion host. Use this to connect via SSM Session Manager',
            value: this.instanceId,
        });
    }
    /**
     * Allow SSH access from the given peer or peers
     *
     * Necessary if you want to connect to the instance using ssh. If not
     * called, you should use SSM Session Manager to connect to the instance.
     */
    allowSshAccessFrom(...peer) {
        peer.forEach(p => {
            this.connections.allowFrom(p, port_1.Port.tcp(22), 'SSH access');
        });
    }
}
exports.BastionHostLinux = BastionHostLinux;
//# sourceMappingURL=data:application/json;base64,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