"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SummarizationAppsyncStepfn = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfnTask = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const eventBridge = require("../../../common/helpers/eventbridge-helper");
const redisHelper = require("../../../common/helpers/redis-helper");
const s3BucketHelper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpcHelper = require("../../../common/helpers/vpc-helper");
class SummarizationAppsyncStepfn extends constructs_1.Construct {
    /**
       * @summary Constructs a new instance of the SummarizationAppsyncStepfn class.
       * @param {Construct} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {SummarizationAppsyncStepfnProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        // observability
        let lambda_tracing = lambda.Tracing.ACTIVE;
        let enable_xray = true;
        let api_log_config = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability == false) {
            enable_xray = false;
            lambda_tracing = lambda.Tracing.DISABLED;
            api_log_config = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        ;
        // vpc
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'summarizationConstructLogGroup');
        const vpcFlowLogrole = new iam.Role(this, 'summarizationConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, vpcFlowLogrole),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // bucket for input document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingInputAssetsBucketObj) {
            this.inputAssetBucket = props.existingInputAssetsBucketObj;
        }
        else if (props?.bucketInputsAssetsProps) {
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + stage, props.bucketInputsAssetsProps);
        }
        else {
            const bucketName = 'input-assets-summary-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID;
            this.inputAssetBucket = new s3.Bucket(this, 'inputAssetsSummaryBucket' + stage, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // bucket for transformed document
        s3BucketHelper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingProcessedAssetsBucketObj) {
            this.processedAssetBucket = props.existingProcessedAssetsBucketObj;
        }
        else if (props?.bucketProcessedAssetsProps) {
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + stage, props.bucketProcessedAssetsProps);
        }
        else {
            const bucketName = 'processed-assets-summary-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID;
            this.processedAssetBucket = new s3.Bucket(this, 'processedAssetsSummaryBucket' + stage, {
                blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                encryption: s3.BucketEncryption.S3_MANAGED,
                bucketName: bucketName,
                serverAccessLogsBucket: serverAccessLogBucket,
                enforceSSL: true,
                versioned: true,
                lifecycleRules: [{
                        expiration: aws_cdk_lib_1.Duration.days(90),
                    }],
            });
        }
        // set up redis cluster
        redisHelper.CheckRedisClusterProps(props);
        // build redis cluster only when cfnCacheClusterProps is set
        if (props?.cfnCacheClusterProps) {
            const redisSecurityGroup = redisHelper.getRedisSecurityGroup(this, {
                existingVpc: this.vpc,
            });
            const redisProps = {
                existingVpc: this.vpc,
                cfnCacheClusterProps: props.cfnCacheClusterProps,
                subnetIds: vpcHelper.getPrivateSubnetIDs(this.vpc),
                inboundSecurityGroup: this.securityGroup,
                redisSecurityGroup: redisSecurityGroup,
                redisPort: 8686,
            };
            this.redisCluster = redisHelper.buildRedisCluster(this, redisProps);
            redisHelper.setInboundRules(redisSecurityGroup, this.securityGroup, redisProps.redisPort);
        }
        else {
            this.redisCluster = props?.existingRedisCulster;
        }
        const redisHost = this.redisCluster?.attrRedisEndpointAddress;
        const redisPort = this.redisCluster?.attrRedisEndpointPort;
        eventBridge.CheckEventBridgeProps(props);
        // Create event bridge
        this.eventBridgeBus = eventBridge.buildEventBus(this, {
            existingEventBusInterface: props.existingBusInterface,
            eventBusProps: props.eventBusProps,
        });
        // make it generic for other auth config as well
        const authorizationConfig = {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.USER_POOL,
                userPoolConfig: { userPool: props.cognitoUserPool },
            },
            additionalAuthorizationModes: [
                {
                    authorizationType: appsync.AuthorizationType.IAM,
                },
            ],
        };
        const apiName = props.summaryApiName || 'summaryApi';
        // graphql api for summary. client invoke this api with given schema and cognito user pool auth.
        const summarizationGraphqlApi = new appsync.GraphqlApi(this, 'summarizationGraphqlApi' + stage, {
            name: apiName + stage,
            logConfig: api_log_config,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-summarization-appsync-stepfn/schema.graphql')),
            authorizationConfig: authorizationConfig,
            xrayEnabled: enable_xray,
        });
        this.graphqlApi = summarizationGraphqlApi;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? summarizationGraphqlApi.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? summarizationGraphqlApi.apiId : props.existingMergedApi.attrApiId;
        // bucket
        const transformedAssetBucketName = this.processedAssetBucket.bucketName;
        const inputAssetBucketName = this.inputAssetBucket.bucketName;
        const isFileTransformationRequired = props?.isFileTransformationRequired || 'false';
        // role
        const inputvalidatorLambdaRole = new iam.Role(this, 'inputvalidatorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        inputvalidatorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                'appsync:GraphQL'],
            resources: ['arn:aws:s3:::' + inputAssetBucketName + '/*',
                'arn:aws:s3:::' + transformedAssetBucketName + '/*',
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(inputvalidatorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        // Lambda function to validate Input
        const inputValidatorLambda = new lambda.DockerImageFunction(this, 'inputValidatorLambda' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/input_validator')),
            functionName: 'summary_input_validator' + stage,
            description: 'Lambda function to validate input for summary api',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 1),
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: inputvalidatorLambdaRole,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        // role
        const documentReaderLambdaRole = new iam.Role(this, 'documentReaderLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        documentReaderLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL'],
            resources: ['arn:aws:s3:::' + inputAssetBucketName + '/*',
                'arn:aws:s3:::' + transformedAssetBucketName + '/*',
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(documentReaderLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used',
            },
        ], true);
        const documentReaderLambda = new lambda.DockerImageFunction(this, 'documentReaderLambda' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/document_reader')),
            functionName: 'summary_document_reader' + stage,
            description: 'Lambda function to read the input transformed document',
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 1),
            tracing: lambda_tracing,
            timeout: aws_cdk_lib_1.Duration.minutes(5),
            role: documentReaderLambdaRole,
            environment: {
                REDIS_HOST: redisHost,
                REDIS_PORT: redisPort,
                TRANSFORMED_ASSET_BUCKET: transformedAssetBucketName,
                INPUT_ASSET_BUCKET: inputAssetBucketName,
                IS_FILE_TRANSFORMED: isFileTransformationRequired,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        const summaryChainType = props?.summaryChainType || 'stuff';
        // role
        const summaryGeneratorLambdaRole = new iam.Role(this, 'summaryGeneratorLambdaRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        summaryGeneratorLambdaRole.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject',
                's3:GetBucketLocation',
                's3:ListBucket',
                's3:PutObject',
                'appsync:GraphQL',
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream'],
            resources: ['arn:aws:s3:::' + inputAssetBucketName + '/*',
                'arn:aws:s3:::' + transformedAssetBucketName + '/*',
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*'],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(summaryGeneratorLambdaRole, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const generateSummarylambda = new lambda.DockerImageFunction(this, 'generateSummarylambda' + stage, {
            functionName: 'summary_generator' + stage,
            description: 'Lambda function to generate the summary',
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-summarization-appsync-stepfn/summary_generator')),
            vpc: this.vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(10),
            tracing: lambda_tracing,
            role: summaryGeneratorLambdaRole,
            environment: {
                REDIS_HOST: redisHost,
                REDIS_PORT: redisPort,
                ASSET_BUCKET_NAME: transformedAssetBucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                SUMMARY_LLM_CHAIN_TYPE: summaryChainType,
            },
        });
        this.inputAssetBucket?.grantRead(generateSummarylambda);
        this.processedAssetBucket?.grantReadWrite(generateSummarylambda);
        this.inputAssetBucket?.grantRead(documentReaderLambda);
        this.processedAssetBucket?.grantReadWrite(documentReaderLambda);
        const enableOperationalMetric = props.enableOperationalMetric !== undefined && props.enableOperationalMetric !== null ? props.enableOperationalMetric : true;
        if (enableOperationalMetric) {
            const solutionId = `genai_cdk_${utils_1.version}/${this.constructor.name}/${id}`;
            documentReaderLambda.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
            generateSummarylambda.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
            inputValidatorLambda.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
        }
        ;
        // create datasource at appsync
        const SummaryStatusDataSource = new appsync.NoneDataSource(this, 'noneDataSource' + stage, {
            api: summarizationGraphqlApi,
            name: 'SummaryStatusDataSource',
        });
        SummaryStatusDataSource.createResolver('summaryResponseresolver' + stage, {
            typeName: 'Mutation',
            fieldName: 'updateSummaryJobStatus',
            requestMappingTemplate: appsync.MappingTemplate.fromString('{ "version": "2017-02-28", "payload": $util.toJson($context.args) }'),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        // step functions
        const inputValidationTask = new sfnTask.LambdaInvoke(this, 'Validate Input ', {
            lambdaFunction: inputValidatorLambda,
            resultPath: '$.validation_result',
        });
        const documentReaderTask = new sfnTask.LambdaInvoke(this, 'Read document and check summary in cache ', {
            lambdaFunction: documentReaderLambda,
            resultPath: '$.document_result',
        });
        const generateSummaryTask = new sfnTask.LambdaInvoke(this, 'Generate Summary with llm', {
            lambdaFunction: generateSummarylambda,
            resultPath: '$.summary_result',
        });
        const dlq = new sqs.Queue(this, 'dlq', {
            queueName: 'summarydlq' + stage,
            retentionPeriod: aws_cdk_lib_1.Duration.days(7),
            enforceSSL: true,
        });
        const jobFailed = new sfn.Fail(this, 'Failed', {
            comment: 'AWS summary Job failed',
        });
        const jobSuccess = new sfn.Succeed(this, 'succeeded', {
            comment: 'AWS summary Job succeeded',
        });
        // step function choice steps
        const validateInputChoice = new sfn.Choice(this, 'is Valid Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const summaryfromCacheChoice = new sfn.Choice(this, 'is Summary in Cache?', {});
        const fileStatusForSummarization = new sfn.Choice(this, 'is file status ready for summarization?', {
            outputPath: '$.document_result.Payload',
        });
        // step function, run files in parallel
        const runFilesInparallel = new sfn.Map(this, 'Run Files in Parallel', {
            maxConcurrency: 100,
        }).iterator(documentReaderTask.next(summaryfromCacheChoice
            .when(sfn.Condition.booleanEquals('$.document_result.Payload.is_summary_available', true), jobSuccess).otherwise(fileStatusForSummarization.when(sfn.Condition.stringMatches('$.document_result.Payload.status', 'Error'), jobSuccess).otherwise(generateSummaryTask.next(jobSuccess)))));
        const maxLogGroupNameLength = 255;
        const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
        const maxGeneratedNameLength = maxLogGroupNameLength - logGroupPrefix.length;
        const nameParts = [
            aws_cdk_lib_1.Stack.of(scope).stackName,
            scope.node.id,
            'StateMachineLogSummarization', // Literal string for log group name portion
        ];
        const logGroupName = (0, utils_1.generatePhysicalName)(logGroupPrefix, nameParts, maxGeneratedNameLength);
        const summarizationLogGroup = new logs.LogGroup(this, 'summarizationLogGroup', {
            logGroupName: logGroupName,
        });
        // step function definition
        const definition = inputValidationTask.next(validateInputChoice
            .when(sfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed)
            .otherwise(runFilesInparallel));
        // step function
        const summarizationStepFunction = new sfn.StateMachine(this, 'summarizationStepFunction', {
            definitionBody: sfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            logs: {
                destination: summarizationLogGroup,
                level: sfn.LogLevel.ALL,
            },
            tracingEnabled: enable_xray,
        });
        this.stateMachine = summarizationStepFunction;
        // event bridge datasource for summarization api
        const eventBridgeDataSource = summarizationGraphqlApi.addEventBridgeDataSource('eventBridgeDataSource', this.eventBridgeBus);
        this.eventBridgeBus.grantPutEventsTo(eventBridgeDataSource.grantPrincipal);
        // add resolver on summary graphql schema
        eventBridgeDataSource.createResolver('generateSummary', {
            typeName: 'Mutation',
            fieldName: 'generateSummary',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
         {
          "version": "2018-05-29",
          "operation": "PutEvents",
          "events": [{
              "source": "summary",
              "detail": $util.toJson($context.arguments),
              "detailType": "genAIdemo"
          }
          ]
  } 
  `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('#if($ctx.error)$utilerror($ctx.error.message, $ctx.error.type, $ctx.result) #end $util.toJson($ctx.result)'),
        });
        new events.Rule(this, 'SummaryMutationRule', {
            description: 'Summary Mutation Rule',
            eventBus: this.eventBridgeBus,
            eventPattern: {
                source: ['summary'],
            },
            targets: [
                new targets.SfnStateMachine(summarizationStepFunction, {
                    deadLetterQueue: dlq,
                    retryAttempts: 1,
                }),
            ],
        });
    }
}
exports.SummarizationAppsyncStepfn = SummarizationAppsyncStepfn;
_a = JSII_RTTI_SYMBOL_1;
SummarizationAppsyncStepfn[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.SummarizationAppsyncStepfn", version: "0.1.34" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvcGF0dGVybnMvZ2VuLWFpL2F3cy1zdW1tYXJpemF0aW9uLWFwcHN5bmMtc3RlcGZuL2luZGV4LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7QUFDSCw2QkFBNkI7QUFDN0IsNkNBQW1EO0FBQ25ELG1EQUFtRDtBQUVuRCwyQ0FBMkM7QUFFM0MsaURBQWlEO0FBQ2pELDBEQUEwRDtBQUMxRCwyQ0FBMkM7QUFDM0MsaURBQWlEO0FBQ2pELDZDQUE2QztBQUM3Qyx5Q0FBeUM7QUFDekMsMkNBQTJDO0FBQzNDLHFEQUFxRDtBQUNyRCwrREFBK0Q7QUFDL0QscUNBQTBDO0FBQzFDLDJDQUF1QztBQUN2QywwRUFBMEU7QUFDMUUsb0VBQW9FO0FBQ3BFLDJFQUEyRTtBQUMzRSx5REFBdUc7QUFDdkcsZ0VBQWdFO0FBOEpoRSxNQUFhLDBCQUEyQixTQUFRLHNCQUFTO0lBc0N2RDs7Ozs7OztTQU9LO0lBQ0wsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQztRQUM5RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksS0FBSyxHQUFHLE1BQU0sQ0FBQztRQUNuQixJQUFJLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDaEIsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUM7U0FDckI7UUFFRCxnQkFBZ0I7UUFDaEIsSUFBSSxjQUFjLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDM0MsSUFBSSxXQUFXLEdBQUcsSUFBSSxDQUFDO1FBQ3ZCLElBQUksY0FBYyxHQUFHO1lBQ25CLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYSxDQUFDLEdBQUc7WUFDeEMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUztTQUN4QyxDQUFDO1FBQ0YsSUFBSSxLQUFLLENBQUMsYUFBYSxJQUFJLEtBQUssRUFBRTtZQUNoQyxXQUFXLEdBQUcsS0FBSyxDQUFDO1lBQ3BCLGNBQWMsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztZQUN6QyxjQUFjLEdBQUc7Z0JBQ2YsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhLENBQUMsSUFBSTtnQkFDekMsU0FBUyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUzthQUN4QyxDQUFDO1NBQ0g7UUFBQSxDQUFDO1FBRUYsTUFBTTtRQUNOLElBQUksS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUN0QixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7U0FDOUI7YUFBTTtZQUNMLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3JEO1FBQ0QsaUJBQWlCO1FBQ2pCLElBQUksS0FBSyxFQUFFLHFCQUFxQixFQUFFO1lBQ2hDLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1NBQ2xEO2FBQU07WUFDTCxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FDeEMsSUFBSSxFQUNKLGVBQWUsRUFDZjtnQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ2IsZ0JBQWdCLEVBQUUsSUFBSTthQUN2QixDQUNGLENBQUM7U0FDSDtRQUVELG1CQUFtQjtRQUNuQixNQUFNLFFBQVEsR0FBRyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGdDQUFnQyxDQUFDLENBQUM7UUFDM0UsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtZQUN0RSxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsNkJBQTZCLENBQUM7U0FDbkUsQ0FBQyxDQUFDO1FBRUgsZUFBZTtRQUNmLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQy9CLFlBQVksRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7WUFDdkQsV0FBVyxFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUUsY0FBYyxDQUFDO1NBQy9FLENBQUMsQ0FBQztRQUVILDJDQUEyQztRQUMzQyxNQUFNLHFCQUFxQixHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQzlDLHVCQUF1QixHQUFDLEtBQUssRUFDN0I7WUFDRSxpQkFBaUIsRUFBRSxFQUFFLENBQUMsaUJBQWlCLENBQUMsU0FBUztZQUNqRCxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7WUFDMUMsVUFBVSxFQUFFLElBQUk7WUFDaEIsU0FBUyxFQUFFLElBQUk7WUFDZixjQUFjLEVBQUUsQ0FBQztvQkFDZixVQUFVLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO2lCQUM5QixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBR0wsNEJBQTRCO1FBQzVCLGNBQWMsQ0FBQyxZQUFZLENBQUM7WUFDMUIsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLDRCQUE0QjtZQUNyRCxXQUFXLEVBQUUsS0FBSyxDQUFDLHVCQUF1QjtTQUMzQyxDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssRUFBRSw0QkFBNEIsRUFBRTtZQUN2QyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLDRCQUE0QixDQUFDO1NBQzVEO2FBQU0sSUFBSSxLQUFLLEVBQUUsdUJBQXVCLEVBQUU7WUFDekMsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQ3hDLDBCQUEwQixHQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsdUJBQXVCLENBQUMsQ0FBQztTQUNwRTthQUFNO1lBQ0wsTUFBTSxVQUFVLEdBQUUsNkJBQTZCLEdBQUMsS0FBSyxHQUFDLEdBQUcsR0FBQyxpQkFBRyxDQUFDLFVBQVUsQ0FBQztZQUN6RSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSwwQkFBMEIsR0FBQyxLQUFLLEVBQzFFO2dCQUNFLGlCQUFpQixFQUFFLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO2dCQUNqRCxVQUFVLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7Z0JBQzFDLFVBQVUsRUFBRSxVQUFVO2dCQUN0QixzQkFBc0IsRUFBRSxxQkFBcUI7Z0JBQzdDLFVBQVUsRUFBRSxJQUFJO2dCQUNoQixTQUFTLEVBQUUsSUFBSTtnQkFDZixjQUFjLEVBQUUsQ0FBQzt3QkFDZixVQUFVLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO3FCQUM5QixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1NBQ047UUFFRCxrQ0FBa0M7UUFDbEMsY0FBYyxDQUFDLFlBQVksQ0FBQztZQUMxQixpQkFBaUIsRUFBRSxLQUFLLENBQUMsZ0NBQWdDO1lBQ3pELFdBQVcsRUFBRSxLQUFLLENBQUMsMEJBQTBCO1NBQzlDLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxFQUFFLGdDQUFnQyxFQUFFO1lBQzNDLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxLQUFLLENBQUMsZ0NBQWdDLENBQUM7U0FDcEU7YUFBTSxJQUFJLEtBQUssRUFBRSwwQkFBMEIsRUFBRTtZQUM1QyxJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFDNUMsOEJBQThCLEdBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1NBQzNFO2FBQU07WUFDTCxNQUFNLFVBQVUsR0FBRSxpQ0FBaUMsR0FBQyxLQUFLLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxDQUFDO1lBRTdFLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLDhCQUE4QixHQUFDLEtBQUssRUFDbEY7Z0JBQ0UsaUJBQWlCLEVBQUUsRUFBRSxDQUFDLGlCQUFpQixDQUFDLFNBQVM7Z0JBQ2pELFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtnQkFDMUMsVUFBVSxFQUFFLFVBQVU7Z0JBQ3RCLHNCQUFzQixFQUFFLHFCQUFxQjtnQkFDN0MsVUFBVSxFQUFFLElBQUk7Z0JBQ2hCLFNBQVMsRUFBRSxJQUFJO2dCQUNmLGNBQWMsRUFBRSxDQUFDO3dCQUNmLFVBQVUsRUFBRSxzQkFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7cUJBQzlCLENBQUM7YUFDSCxDQUFDLENBQUM7U0FDTjtRQUdELHVCQUF1QjtRQUN2QixXQUFXLENBQUMsc0JBQXNCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFHMUMsNERBQTREO1FBQzVELElBQUksS0FBSyxFQUFFLG9CQUFvQixFQUFFO1lBQy9CLE1BQU0sa0JBQWtCLEdBQUUsV0FBVyxDQUFDLHFCQUFxQixDQUFDLElBQUksRUFBRTtnQkFDaEUsV0FBVyxFQUFFLElBQUksQ0FBQyxHQUFHO2FBQ3RCLENBQUMsQ0FBQztZQUNILE1BQU0sVUFBVSxHQUFHO2dCQUNqQixXQUFXLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ3JCLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7Z0JBQ2hELFNBQVMsRUFBRSxTQUFTLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztnQkFDbEQsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLGFBQWE7Z0JBQ3hDLGtCQUFrQixFQUFFLGtCQUFrQjtnQkFDdEMsU0FBUyxFQUFFLElBQUk7YUFDaEIsQ0FBQztZQUNGLElBQUksQ0FBQyxZQUFZLEdBQUcsV0FBVyxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztZQUNwRSxXQUFXLENBQUMsZUFBZSxDQUFDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxhQUFhLEVBQUUsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQzNGO2FBQU07WUFDTCxJQUFJLENBQUMsWUFBWSxHQUFFLEtBQUssRUFBRSxvQkFBcUIsQ0FBQztTQUNqRDtRQUVELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsd0JBQXlCLENBQUM7UUFDL0QsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLFlBQVksRUFBRSxxQkFBc0IsQ0FBQztRQUc1RCxXQUFXLENBQUMscUJBQXFCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDekMsc0JBQXNCO1FBQ3RCLElBQUksQ0FBQyxjQUFjLEdBQUcsV0FBVyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUU7WUFDcEQseUJBQXlCLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtZQUNyRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7U0FDbkMsQ0FBQyxDQUFDO1FBRUgsZ0RBQWdEO1FBQ2hELE1BQU0sbUJBQW1CLEdBQWdDO1lBQ3ZELG9CQUFvQixFQUFFO2dCQUNwQixpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUztnQkFDdEQsY0FBYyxFQUFFLEVBQUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxlQUFlLEVBQUU7YUFDcEQ7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUI7b0JBQ0UsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUc7aUJBQ2pEO2FBQ0Y7U0FDRixDQUFDO1FBRUYsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxZQUFZLENBQUM7UUFFckQsZ0dBQWdHO1FBQ2hHLE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLElBQUksRUFBRSx5QkFBeUIsR0FBQyxLQUFLLEVBQzFGO1lBQ0UsSUFBSSxFQUFFLE9BQU8sR0FBQyxLQUFLO1lBQ25CLFNBQVMsRUFBRSxjQUFjO1lBQ3pCLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FDckMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsOEVBQThFLENBQUMsQ0FDckc7WUFDRCxtQkFBbUIsRUFBRSxtQkFBbUI7WUFDeEMsV0FBVyxFQUFFLFdBQVc7U0FDekIsQ0FBQyxDQUFDO1FBQ0wsSUFBSSxDQUFDLFVBQVUsR0FBRSx1QkFBdUIsQ0FBQztRQUV6Qyx3REFBd0Q7UUFDeEQsZ0VBQWdFO1FBQ2hFLE1BQU0sd0JBQXdCLEdBQUcsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLHVCQUF1QixDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLGNBQWMsQ0FBQztRQUN4SSxNQUFNLGtCQUFrQixHQUFHLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUM7UUFFeEgsU0FBUztRQUNULE1BQU0sMEJBQTBCLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFVBQVUsQ0FBQztRQUN4RSxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUM7UUFDOUQsTUFBTSw0QkFBNEIsR0FBRyxLQUFLLEVBQUUsNEJBQTRCLElBQUksT0FBTyxDQUFDO1FBRXBGLE9BQU87UUFDUCxNQUFNLHdCQUF3QixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsMEJBQTBCLEVBQUU7WUFDOUUsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELGNBQWMsRUFBRTtnQkFDZCwrQkFBK0IsRUFBRSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7b0JBQ3RELFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDbkMsT0FBTyxFQUFFO2dDQUNQLHFCQUFxQjtnQ0FDckIsc0JBQXNCO2dDQUN0QixtQkFBbUI7NkJBQ3BCOzRCQUNELFNBQVMsRUFBRSxDQUFDLE9BQU8saUJBQUcsQ0FBQyxTQUFTLFNBQVMsaUJBQUcsQ0FBQyxNQUFNLElBQUksaUJBQUcsQ0FBQyxVQUFVLDBCQUEwQixDQUFDO3lCQUNqRyxDQUFDLENBQUM7aUJBQ0osQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsK0ZBQStGO1FBQy9GLHdCQUF3QixDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDM0QsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsNEJBQTRCO2dCQUM1Qiw0QkFBNEI7Z0JBQzVCLDhCQUE4QjtnQkFDOUIsZ0NBQWdDO2FBQ2pDO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGNBQWMsR0FBQyxpQkFBRyxDQUFDLE1BQU0sR0FBQyxHQUFHLEdBQUMsaUJBQUcsQ0FBQyxVQUFVLEdBQUMsTUFBTTthQUNwRDtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osdURBQXVEO1FBQ3ZELGlHQUFpRztRQUNqRyx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzNELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLCtCQUErQjthQUNoQztZQUNELFNBQVMsRUFBRTtnQkFDVCxHQUFHO2FBQ0o7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLHdCQUF3QixDQUFDLFdBQVcsQ0FDbEMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFLENBQUMsY0FBYztnQkFDdEIsc0JBQXNCO2dCQUN0QixlQUFlO2dCQUNmLGlCQUFpQixDQUFDO1lBRXBCLFNBQVMsRUFBRSxDQUFDLGVBQWUsR0FBRyxvQkFBb0IsR0FBRyxJQUFJO2dCQUN2RCxlQUFlLEdBQUcsMEJBQTBCLEdBQUUsSUFBSTtnQkFDbEQsa0JBQWtCLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLFFBQVEsR0FBQyxrQkFBa0IsR0FBRSxJQUFJLENBQUM7U0FDdEYsQ0FBQyxDQUNILENBQUM7UUFFRix5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyx3QkFBd0IsRUFDeEI7WUFDRTtnQkFDRSxFQUFFLEVBQUUsbUJBQW1CO2dCQUN2QixNQUFNLEVBQUUsc0NBQXNDO2FBQy9DO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUVGLG9DQUFvQztRQUNwQyxNQUFNLG9CQUFvQixHQUMxQixJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEdBQUMsS0FBSyxFQUMvRDtZQUNFLElBQUksRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxRUFBcUUsQ0FBQyxDQUFDO1lBQ3hJLFlBQVksRUFBRSx5QkFBeUIsR0FBQyxLQUFLO1lBQzdDLFdBQVcsRUFBRSxtREFBbUQ7WUFDaEUsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO1lBQ2IsT0FBTyxFQUFFLGNBQWM7WUFDdkIsVUFBVSxFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsbUJBQW1CLEVBQUU7WUFDOUQsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQztZQUNwQyxVQUFVLEVBQUUsSUFBQSwrQkFBdUIsRUFBQyxJQUFJLEVBQUUsSUFBSyxHQUFHLENBQUMsQ0FBQztZQUNwRCxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzVCLElBQUksRUFBRSx3QkFBd0I7WUFDOUIsV0FBVyxFQUFFO2dCQUNYLFdBQVcsRUFBRSx3QkFBd0I7YUFDdEM7U0FDRixDQUFDLENBQUM7UUFFTCxPQUFPO1FBQ1AsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLDBCQUEwQixFQUFFO1lBQzlFLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUMzRCxjQUFjLEVBQUU7Z0JBQ2QsK0JBQStCLEVBQUUsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUN0RCxVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7NEJBQ25DLE9BQU8sRUFBRTtnQ0FDUCxxQkFBcUI7Z0NBQ3JCLHNCQUFzQjtnQ0FDdEIsbUJBQW1COzZCQUNwQjs0QkFDRCxTQUFTLEVBQUUsQ0FBQyxPQUFPLGlCQUFHLENBQUMsU0FBUyxTQUFTLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSwwQkFBMEIsQ0FBQzt5QkFDakcsQ0FBQyxDQUFDO2lCQUNKLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILCtGQUErRjtRQUMvRix3QkFBd0IsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzNELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsT0FBTyxFQUFFO2dCQUNQLDRCQUE0QjtnQkFDNUIsNEJBQTRCO2dCQUM1Qiw4QkFBOEI7Z0JBQzlCLGdDQUFnQzthQUNqQztZQUNELFNBQVMsRUFBRTtnQkFDVCxjQUFjLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLE1BQU07YUFDcEQ7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLHVEQUF1RDtRQUN2RCxpR0FBaUc7UUFDakcsd0JBQXdCLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUMzRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCwrQkFBK0I7YUFDaEM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsR0FBRzthQUNKO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSix3QkFBd0IsQ0FBQyxXQUFXLENBQ2xDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0QixNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLGNBQWM7Z0JBQ3RCLHNCQUFzQjtnQkFDdEIsZUFBZTtnQkFDZixjQUFjO2dCQUNkLGlCQUFpQixDQUFDO1lBQ3BCLFNBQVMsRUFBRSxDQUFDLGVBQWUsR0FBRyxvQkFBb0IsR0FBRSxJQUFJO2dCQUN0RCxlQUFlLEdBQUcsMEJBQTBCLEdBQUUsSUFBSTtnQkFDbEQsa0JBQWtCLEdBQUMsaUJBQUcsQ0FBQyxNQUFNLEdBQUMsR0FBRyxHQUFDLGlCQUFHLENBQUMsVUFBVSxHQUFDLFFBQVEsR0FBQyxrQkFBa0IsR0FBRSxJQUFJLENBQUM7U0FDdEYsQ0FBQyxDQUNILENBQUM7UUFFRix5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyx3QkFBd0IsRUFDeEI7WUFDRTtnQkFDRSxFQUFFLEVBQUUsbUJBQW1CO2dCQUN2QixNQUFNLEVBQUUscUNBQXFDO2FBQzlDO1NBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUVGLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxNQUFNLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLHNCQUFzQixHQUFDLEtBQUssRUFBRTtZQUM5RixJQUFJLEVBQUUsTUFBTSxDQUFDLGVBQWUsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUscUVBQXFFLENBQUMsQ0FBQztZQUN4SSxZQUFZLEVBQUUseUJBQXlCLEdBQUMsS0FBSztZQUM3QyxXQUFXLEVBQUUsd0RBQXdEO1lBQ3JFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLFVBQVUsRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLG1CQUFtQixFQUFFO1lBQzlELGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7WUFDcEMsVUFBVSxFQUFFLElBQUEsK0JBQXVCLEVBQUMsSUFBSSxFQUFFLElBQUssR0FBRyxDQUFDLENBQUM7WUFDcEQsT0FBTyxFQUFFLGNBQWM7WUFDdkIsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUM1QixJQUFJLEVBQUUsd0JBQXdCO1lBQzlCLFdBQVcsRUFBRTtnQkFDWCxVQUFVLEVBQUUsU0FBUztnQkFDckIsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLHdCQUF3QixFQUFFLDBCQUEwQjtnQkFDcEQsa0JBQWtCLEVBQUUsb0JBQW9CO2dCQUN4QyxtQkFBbUIsRUFBRSw0QkFBNEI7Z0JBQ2pELFdBQVcsRUFBRSx3QkFBd0I7YUFFdEM7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLGdCQUFnQixHQUFHLEtBQUssRUFBRSxnQkFBZ0IsSUFBSSxPQUFPLENBQUM7UUFFNUQsT0FBTztRQUNQLE1BQU0sMEJBQTBCLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtZQUNsRixTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDM0QsY0FBYyxFQUFFO2dCQUNkLCtCQUErQixFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztvQkFDdEQsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDOzRCQUNuQyxPQUFPLEVBQUU7Z0NBQ1AscUJBQXFCO2dDQUNyQixzQkFBc0I7Z0NBQ3RCLG1CQUFtQjs2QkFDcEI7NEJBQ0QsU0FBUyxFQUFFLENBQUMsT0FBTyxpQkFBRyxDQUFDLFNBQVMsU0FBUyxpQkFBRyxDQUFDLE1BQU0sSUFBSSxpQkFBRyxDQUFDLFVBQVUsMEJBQTBCLENBQUM7eUJBQ2pHLENBQUMsQ0FBQztpQkFDSixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCwrRkFBK0Y7UUFDL0YsMEJBQTBCLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUM3RCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCw0QkFBNEI7Z0JBQzVCLDRCQUE0QjtnQkFDNUIsOEJBQThCO2dCQUM5QixnQ0FBZ0M7YUFDakM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsY0FBYyxHQUFDLGlCQUFHLENBQUMsTUFBTSxHQUFDLEdBQUcsR0FBQyxpQkFBRyxDQUFDLFVBQVUsR0FBQyxNQUFNO2FBQ3BEO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSix1REFBdUQ7UUFDdkQsaUdBQWlHO1FBQ2pHLDBCQUEwQixDQUFDLFdBQVcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDN0QsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUU7Z0JBQ1AsK0JBQStCO2FBQ2hDO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEdBQUc7YUFDSjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosMEJBQTBCLENBQUMsV0FBVyxDQUNwQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixPQUFPLEVBQUUsQ0FBQyxjQUFjO2dCQUN0QixzQkFBc0I7Z0JBQ3RCLGVBQWU7Z0JBQ2YsY0FBYztnQkFDZCxpQkFBaUI7Z0JBQ2pCLHFCQUFxQjtnQkFDckIsdUNBQXVDLENBQUM7WUFDMUMsU0FBUyxFQUFFLENBQUMsZUFBZSxHQUFHLG9CQUFvQixHQUFFLElBQUk7Z0JBQ3RELGVBQWUsR0FBRywwQkFBMEIsR0FBRSxJQUFJO2dCQUNsRCxrQkFBa0IsR0FBQyxpQkFBRyxDQUFDLE1BQU0sR0FBQyxHQUFHLEdBQUMsaUJBQUcsQ0FBQyxVQUFVLEdBQUMsUUFBUSxHQUFDLGtCQUFrQixHQUFFLElBQUk7Z0JBQ2xGLGtCQUFrQixHQUFDLGlCQUFHLENBQUMsTUFBTSxHQUFDLHNCQUFzQixDQUFDO1NBRXhELENBQUMsQ0FDSCxDQUFDO1FBRUYseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsMEJBQTBCLEVBQzFCO1lBQ0U7Z0JBQ0UsRUFBRSxFQUFFLG1CQUFtQjtnQkFDdkIsTUFBTSxFQUFFLHNDQUFzQzthQUMvQztTQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFFRixNQUFNLHFCQUFxQixHQUFHLElBQUksTUFBTSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSx1QkFBdUIsR0FBQyxLQUFLLEVBQUU7WUFDaEcsWUFBWSxFQUFFLG1CQUFtQixHQUFDLEtBQUs7WUFDdkMsV0FBVyxFQUFFLHlDQUF5QztZQUN0RCxJQUFJLEVBQUUsTUFBTSxDQUFDLGVBQWUsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsdUVBQXVFLENBQUMsQ0FBQztZQUMxSSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixVQUFVLEVBQUUsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsRUFBRTtZQUM5RCxjQUFjLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1lBQ3BDLFVBQVUsRUFBRSxJQUFBLCtCQUF1QixFQUFDLElBQUksRUFBRSxJQUFLLEdBQUcsQ0FBQyxDQUFDO1lBQ3BELE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsT0FBTyxFQUFFLGNBQWM7WUFDdkIsSUFBSSxFQUFFLDBCQUEwQjtZQUNoQyxXQUFXLEVBQUU7Z0JBQ1gsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixpQkFBaUIsRUFBRSwwQkFBMEI7Z0JBQzdDLFdBQVcsRUFBRSx3QkFBd0I7Z0JBQ3JDLHNCQUFzQixFQUFFLGdCQUFnQjthQUN6QztTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxTQUFTLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsb0JBQW9CLEVBQUUsY0FBYyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDakUsSUFBSSxDQUFDLGdCQUFnQixFQUFFLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxvQkFBb0IsRUFBRSxjQUFjLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUdoRSxNQUFNLHVCQUF1QixHQUMzQixLQUFLLENBQUMsdUJBQXVCLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsS0FBSyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBRS9ILElBQUksdUJBQXVCLEVBQUU7WUFDM0IsTUFBTSxVQUFVLEdBQUcsYUFBYSxlQUFPLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLElBQUksRUFBRSxFQUFFLENBQUM7WUFDekUsb0JBQW9CLENBQUMsY0FBYyxDQUNqQyxtQkFBbUIsRUFBRSxVQUFVLENBQ2hDLENBQUM7WUFDRixxQkFBcUIsQ0FBQyxjQUFjLENBQ2xDLG1CQUFtQixFQUFFLFVBQVUsQ0FDaEMsQ0FBQztZQUNGLG9CQUFvQixDQUFDLGNBQWMsQ0FDakMsbUJBQW1CLEVBQUUsVUFBVSxDQUNoQyxDQUFDO1NBQ0g7UUFBQSxDQUFDO1FBR0YsK0JBQStCO1FBQy9CLE1BQU0sdUJBQXVCLEdBQUcsSUFBSSxPQUFPLENBQUMsY0FBYyxDQUN6RCxJQUFJLEVBQUUsZ0JBQWdCLEdBQUMsS0FBSyxFQUFFO1lBQzdCLEdBQUcsRUFBRSx1QkFBdUI7WUFDNUIsSUFBSSxFQUFFLHlCQUF5QjtTQUNoQyxDQUFDLENBQUM7UUFFSCx1QkFBdUIsQ0FBQyxjQUFjLENBQ3JDLHlCQUF5QixHQUFDLEtBQUssRUFBRTtZQUNoQyxRQUFRLEVBQUUsVUFBVTtZQUNwQixTQUFTLEVBQUUsd0JBQXdCO1lBQ25DLHNCQUFzQixFQUFFLE9BQU8sQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUN4RCxxRUFBcUUsQ0FDdEU7WUFFRCx1QkFBdUIsRUFBRSxPQUFPLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FDekQsK0JBQStCLENBQUM7U0FDbkMsQ0FBQyxDQUFDO1FBRUgsaUJBQWlCO1FBQ2pCLE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxPQUFPLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUM1RSxjQUFjLEVBQUUsb0JBQW9CO1lBQ3BDLFVBQVUsRUFBRSxxQkFBcUI7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLE9BQU8sQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLDJDQUEyQyxFQUFFO1lBQ3JHLGNBQWMsRUFBRSxvQkFBb0I7WUFDcEMsVUFBVSxFQUFFLG1CQUFtQjtTQUNoQyxDQUFDLENBQUM7UUFHSCxNQUFNLG1CQUFtQixHQUFHLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLEVBQUU7WUFDdEYsY0FBYyxFQUFFLHFCQUFxQjtZQUNyQyxVQUFVLEVBQUUsa0JBQWtCO1NBQy9CLENBQUMsQ0FBQztRQUVILE1BQU0sR0FBRyxHQUFjLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO1lBQ2hELFNBQVMsRUFBRSxZQUFZLEdBQUMsS0FBSztZQUM3QixlQUFlLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ2pDLFVBQVUsRUFBRSxJQUFJO1NBQ2pCLENBQUMsQ0FBQztRQUdILE1BQU0sU0FBUyxHQUFFLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFO1lBQzVDLE9BQU8sRUFBRSx3QkFBd0I7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxVQUFVLEdBQUUsSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDbkQsT0FBTyxFQUFFLDJCQUEyQjtTQUNyQyxDQUFDLENBQUM7UUFFSCw2QkFBNkI7UUFDN0IsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ3ZFLFVBQVUsRUFBRSxtQ0FBbUM7U0FDaEQsQ0FBQyxDQUFDO1FBRUgsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFLEVBQzNFLENBQUMsQ0FBQztRQUVILE1BQU0sMEJBQTBCLEdBQUcsSUFBSSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSx5Q0FBeUMsRUFBRTtZQUNqRyxVQUFVLEVBQUUsMkJBQTJCO1NBQ3hDLENBQUMsQ0FBQztRQUdILHVDQUF1QztRQUN2QyxNQUFNLGtCQUFrQixHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7WUFDcEUsY0FBYyxFQUFFLEdBQUc7U0FDcEIsQ0FBQyxDQUFDLFFBQVEsQ0FDVCxrQkFBa0IsQ0FBQyxJQUFJLENBQ3JCLHNCQUFzQjthQUNuQixJQUFJLENBQ0gsR0FBRyxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsZ0RBQWdELEVBQUUsSUFBSSxDQUFDLEVBQ25GLFVBQVUsQ0FDWCxDQUFDLFNBQVMsQ0FDVCwwQkFBMEIsQ0FBQyxJQUFJLENBQzdCLEdBQUcsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLGtDQUFrQyxFQUFFLE9BQU8sQ0FBQyxFQUN4RSxVQUFVLENBQ1gsQ0FBQyxTQUFTLENBQ1QsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUNyQyxDQUNGLENBQ0osQ0FDRixDQUFDO1FBRUYsTUFBTSxxQkFBcUIsR0FBRyxHQUFHLENBQUM7UUFDbEMsTUFBTSxjQUFjLEdBQUcsb0NBQW9DLENBQUM7UUFDNUQsTUFBTSxzQkFBc0IsR0FBRyxxQkFBcUIsR0FBRyxjQUFjLENBQUMsTUFBTSxDQUFDO1FBQzdFLE1BQU0sU0FBUyxHQUFhO1lBQzFCLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVM7WUFDekIsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2IsOEJBQThCLEVBQUUsNENBQTRDO1NBQzdFLENBQUM7UUFDRixNQUFNLFlBQVksR0FBRyxJQUFBLDRCQUFvQixFQUFDLGNBQWMsRUFBRSxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztRQUM3RixNQUFNLHFCQUFxQixHQUFHLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7WUFDN0UsWUFBWSxFQUFFLFlBQVk7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsMkJBQTJCO1FBQzNCLE1BQU0sVUFBVSxHQUFHLG1CQUFtQixDQUFDLElBQUksQ0FDekMsbUJBQW1CO2FBQ2hCLElBQUksQ0FDSCxHQUFHLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxxQ0FBcUMsRUFBRSxLQUFLLENBQUMsRUFDekUsU0FBUyxDQUNWO2FBQ0EsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQ2pDLENBQUM7UUFFRixnQkFBZ0I7UUFFaEIsTUFBTSx5QkFBeUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO1lBQ3hGLGNBQWMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUM7WUFDNUQsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixJQUFJLEVBQUU7Z0JBQ0osV0FBVyxFQUFFLHFCQUFxQjtnQkFDbEMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsR0FBRzthQUN4QjtZQUNELGNBQWMsRUFBRSxXQUFXO1NBQzVCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxZQUFZLEdBQUMseUJBQXlCLENBQUM7UUFDNUMsZ0RBQWdEO1FBQ2hELE1BQU0scUJBQXFCLEdBQUcsdUJBQXVCLENBQUMsd0JBQXdCLENBQzVFLHVCQUF1QixFQUN2QixJQUFJLENBQUMsY0FBYyxDQUNwQixDQUFDO1FBRUYsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUUzRSx5Q0FBeUM7UUFDekMscUJBQXFCLENBQUMsY0FBYyxDQUFDLGlCQUFpQixFQUFFO1lBQ3RELFFBQVEsRUFBRSxVQUFVO1lBQ3BCLFNBQVMsRUFBRSxpQkFBaUI7WUFDNUIsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQ3hEOzs7Ozs7Ozs7OztHQVdMLENBQ0k7WUFFRCx1QkFBdUIsRUFBRSxPQUFPLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FDekQsNEdBQTRHLENBQzdHO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUMzQyxXQUFXLEVBQUUsdUJBQXVCO1lBQ3BDLFFBQVEsRUFBRSxJQUFJLENBQUMsY0FBYztZQUM3QixZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLENBQUMsU0FBUyxDQUFDO2FBQ3BCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQyx5QkFBeUIsRUFBRTtvQkFDckQsZUFBZSxFQUFFLEdBQUc7b0JBQ3BCLGFBQWEsRUFBRSxDQUFDO2lCQUNqQixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7SUFFTCxDQUFDOztBQTVyQkgsZ0VBNnJCQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgRHVyYXRpb24sIEF3cywgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBhcHBzeW5jIGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcHBzeW5jJztcbmltcG9ydCAqIGFzIGNvZ25pdG8gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNvZ25pdG8nO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWxhc3RpY2FjaGUgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVsYXN0aWNhY2hlJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZXZlbnRzJztcbmltcG9ydCAqIGFzIHRhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCAqIGFzIHNxcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3FzJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBzZm5UYXNrIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zLXRhc2tzJztcbmltcG9ydCB7IE5hZ1N1cHByZXNzaW9ucyB9IGZyb20gJ2Nkay1uYWcnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBldmVudEJyaWRnZSBmcm9tICcuLi8uLi8uLi9jb21tb24vaGVscGVycy9ldmVudGJyaWRnZS1oZWxwZXInO1xuaW1wb3J0ICogYXMgcmVkaXNIZWxwZXIgZnJvbSAnLi4vLi4vLi4vY29tbW9uL2hlbHBlcnMvcmVkaXMtaGVscGVyJztcbmltcG9ydCAqIGFzIHMzQnVja2V0SGVscGVyIGZyb20gJy4uLy4uLy4uL2NvbW1vbi9oZWxwZXJzL3MzLWJ1Y2tldC1oZWxwZXInO1xuaW1wb3J0IHsgZ2VuZXJhdGVQaHlzaWNhbE5hbWUsIHZlcnNpb24sIGxhbWJkYU1lbW9yeVNpemVMaW1pdGVyIH0gZnJvbSAnLi4vLi4vLi4vY29tbW9uL2hlbHBlcnMvdXRpbHMnO1xuaW1wb3J0ICogYXMgdnBjSGVscGVyIGZyb20gJy4uLy4uLy4uL2NvbW1vbi9oZWxwZXJzL3ZwYy1oZWxwZXInO1xuXG5leHBvcnQgaW50ZXJmYWNlIFN1bW1hcml6YXRpb25BcHBzeW5jU3RlcGZuUHJvcHMge1xuICAvKipcbiAgICogT3B0aW9uYWwuIFRoZSBjb25zdHJ1Y3QgY3JlYXRlcyBhIGN1c3RvbSBWUEMgYmFzZWQgb24gdnBjUHJvcHMuXG4gICAqIFByb3ZpZGluZyBib3RoIHRoaXMgYW5kIGV4aXN0aW5nVnBjIGlzIGFuIGVycm9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHZwY1Byb3BzPzogZWMyLlZwY1Byb3BzO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC4gQW4gZXhpc3RpbmcgVlBDIGNhbiBiZSB1c2VkIHRvIGRlcGxveSB0aGUgY29uc3RydWN0LlxuICAgKiBQcm92aWRpbmcgYm90aCB0aGlzIGFuZCB2cGNQcm9wcyBpcyBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ1ZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC4gRXhpc3RpbmcgUmVkaXMgY2x1c3RlciB0byBjYWNoZSB0aGUgZ2VuZXJhdGVkIHN1bW1hcnlcbiAgICogZm9yIHN1YnNlcXVlbnQgcmVxdWVzdCBvZiBzYW1lIGRvY3VtZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nUmVkaXNDdWxzdGVyPzogZWxhc3RpY2FjaGUuQ2ZuQ2FjaGVDbHVzdGVyO1xuXG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsLiBDdXN0b20gY2ZuQ2FjaGVDbHVzdGVyUHJvcHMgZm9yIFJlZGlzLlxuICAgKiBQcm92aWRpbmcgZXhpc3RpbmdSZWRpc0N1bHN0ZXIgYW5kIGNmbkNhY2hlQ2x1c3RlclByb3BzIHRvZ2V0aGVyIHdpbGwgcmVzdWx0IGluIGVycm9yLlxuICAgKiBAZGVmYXVsdCBjYWNoZU5vZGVUeXBlIC0gICdjYWNoZS5yNmcueGxhcmdlJ1xuICAgKiBAZGVmYXVsdCBudW1DYWNoZU5vZGVzLSAxXG4gICAqL1xuICByZWFkb25seSBjZm5DYWNoZUNsdXN0ZXJQcm9wcz86IGVsYXN0aWNhY2hlLkNmbkNhY2hlQ2x1c3RlclByb3BzO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC4gU2VjdXJpdHkgZ3JvdXAgZm9yIHRoZSBsYW1iZGEgZnVuY3Rpb24gd2hpY2ggdGhpcyBjb25zdHJ1Y3Qgd2lsbCB1c2UuXG4gICAqIElmIG5vIGV4aXNpdGluZyBzZWN1cml0eSBncm91cCBpcyBwcm92aWRlZCBpdCB3aWxsIGNyZWF0ZSBvbmUgZnJvbSB0aGUgdnBjLlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nU2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcblxuICAvKipcbiAgICogUmVxdWlyZWQuIENvZ25pdG8gdXNlciBwb29sIHVzZWQgZm9yIGF1dGhlbnRpY2F0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGNvZ25pdG9Vc2VyUG9vbDogY29nbml0by5JVXNlclBvb2w7XG4gIC8qKlxuICAgKiBPcHRpb25hbC4gRXhpc3RpbmcgczMgQnVja2V0IHRvIHN0b3JlIHRoZSBpbnB1dCBkb2N1bWVudCB3aGljaCBuZWVkcyB0byBiZSBzdW1tYXJpemVkLlxuICAgKiBwZGYgaXMgdGhlIHN1cHBvcnRlZCBpbnB1dCBkb2N1bWVudCBmb3JtYXQuIElmIHRyYW5zZm9ybWVkICh0eHQgZm9ybWF0KSBmaWxlIGlzXG4gICAqIGF2YWlsYWJsZSB0aGVuIHRoaXMgYnVja2V0IGlzIG9wdGlvbmFsLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nSW5wdXRBc3NldHNCdWNrZXRPYmo/OiBzMy5JQnVja2V0O1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC4gVXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIFMzIEJ1Y2tldC5cbiAgICogUHJvdmlkaW5nIGJvdGggdGhpcyBhbmQgYGV4aXN0aW5nSW5wdXRBc3NldHNCdWNrZXRPYmpgIHdpbGwgY2F1c2UgYW4gZXJyb3IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRGVmYXVsdCBwcm9wcyBhcmUgdXNlZFxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0SW5wdXRzQXNzZXRzUHJvcHM/OiBzMy5CdWNrZXRQcm9wcztcblxuICAvKipcbiAgICogT3B0aW9uYWwuIFRoZSBzdW1tYXJ5IGNvbnN0cnVjdCB0cmFuc2Zvcm0gdGhlIGlucHV0IGRvY3VtZW50IGludG8gdHh0IGZvcm1hdC4gSWYgdGhlXG4gICAqIHRyYW5zZm9ybWF0aW9uIGlzIG5vdCByZXF1aXJlZCB0aGVuIHRoaXMgZmxhZyBjYW4gYmUgc2V0IHRvIGZhbHNlLiBJZiBzZXQgdG8gdHJ1ZVxuICAgKiB0aGVuIGEgdHJhbnNmb3JtZWQgYXNzZXQgYnVja2V0IGlzIGNyZWF0ZWQgd2hpY2ggdHJhbnNmb3JtIHRoZSBpbnB1dCBkb2N1bWVudCBmcm9tXG4gICAqIGlucHV0IGFzc2V0IGJ1Y2tldCB0byB0eHQgZm9ybWF0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEZhbHNlXG4gICAqL1xuICByZWFkb25seSBpc0ZpbGVUcmFuc2Zvcm1hdGlvblJlcXVpcmVkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC4gVGhpcyBidWNrZXQgc3RvcmVzIHRoZSB0cmFuc2Zvcm1lZCAodHh0KSBhc3NldHMgZm9yIGdlbmVyYXRpbmcgc3VtbWFyeS5cbiAgICogSWYgTm9uZSBpcyBwcm92aWRlZCB0aGVuIHRoaXMgY29udHJ1Y3Qgd2lsbCBjcmVhdGUgb25lLlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqPzogczMuSUJ1Y2tldDtcblxuXG4gIC8qKlxuICAgKiBPcHRpb25hbC4gVXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIFMzIEJ1Y2tldC5cbiAgICogUHJvdmlkaW5nIGJvdGggdGhpcyBhbmQgYGV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqYCB3aWxsIGNhdXNlIGFuIGVycm9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldFByb2Nlc3NlZEFzc2V0c1Byb3BzPzogczMuQnVja2V0UHJvcHM7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsLiBFeGlzdGluZyBpbnN0YW5jZSBvZiBFdmVudEJ1cy4gVGhlIHN1bW1hcnkgY29uc3RydWN0IGludGVncmF0ZSBhcHBzeW5jIHdpdGggZXZlbnQgYnJpZGdlJ1xuICAgKiB0byByb3V0ZSB0aGUgcmVxdWVzdCB0byBzdGVwIGZ1bmN0aW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ0J1c0ludGVyZmFjZT86IGV2ZW50cy5JRXZlbnRCdXM7XG5cbiAgLyoqXG4gICAgKiBPcHRpb25hbC4gQSBuZXcgY3VzdG9tIEV2ZW50QnVzIGlzIGNyZWF0ZWQgd2l0aCBwcm92aWRlZCBwcm9wcy5cbiAgICAqIFByb3ZpZGluZyBleGlzdGluZ0V2ZW50QnVzSW50ZXJmYWNlIGFuZCBldmVudEJ1c1Byb3BzIGJvdGggd2lsbCByZXN1bHQgaW4gdmFsaWRhdGlvbiBlcnJvci5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICAqL1xuICByZWFkb25seSBldmVudEJ1c1Byb3BzPzogZXZlbnRzLkV2ZW50QnVzUHJvcHM7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIC0gRXhpc3RpbmcgbWVyZ2VkIEFwcHN5bmMgR3JhcGhRTCBhcGkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhpc3RpbmdNZXJnZWRBcGk/OiBhcHBzeW5jLkNmbkdyYXBoUUxBcGk7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsLiBVc2VyIHByb3ZpZGVkIE5hbWUgZm9yIHN1bW1hcnkgYXBpIG9uIGFwcHN5bmMuXG4gICAqIEEgZ3JhcGhxbCBhcGkgd2lsbCBiZSBjcmVhdGVkIGJ5IHRoaXMgY29uc3RydWN0IHdpdGggdGhpcyBuYW1lLlxuICAgKiBAZGVmYXVsdCAnc3VtbWFyeUFwaSdcbiAgICovXG4gIHJlYWRvbmx5IHN1bW1hcnlBcGlOYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogRW5hYmxlIG9ic2VydmFiaWxpdHkuIFdhcm5pbmc6IGFzc29jaWF0ZWQgY29zdCB3aXRoIHRoZSBzZXJ2aWNlc1xuICAgKiB1c2VkLiBCZXN0IHByYWN0aWNlIHRvIGVuYWJsZSBieSBkZWZhdWx0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IG9ic2VydmFiaWxpdHk/OiBib29sZWFuO1xuICAvKipcbiAgICogT3B0aW9uYWwuIENoYWluIHR5cGUgZGVmaW5lcyBob3cgdG8gcGFzcyB0aGUgZG9jdW1lbnQgdG8gTExNLlxuICAgKiB0aGVyZSBhcmUgdGhyZWUgdHlwZXMgb2YgY2hhaW4gdHlwZXMuXG4gICAqIFN0dWZmOiBTaW1wbHkgXCJzdHVmZlwiIGFsbCB5b3VyIGRvY3VtZW50cyBpbnRvIGEgc2luZ2xlIHByb21wdC5cbiAgICogTWFwLXJlZHVjZTogU3VtbWFyaXplIGVhY2ggZG9jdW1lbnQgb24gaXQncyBvd24gaW4gYSBcIm1hcFwiIHN0ZXAgYW5kIHRoZW4gXCJyZWR1Y2VcIiB0aGUgc3VtbWFyaWVzIGludG8gYSBmaW5hbCBzdW1tYXJ5XG4gICAqIFJlZmluZSA6ICBUaGlzIGNvbnN0cnVjdHMgYSByZXNwb25zZSBieSBsb29waW5nIG92ZXIgdGhlIGlucHV0IGRvY3VtZW50cyBhbmQgaXRlcmF0aXZlbHkgdXBkYXRpbmcgaXRzIGFuc3dlclxuICAgKiBAZGVmYXVsdCAtIFN0dWZmXG4gICAqL1xuICByZWFkb25seSBzdW1tYXJ5Q2hhaW5UeXBlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbC5DREsgY29uc3RydWN0cyBwcm92aWRlZCBjb2xsZWN0cyBhbm9ueW1vdXMgb3BlcmF0aW9uYWxcbiAgICogbWV0cmljcyB0byBoZWxwIEFXUyBpbXByb3ZlIHRoZSBxdWFsaXR5IGFuZCBmZWF0dXJlcyBvZiB0aGVcbiAgICogY29uc3RydWN0cy4gRGF0YSBjb2xsZWN0aW9uIGlzIHN1YmplY3QgdG8gdGhlIEFXUyBQcml2YWN5IFBvbGljeVxuICAgKiAoaHR0cHM6Ly9hd3MuYW1hem9uLmNvbS9wcml2YWN5LykuIFRvIG9wdCBvdXQgb2YgdGhpcyBmZWF0dXJlLFxuICAgKiBzaW1wbHkgZGlzYWJsZSBpdCBieSBzZXR0aW5nIHRoZSBjb25zdHJ1Y3QgcHJvcGVydHlcbiAgICogXCJlbmFibGVPcGVyYXRpb25hbE1ldHJpY1wiIHRvIGZhbHNlIGZvciBlYWNoIGNvbnN0cnVjdCB1c2VkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZU9wZXJhdGlvbmFsTWV0cmljPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVmFsdWUgd2lsbCBiZSBhcHBlbmRlZCB0byByZXNvdXJjZXMgbmFtZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBfZGV2XG4gICAqL1xuICByZWFkb25seSBzdGFnZT86IHN0cmluZztcbn1cblxuZXhwb3J0IGNsYXNzIFN1bW1hcml6YXRpb25BcHBzeW5jU3RlcGZuIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFJldHVybnMgYW4gaW5zdGFuY2Ugb2YgZXZlbnRzLklFdmVudEJ1cyBjcmVhdGVkIGJ5IHRoZSBjb25zdHJ1Y3RcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBldmVudEJyaWRnZUJ1czogZXZlbnRzLklFdmVudEJ1cztcbiAgLyoqXG4gICAqIFJldHVybnMgYW4gaW5zdGFuY2Ugb2YgYXBwc3luYy5DZm5HcmFwaFFMQXBpIGZvciBzdW1tYXJ5IGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGdyYXBocWxBcGk6IGFwcHN5bmMuSUdyYXBocWxBcGkgO1xuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGluc3RhbmNlIG9mIHJlZGlzIGNsdXN0ZXIgY3JlYXRlZCBieSB0aGUgY29uc3RydWN0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVkaXNDbHVzdGVyOiBlbGFzdGljYWNoZS5DZm5DYWNoZUNsdXN0ZXI7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGluc3RhbmNlIG9mIGVjMi5JVnBjIHVzZWQgYnkgdGhlIGNvbnN0cnVjdFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBpbnN0YW5jZSBvZiBlYzIuSVNlY3VyaXR5R3JvdXAgdXNlZCBieSB0aGUgY29uc3RydWN0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc2VjdXJpdHlHcm91cDogZWMyLklTZWN1cml0eUdyb3VwO1xuICAvKipcbiAgICogUmV0dXJucyB0aGUgaW5zdGFuY2Ugb2YgczMuSUJ1Y2tldCB1c2VkIGJ5IHRoZSBjb25zdHJ1Y3RcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbnB1dEFzc2V0QnVja2V0OiBzMy5JQnVja2V0O1xuICAvKipcbiAgICogUmV0dXJucyB0aGUgaW5zdGFuY2Ugb2YgczMuSUJ1Y2tldCB1c2VkIGJ5IHRoZSBjb25zdHJ1Y3RcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwcm9jZXNzZWRBc3NldEJ1Y2tldDogczMuSUJ1Y2tldDtcbiAgLyoqXG4gICAqIFN0ZXAgZnVuY3Rpb25cbiAgICogQGRlZmF1bHQgLSBmaWVsZExvZ0xldmVsIC0gTm9uZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN0YXRlTWFjaGluZTogc2ZuLlN0YXRlTWFjaGluZTtcblxuXG4gIC8qKlxuICAgICAqIEBzdW1tYXJ5IENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIFN1bW1hcml6YXRpb25BcHBzeW5jU3RlcGZuIGNsYXNzLlxuICAgICAqIEBwYXJhbSB7Q29uc3RydWN0fSBzY29wZSAtIHJlcHJlc2VudHMgdGhlIHNjb3BlIGZvciBhbGwgdGhlIHJlc291cmNlcy5cbiAgICAgKiBAcGFyYW0ge3N0cmluZ30gaWQgLSB0aGlzIGlzIGEgYSBzY29wZS11bmlxdWUgaWQuXG4gICAgICogQHBhcmFtIHtTdW1tYXJpemF0aW9uQXBwc3luY1N0ZXBmblByb3BzfSBwcm9wcyAtIHVzZXIgcHJvdmlkZWQgcHJvcHMgZm9yIHRoZSBjb25zdHJ1Y3QuXG4gICAgICogQHNpbmNlIDAuMC4wXG4gICAgICogQGFjY2VzcyBwdWJsaWNcbiAgICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFN1bW1hcml6YXRpb25BcHBzeW5jU3RlcGZuUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbGV0IHN0YWdlID0gJy1kZXYnO1xuICAgIGlmIChwcm9wcz8uc3RhZ2UpIHtcbiAgICAgIHN0YWdlID0gcHJvcHMuc3RhZ2U7XG4gICAgfVxuXG4gICAgLy8gb2JzZXJ2YWJpbGl0eVxuICAgIGxldCBsYW1iZGFfdHJhY2luZyA9IGxhbWJkYS5UcmFjaW5nLkFDVElWRTtcbiAgICBsZXQgZW5hYmxlX3hyYXkgPSB0cnVlO1xuICAgIGxldCBhcGlfbG9nX2NvbmZpZyA9IHtcbiAgICAgIGZpZWxkTG9nTGV2ZWw6IGFwcHN5bmMuRmllbGRMb2dMZXZlbC5BTEwsXG4gICAgICByZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5URU5fWUVBUlMsXG4gICAgfTtcbiAgICBpZiAocHJvcHMub2JzZXJ2YWJpbGl0eSA9PSBmYWxzZSkge1xuICAgICAgZW5hYmxlX3hyYXkgPSBmYWxzZTtcbiAgICAgIGxhbWJkYV90cmFjaW5nID0gbGFtYmRhLlRyYWNpbmcuRElTQUJMRUQ7XG4gICAgICBhcGlfbG9nX2NvbmZpZyA9IHtcbiAgICAgICAgZmllbGRMb2dMZXZlbDogYXBwc3luYy5GaWVsZExvZ0xldmVsLk5PTkUsXG4gICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLlRFTl9ZRUFSUyxcbiAgICAgIH07XG4gICAgfTtcblxuICAgIC8vIHZwY1xuICAgIGlmIChwcm9wcz8uZXhpc3RpbmdWcGMpIHtcbiAgICAgIHRoaXMudnBjID0gcHJvcHMuZXhpc3RpbmdWcGM7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMudnBjID0gbmV3IGVjMi5WcGModGhpcywgJ1ZwYycsIHByb3BzLnZwY1Byb3BzKTtcbiAgICB9XG4gICAgLy8gU2VjdXJpdHkgZ3JvdXBcbiAgICBpZiAocHJvcHM/LmV4aXN0aW5nU2VjdXJpdHlHcm91cCkge1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gcHJvcHMuZXhpc3RpbmdTZWN1cml0eUdyb3VwO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnNlY3VyaXR5R3JvdXAgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAoXG4gICAgICAgIHRoaXMsXG4gICAgICAgICdzZWN1cml0eUdyb3VwJyxcbiAgICAgICAge1xuICAgICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgLy8gdnBjIGZsb3dsb2dncm91cFxuICAgIGNvbnN0IGxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ3N1bW1hcml6YXRpb25Db25zdHJ1Y3RMb2dHcm91cCcpO1xuICAgIGNvbnN0IHZwY0Zsb3dMb2dyb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdzdW1tYXJpemF0aW9uQ29uc3RydWN0Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCd2cGMtZmxvdy1sb2dzLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIHZwYyBmbG93bG9nc1xuICAgIG5ldyBlYzIuRmxvd0xvZyh0aGlzLCAnRmxvd0xvZycsIHtcbiAgICAgIHJlc291cmNlVHlwZTogZWMyLkZsb3dMb2dSZXNvdXJjZVR5cGUuZnJvbVZwYyh0aGlzLnZwYyksXG4gICAgICBkZXN0aW5hdGlvbjogZWMyLkZsb3dMb2dEZXN0aW5hdGlvbi50b0Nsb3VkV2F0Y2hMb2dzKGxvZ0dyb3VwLCB2cGNGbG93TG9ncm9sZSksXG4gICAgfSk7XG5cbiAgICAvLyBidWNrZXQgZm9yIHN0b3Jpbmcgc2VydmVyIGFjY2VzcyBsb2dnaW5nXG4gICAgY29uc3Qgc2VydmVyQWNjZXNzTG9nQnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLFxuICAgICAgJ3NlcnZlckFjY2Vzc0xvZ0J1Y2tldCcrc3RhZ2UsXG4gICAgICB7XG4gICAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBzMy5CbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICAgIGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgICAgZW5mb3JjZVNTTDogdHJ1ZSxcbiAgICAgICAgdmVyc2lvbmVkOiB0cnVlLFxuICAgICAgICBsaWZlY3ljbGVSdWxlczogW3tcbiAgICAgICAgICBleHBpcmF0aW9uOiBEdXJhdGlvbi5kYXlzKDkwKSxcbiAgICAgICAgfV0sXG4gICAgICB9KTtcblxuXG4gICAgLy8gYnVja2V0IGZvciBpbnB1dCBkb2N1bWVudFxuICAgIHMzQnVja2V0SGVscGVyLkNoZWNrUzNQcm9wcyh7XG4gICAgICBleGlzdGluZ0J1Y2tldE9iajogcHJvcHMuZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iaixcbiAgICAgIGJ1Y2tldFByb3BzOiBwcm9wcy5idWNrZXRJbnB1dHNBc3NldHNQcm9wcyxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcz8uZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iaikge1xuICAgICAgdGhpcy5pbnB1dEFzc2V0QnVja2V0ID0gcHJvcHMuZXhpc3RpbmdJbnB1dEFzc2V0c0J1Y2tldE9iajtcbiAgICB9IGVsc2UgaWYgKHByb3BzPy5idWNrZXRJbnB1dHNBc3NldHNQcm9wcykge1xuICAgICAgdGhpcy5pbnB1dEFzc2V0QnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLFxuICAgICAgICAnaW5wdXRBc3NldHNTdW1tYXJ5QnVja2V0JytzdGFnZSwgcHJvcHMuYnVja2V0SW5wdXRzQXNzZXRzUHJvcHMpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBidWNrZXROYW1lPSAnaW5wdXQtYXNzZXRzLXN1bW1hcnktYnVja2V0JytzdGFnZSsnLScrQXdzLkFDQ09VTlRfSUQ7XG4gICAgICB0aGlzLmlucHV0QXNzZXRCdWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsICdpbnB1dEFzc2V0c1N1bW1hcnlCdWNrZXQnK3N0YWdlLFxuICAgICAgICB7XG4gICAgICAgICAgYmxvY2tQdWJsaWNBY2Nlc3M6IHMzLkJsb2NrUHVibGljQWNjZXNzLkJMT0NLX0FMTCxcbiAgICAgICAgICBlbmNyeXB0aW9uOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gICAgICAgICAgYnVja2V0TmFtZTogYnVja2V0TmFtZSxcbiAgICAgICAgICBzZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0OiBzZXJ2ZXJBY2Nlc3NMb2dCdWNrZXQsXG4gICAgICAgICAgZW5mb3JjZVNTTDogdHJ1ZSxcbiAgICAgICAgICB2ZXJzaW9uZWQ6IHRydWUsXG4gICAgICAgICAgbGlmZWN5Y2xlUnVsZXM6IFt7XG4gICAgICAgICAgICBleHBpcmF0aW9uOiBEdXJhdGlvbi5kYXlzKDkwKSxcbiAgICAgICAgICB9XSxcbiAgICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gYnVja2V0IGZvciB0cmFuc2Zvcm1lZCBkb2N1bWVudFxuICAgIHMzQnVja2V0SGVscGVyLkNoZWNrUzNQcm9wcyh7XG4gICAgICBleGlzdGluZ0J1Y2tldE9iajogcHJvcHMuZXhpc3RpbmdQcm9jZXNzZWRBc3NldHNCdWNrZXRPYmosXG4gICAgICBidWNrZXRQcm9wczogcHJvcHMuYnVja2V0UHJvY2Vzc2VkQXNzZXRzUHJvcHMsXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHM/LmV4aXN0aW5nUHJvY2Vzc2VkQXNzZXRzQnVja2V0T2JqKSB7XG4gICAgICB0aGlzLnByb2Nlc3NlZEFzc2V0QnVja2V0ID0gcHJvcHMuZXhpc3RpbmdQcm9jZXNzZWRBc3NldHNCdWNrZXRPYmo7XG4gICAgfSBlbHNlIGlmIChwcm9wcz8uYnVja2V0UHJvY2Vzc2VkQXNzZXRzUHJvcHMpIHtcbiAgICAgIHRoaXMucHJvY2Vzc2VkQXNzZXRCdWNrZXQgPSBuZXcgczMuQnVja2V0KHRoaXMsXG4gICAgICAgICdwcm9jZXNzZWRBc3NldHNTdW1tYXJ5QnVja2V0JytzdGFnZSwgcHJvcHMuYnVja2V0UHJvY2Vzc2VkQXNzZXRzUHJvcHMpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBidWNrZXROYW1lPSAncHJvY2Vzc2VkLWFzc2V0cy1zdW1tYXJ5LWJ1Y2tldCcrc3RhZ2UrJy0nK0F3cy5BQ0NPVU5UX0lEO1xuXG4gICAgICB0aGlzLnByb2Nlc3NlZEFzc2V0QnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0aGlzLCAncHJvY2Vzc2VkQXNzZXRzU3VtbWFyeUJ1Y2tldCcrc3RhZ2UsXG4gICAgICAgIHtcbiAgICAgICAgICBibG9ja1B1YmxpY0FjY2VzczogczMuQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgICAgICAgIGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uUzNfTUFOQUdFRCxcbiAgICAgICAgICBidWNrZXROYW1lOiBidWNrZXROYW1lLFxuICAgICAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQ6IHNlcnZlckFjY2Vzc0xvZ0J1Y2tldCxcbiAgICAgICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgICAgIHZlcnNpb25lZDogdHJ1ZSxcbiAgICAgICAgICBsaWZlY3ljbGVSdWxlczogW3tcbiAgICAgICAgICAgIGV4cGlyYXRpb246IER1cmF0aW9uLmRheXMoOTApLFxuICAgICAgICAgIH1dLFxuICAgICAgICB9KTtcbiAgICB9XG5cblxuICAgIC8vIHNldCB1cCByZWRpcyBjbHVzdGVyXG4gICAgcmVkaXNIZWxwZXIuQ2hlY2tSZWRpc0NsdXN0ZXJQcm9wcyhwcm9wcyk7XG5cblxuICAgIC8vIGJ1aWxkIHJlZGlzIGNsdXN0ZXIgb25seSB3aGVuIGNmbkNhY2hlQ2x1c3RlclByb3BzIGlzIHNldFxuICAgIGlmIChwcm9wcz8uY2ZuQ2FjaGVDbHVzdGVyUHJvcHMpIHtcbiAgICAgIGNvbnN0IHJlZGlzU2VjdXJpdHlHcm91cCA9cmVkaXNIZWxwZXIuZ2V0UmVkaXNTZWN1cml0eUdyb3VwKHRoaXMsIHtcbiAgICAgICAgZXhpc3RpbmdWcGM6IHRoaXMudnBjLFxuICAgICAgfSk7XG4gICAgICBjb25zdCByZWRpc1Byb3BzID0ge1xuICAgICAgICBleGlzdGluZ1ZwYzogdGhpcy52cGMsXG4gICAgICAgIGNmbkNhY2hlQ2x1c3RlclByb3BzOiBwcm9wcy5jZm5DYWNoZUNsdXN0ZXJQcm9wcyxcbiAgICAgICAgc3VibmV0SWRzOiB2cGNIZWxwZXIuZ2V0UHJpdmF0ZVN1Ym5ldElEcyh0aGlzLnZwYyksXG4gICAgICAgIGluYm91bmRTZWN1cml0eUdyb3VwOiB0aGlzLnNlY3VyaXR5R3JvdXAsXG4gICAgICAgIHJlZGlzU2VjdXJpdHlHcm91cDogcmVkaXNTZWN1cml0eUdyb3VwLFxuICAgICAgICByZWRpc1BvcnQ6IDg2ODYsXG4gICAgICB9O1xuICAgICAgdGhpcy5yZWRpc0NsdXN0ZXIgPSByZWRpc0hlbHBlci5idWlsZFJlZGlzQ2x1c3Rlcih0aGlzLCByZWRpc1Byb3BzKTtcbiAgICAgIHJlZGlzSGVscGVyLnNldEluYm91bmRSdWxlcyhyZWRpc1NlY3VyaXR5R3JvdXAsIHRoaXMuc2VjdXJpdHlHcm91cCwgcmVkaXNQcm9wcy5yZWRpc1BvcnQpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnJlZGlzQ2x1c3Rlcj0gcHJvcHM/LmV4aXN0aW5nUmVkaXNDdWxzdGVyITtcbiAgICB9XG5cbiAgICBjb25zdCByZWRpc0hvc3QgPSB0aGlzLnJlZGlzQ2x1c3Rlcj8uYXR0clJlZGlzRW5kcG9pbnRBZGRyZXNzITtcbiAgICBjb25zdCByZWRpc1BvcnQgPSB0aGlzLnJlZGlzQ2x1c3Rlcj8uYXR0clJlZGlzRW5kcG9pbnRQb3J0ITtcblxuXG4gICAgZXZlbnRCcmlkZ2UuQ2hlY2tFdmVudEJyaWRnZVByb3BzKHByb3BzKTtcbiAgICAvLyBDcmVhdGUgZXZlbnQgYnJpZGdlXG4gICAgdGhpcy5ldmVudEJyaWRnZUJ1cyA9IGV2ZW50QnJpZGdlLmJ1aWxkRXZlbnRCdXModGhpcywge1xuICAgICAgZXhpc3RpbmdFdmVudEJ1c0ludGVyZmFjZTogcHJvcHMuZXhpc3RpbmdCdXNJbnRlcmZhY2UsXG4gICAgICBldmVudEJ1c1Byb3BzOiBwcm9wcy5ldmVudEJ1c1Byb3BzLFxuICAgIH0pO1xuXG4gICAgLy8gbWFrZSBpdCBnZW5lcmljIGZvciBvdGhlciBhdXRoIGNvbmZpZyBhcyB3ZWxsXG4gICAgY29uc3QgYXV0aG9yaXphdGlvbkNvbmZpZzogYXBwc3luYy5BdXRob3JpemF0aW9uQ29uZmlnID0ge1xuICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHtcbiAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MLFxuICAgICAgICB1c2VyUG9vbENvbmZpZzogeyB1c2VyUG9vbDogcHJvcHMuY29nbml0b1VzZXJQb29sIH0sXG4gICAgICB9LFxuICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW1xuICAgICAgICB7XG4gICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuSUFNLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9O1xuXG4gICAgY29uc3QgYXBpTmFtZSA9IHByb3BzLnN1bW1hcnlBcGlOYW1lIHx8ICdzdW1tYXJ5QXBpJztcblxuICAgIC8vIGdyYXBocWwgYXBpIGZvciBzdW1tYXJ5LiBjbGllbnQgaW52b2tlIHRoaXMgYXBpIHdpdGggZ2l2ZW4gc2NoZW1hIGFuZCBjb2duaXRvIHVzZXIgcG9vbCBhdXRoLlxuICAgIGNvbnN0IHN1bW1hcml6YXRpb25HcmFwaHFsQXBpID0gbmV3IGFwcHN5bmMuR3JhcGhxbEFwaSh0aGlzLCAnc3VtbWFyaXphdGlvbkdyYXBocWxBcGknK3N0YWdlLFxuICAgICAge1xuICAgICAgICBuYW1lOiBhcGlOYW1lK3N0YWdlLFxuICAgICAgICBsb2dDb25maWc6IGFwaV9sb2dfY29uZmlnLFxuICAgICAgICBkZWZpbml0aW9uOiBhcHBzeW5jLkRlZmluaXRpb24uZnJvbUZpbGUoXG4gICAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uLy4uLy4uLy4uL3Jlc291cmNlcy9nZW4tYWkvYXdzLXN1bW1hcml6YXRpb24tYXBwc3luYy1zdGVwZm4vc2NoZW1hLmdyYXBocWwnKSxcbiAgICAgICAgKSxcbiAgICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzogYXV0aG9yaXphdGlvbkNvbmZpZyxcbiAgICAgICAgeHJheUVuYWJsZWQ6IGVuYWJsZV94cmF5LFxuICAgICAgfSk7XG4gICAgdGhpcy5ncmFwaHFsQXBpPSBzdW1tYXJpemF0aW9uR3JhcGhxbEFwaTtcblxuICAgIC8vIElmIHRoZSB1c2VyIHByb3ZpZGVzIGEgbWVyZ2VkQXBpIGVuZHBvaW50LCB0aGUgbGFtYmRhXG4gICAgLy8gZnVuY3Rpb25zIHdpbGwgdXNlIHRoaXMgZW5kcG9pbnQgdG8gc2VuZCB0aGVpciBzdGF0dXMgdXBkYXRlc1xuICAgIGNvbnN0IHVwZGF0ZUdyYXBoUWxBcGlFbmRwb2ludCA9ICFwcm9wcy5leGlzdGluZ01lcmdlZEFwaSA/IHN1bW1hcml6YXRpb25HcmFwaHFsQXBpLmdyYXBocWxVcmwgOiBwcm9wcy5leGlzdGluZ01lcmdlZEFwaS5hdHRyR3JhcGhRbFVybDtcbiAgICBjb25zdCB1cGRhdGVHcmFwaFFsQXBpSWQgPSAhcHJvcHMuZXhpc3RpbmdNZXJnZWRBcGkgPyBzdW1tYXJpemF0aW9uR3JhcGhxbEFwaS5hcGlJZCA6IHByb3BzLmV4aXN0aW5nTWVyZ2VkQXBpLmF0dHJBcGlJZDtcblxuICAgIC8vIGJ1Y2tldFxuICAgIGNvbnN0IHRyYW5zZm9ybWVkQXNzZXRCdWNrZXROYW1lID0gdGhpcy5wcm9jZXNzZWRBc3NldEJ1Y2tldC5idWNrZXROYW1lO1xuICAgIGNvbnN0IGlucHV0QXNzZXRCdWNrZXROYW1lID0gdGhpcy5pbnB1dEFzc2V0QnVja2V0LmJ1Y2tldE5hbWU7XG4gICAgY29uc3QgaXNGaWxlVHJhbnNmb3JtYXRpb25SZXF1aXJlZCA9IHByb3BzPy5pc0ZpbGVUcmFuc2Zvcm1hdGlvblJlcXVpcmVkIHx8ICdmYWxzZSc7XG5cbiAgICAvLyByb2xlXG4gICAgY29uc3QgaW5wdXR2YWxpZGF0b3JMYW1iZGFSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdpbnB1dHZhbGlkYXRvckxhbWJkYVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIExhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGVQb2xpY3k6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOiR7QXdzLlBBUlRJVElPTn06bG9nczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8qYF0sXG4gICAgICAgICAgfSldLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBNaW5pbXVtIHBlcm1pc3Npb25zIGZvciBhIExhbWJkYSBmdW5jdGlvbiB0byBleGVjdXRlIHdoaWxlIGFjY2Vzc2luZyBhIHJlc291cmNlIHdpdGhpbiBhIFZQQ1xuICAgIGlucHV0dmFsaWRhdG9yTGFtYmRhUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICdlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICdlYzI6QXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzJyxcbiAgICAgICAgJ2VjMjpVbmFzc2lnblByaXZhdGVJcEFkZHJlc3NlcycsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICdhcm46YXdzOmVjMjonK0F3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6Ki8qJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuICAgIC8vIERlY3JpYmUgb25seSB3b3JrcyBpZiBpdCdzIGFsbG93ZWQgb24gYWxsIHJlc291cmNlcy5cbiAgICAvLyBSZWZlcmVuY2U6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYW1iZGEvbGF0ZXN0L2RnL2NvbmZpZ3VyYXRpb24tdnBjLmh0bWwjdnBjLXBlcm1pc3Npb25zXG4gICAgaW5wdXR2YWxpZGF0b3JMYW1iZGFSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJyonLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICBpbnB1dHZhbGlkYXRvckxhbWJkYVJvbGUuYWRkVG9Qb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICdzMzpHZXRCdWNrZXRMb2NhdGlvbicsXG4gICAgICAgICAgJ3MzOkxpc3RCdWNrZXQnLFxuICAgICAgICAgICdhcHBzeW5jOkdyYXBoUUwnXSxcblxuICAgICAgICByZXNvdXJjZXM6IFsnYXJuOmF3czpzMzo6OicgKyBpbnB1dEFzc2V0QnVja2V0TmFtZSArICcvKicsXG4gICAgICAgICAgJ2Fybjphd3M6czM6OjonICsgdHJhbnNmb3JtZWRBc3NldEJ1Y2tldE5hbWUrICcvKicsXG4gICAgICAgICAgJ2Fybjphd3M6YXBwc3luYzonK0F3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6YXBpcy8nK3VwZGF0ZUdyYXBoUWxBcGlJZCsgJy8qJ10sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgaW5wdXR2YWxpZGF0b3JMYW1iZGFSb2xlLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNScsXG4gICAgICAgICAgcmVhc29uOiAnQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIGlzIHVzZWQuJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICB0cnVlLFxuICAgICk7XG5cbiAgICAvLyBMYW1iZGEgZnVuY3Rpb24gdG8gdmFsaWRhdGUgSW5wdXRcbiAgICBjb25zdCBpbnB1dFZhbGlkYXRvckxhbWJkYSA9XG4gICAgbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdpbnB1dFZhbGlkYXRvckxhbWJkYScrc3RhZ2UsXG4gICAgICB7XG4gICAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uLy4uLy4uLy4uL2xhbWJkYS9hd3Mtc3VtbWFyaXphdGlvbi1hcHBzeW5jLXN0ZXBmbi9pbnB1dF92YWxpZGF0b3InKSksXG4gICAgICAgIGZ1bmN0aW9uTmFtZTogJ3N1bW1hcnlfaW5wdXRfdmFsaWRhdG9yJytzdGFnZSxcbiAgICAgICAgZGVzY3JpcHRpb246ICdMYW1iZGEgZnVuY3Rpb24gdG8gdmFsaWRhdGUgaW5wdXQgZm9yIHN1bW1hcnkgYXBpJyxcbiAgICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgICAgdHJhY2luZzogbGFtYmRhX3RyYWNpbmcsXG4gICAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX0VHUkVTUyB9LFxuICAgICAgICBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0sXG4gICAgICAgIG1lbW9yeVNpemU6IGxhbWJkYU1lbW9yeVNpemVMaW1pdGVyKHRoaXMsIDFfNzY5ICogMSksXG4gICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICAgIHJvbGU6IGlucHV0dmFsaWRhdG9yTGFtYmRhUm9sZSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBHUkFQSFFMX1VSTDogdXBkYXRlR3JhcGhRbEFwaUVuZHBvaW50LFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAvLyByb2xlXG4gICAgY29uc3QgZG9jdW1lbnRSZWFkZXJMYW1iZGFSb2xlID0gbmV3IGlhbS5Sb2xlKHRoaXMsICdkb2N1bWVudFJlYWRlckxhbWJkYVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIExhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGVQb2xpY3k6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOiR7QXdzLlBBUlRJVElPTn06bG9nczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8qYF0sXG4gICAgICAgICAgfSldLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBNaW5pbXVtIHBlcm1pc3Npb25zIGZvciBhIExhbWJkYSBmdW5jdGlvbiB0byBleGVjdXRlIHdoaWxlIGFjY2Vzc2luZyBhIHJlc291cmNlIHdpdGhpbiBhIFZQQ1xuICAgIGRvY3VtZW50UmVhZGVyTGFtYmRhUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICdlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICdlYzI6QXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzJyxcbiAgICAgICAgJ2VjMjpVbmFzc2lnblByaXZhdGVJcEFkZHJlc3NlcycsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICdhcm46YXdzOmVjMjonK0F3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6Ki8qJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuICAgIC8vIERlY3JpYmUgb25seSB3b3JrcyBpZiBpdCdzIGFsbG93ZWQgb24gYWxsIHJlc291cmNlcy5cbiAgICAvLyBSZWZlcmVuY2U6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYW1iZGEvbGF0ZXN0L2RnL2NvbmZpZ3VyYXRpb24tdnBjLmh0bWwjdnBjLXBlcm1pc3Npb25zXG4gICAgZG9jdW1lbnRSZWFkZXJMYW1iZGFSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2VjMjpEZXNjcmliZU5ldHdvcmtJbnRlcmZhY2VzJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJyonLFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICBkb2N1bWVudFJlYWRlckxhbWJkYVJvbGUuYWRkVG9Qb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICdzMzpHZXRCdWNrZXRMb2NhdGlvbicsXG4gICAgICAgICAgJ3MzOkxpc3RCdWNrZXQnLFxuICAgICAgICAgICdzMzpQdXRPYmplY3QnLFxuICAgICAgICAgICdhcHBzeW5jOkdyYXBoUUwnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbJ2Fybjphd3M6czM6OjonICsgaW5wdXRBc3NldEJ1Y2tldE5hbWUrICcvKicsXG4gICAgICAgICAgJ2Fybjphd3M6czM6OjonICsgdHJhbnNmb3JtZWRBc3NldEJ1Y2tldE5hbWUrICcvKicsXG4gICAgICAgICAgJ2Fybjphd3M6YXBwc3luYzonK0F3cy5SRUdJT04rJzonK0F3cy5BQ0NPVU5UX0lEKyc6YXBpcy8nK3VwZGF0ZUdyYXBoUWxBcGlJZCsgJy8qJ10sXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgZG9jdW1lbnRSZWFkZXJMYW1iZGFSb2xlLFxuICAgICAgW1xuICAgICAgICB7XG4gICAgICAgICAgaWQ6ICdBd3NTb2x1dGlvbnMtSUFNNScsXG4gICAgICAgICAgcmVhc29uOiAnQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIGlzIHVzZWQnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWUsXG4gICAgKTtcblxuICAgIGNvbnN0IGRvY3VtZW50UmVhZGVyTGFtYmRhID0gbmV3IGxhbWJkYS5Eb2NrZXJJbWFnZUZ1bmN0aW9uKHRoaXMsICdkb2N1bWVudFJlYWRlckxhbWJkYScrc3RhZ2UsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Eb2NrZXJJbWFnZUNvZGUuZnJvbUltYWdlQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uLy4uLy4uLy4uL2xhbWJkYS9hd3Mtc3VtbWFyaXphdGlvbi1hcHBzeW5jLXN0ZXBmbi9kb2N1bWVudF9yZWFkZXInKSksXG4gICAgICBmdW5jdGlvbk5hbWU6ICdzdW1tYXJ5X2RvY3VtZW50X3JlYWRlcicrc3RhZ2UsXG4gICAgICBkZXNjcmlwdGlvbjogJ0xhbWJkYSBmdW5jdGlvbiB0byByZWFkIHRoZSBpbnB1dCB0cmFuc2Zvcm1lZCBkb2N1bWVudCcsXG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgICAgdnBjU3VibmV0czogeyBzdWJuZXRUeXBlOiBlYzIuU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfRUdSRVNTIH0sXG4gICAgICBzZWN1cml0eUdyb3VwczogW3RoaXMuc2VjdXJpdHlHcm91cF0sXG4gICAgICBtZW1vcnlTaXplOiBsYW1iZGFNZW1vcnlTaXplTGltaXRlcih0aGlzLCAxXzc2OSAqIDEpLFxuICAgICAgdHJhY2luZzogbGFtYmRhX3RyYWNpbmcsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgcm9sZTogZG9jdW1lbnRSZWFkZXJMYW1iZGFSb2xlLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgUkVESVNfSE9TVDogcmVkaXNIb3N0LFxuICAgICAgICBSRURJU19QT1JUOiByZWRpc1BvcnQsXG4gICAgICAgIFRSQU5TRk9STUVEX0FTU0VUX0JVQ0tFVDogdHJhbnNmb3JtZWRBc3NldEJ1Y2tldE5hbWUsXG4gICAgICAgIElOUFVUX0FTU0VUX0JVQ0tFVDogaW5wdXRBc3NldEJ1Y2tldE5hbWUsXG4gICAgICAgIElTX0ZJTEVfVFJBTlNGT1JNRUQ6IGlzRmlsZVRyYW5zZm9ybWF0aW9uUmVxdWlyZWQsXG4gICAgICAgIEdSQVBIUUxfVVJMOiB1cGRhdGVHcmFwaFFsQXBpRW5kcG9pbnQsXG5cbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBzdW1tYXJ5Q2hhaW5UeXBlID0gcHJvcHM/LnN1bW1hcnlDaGFpblR5cGUgfHwgJ3N0dWZmJztcblxuICAgIC8vIHJvbGVcbiAgICBjb25zdCBzdW1tYXJ5R2VuZXJhdG9yTGFtYmRhUm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnc3VtbWFyeUdlbmVyYXRvckxhbWJkYVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIExhbWJkYUZ1bmN0aW9uU2VydmljZVJvbGVQb2xpY3k6IG5ldyBpYW0uUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtgYXJuOiR7QXdzLlBBUlRJVElPTn06bG9nczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8qYF0sXG4gICAgICAgICAgfSldLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBNaW5pbXVtIHBlcm1pc3Npb25zIGZvciBhIExhbWJkYSBmdW5jdGlvbiB0byBleGVjdXRlIHdoaWxlIGFjY2Vzc2luZyBhIHJlc291cmNlIHdpdGhpbiBhIFZQQ1xuICAgIHN1bW1hcnlHZW5lcmF0b3JMYW1iZGFSb2xlLmFkZFRvUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2VjMjpDcmVhdGVOZXR3b3JrSW50ZXJmYWNlJyxcbiAgICAgICAgJ2VjMjpEZWxldGVOZXR3b3JrSW50ZXJmYWNlJyxcbiAgICAgICAgJ2VjMjpBc3NpZ25Qcml2YXRlSXBBZGRyZXNzZXMnLFxuICAgICAgICAnZWMyOlVuYXNzaWduUHJpdmF0ZUlwQWRkcmVzc2VzJyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgJ2Fybjphd3M6ZWMyOicrQXdzLlJFR0lPTisnOicrQXdzLkFDQ09VTlRfSUQrJzoqLyonLFxuICAgICAgXSxcbiAgICB9KSk7XG4gICAgLy8gRGVjcmliZSBvbmx5IHdvcmtzIGlmIGl0J3MgYWxsb3dlZCBvbiBhbGwgcmVzb3VyY2VzLlxuICAgIC8vIFJlZmVyZW5jZTogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xhbWJkYS9sYXRlc3QvZGcvY29uZmlndXJhdGlvbi12cGMuaHRtbCN2cGMtcGVybWlzc2lvbnNcbiAgICBzdW1tYXJ5R2VuZXJhdG9yTGFtYmRhUm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgICdlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlcycsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICcqJyxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgc3VtbWFyeUdlbmVyYXRvckxhbWJkYVJvbGUuYWRkVG9Qb2xpY3koXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLFxuICAgICAgICAgICdzMzpHZXRCdWNrZXRMb2NhdGlvbicsXG4gICAgICAgICAgJ3MzOkxpc3RCdWNrZXQnLFxuICAgICAgICAgICdzMzpQdXRPYmplY3QnLFxuICAgICAgICAgICdhcHBzeW5jOkdyYXBoUUwnLFxuICAgICAgICAgICdiZWRyb2NrOkludm9rZU1vZGVsJyxcbiAgICAgICAgICAnYmVkcm9jazpJbnZva2VNb2RlbFdpdGhSZXNwb25zZVN0cmVhbSddLFxuICAgICAgICByZXNvdXJjZXM6IFsnYXJuOmF3czpzMzo6OicgKyBpbnB1dEFzc2V0QnVja2V0TmFtZSsgJy8qJyxcbiAgICAgICAgICAnYXJuOmF3czpzMzo6OicgKyB0cmFuc2Zvcm1lZEFzc2V0QnVja2V0TmFtZSsgJy8qJyxcbiAgICAgICAgICAnYXJuOmF3czphcHBzeW5jOicrQXdzLlJFR0lPTisnOicrQXdzLkFDQ09VTlRfSUQrJzphcGlzLycrdXBkYXRlR3JhcGhRbEFwaUlkKyAnLyonLFxuICAgICAgICAgICdhcm46YXdzOmJlZHJvY2s6JytBd3MuUkVHSU9OKyc6OmZvdW5kYXRpb24tbW9kZWwvKiddLFxuXG4gICAgICB9KSxcbiAgICApO1xuXG4gICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgc3VtbWFyeUdlbmVyYXRvckxhbWJkYVJvbGUsXG4gICAgICBbXG4gICAgICAgIHtcbiAgICAgICAgICBpZDogJ0F3c1NvbHV0aW9ucy1JQU01JyxcbiAgICAgICAgICByZWFzb246ICdBV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUgaXMgdXNlZC4nLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIHRydWUsXG4gICAgKTtcblxuICAgIGNvbnN0IGdlbmVyYXRlU3VtbWFyeWxhbWJkYSA9IG5ldyBsYW1iZGEuRG9ja2VySW1hZ2VGdW5jdGlvbih0aGlzLCAnZ2VuZXJhdGVTdW1tYXJ5bGFtYmRhJytzdGFnZSwge1xuICAgICAgZnVuY3Rpb25OYW1lOiAnc3VtbWFyeV9nZW5lcmF0b3InK3N0YWdlLFxuICAgICAgZGVzY3JpcHRpb246ICdMYW1iZGEgZnVuY3Rpb24gdG8gZ2VuZXJhdGUgdGhlIHN1bW1hcnknLFxuICAgICAgY29kZTogbGFtYmRhLkRvY2tlckltYWdlQ29kZS5mcm9tSW1hZ2VBc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vLi4vLi4vLi4vbGFtYmRhL2F3cy1zdW1tYXJpemF0aW9uLWFwcHN5bmMtc3RlcGZuL3N1bW1hcnlfZ2VuZXJhdG9yJykpLFxuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX0VHUkVTUyB9LFxuICAgICAgc2VjdXJpdHlHcm91cHM6IFt0aGlzLnNlY3VyaXR5R3JvdXBdLFxuICAgICAgbWVtb3J5U2l6ZTogbGFtYmRhTWVtb3J5U2l6ZUxpbWl0ZXIodGhpcywgMV83NjkgKiA0KSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTApLFxuICAgICAgdHJhY2luZzogbGFtYmRhX3RyYWNpbmcsXG4gICAgICByb2xlOiBzdW1tYXJ5R2VuZXJhdG9yTGFtYmRhUm9sZSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIFJFRElTX0hPU1Q6IHJlZGlzSG9zdCxcbiAgICAgICAgUkVESVNfUE9SVDogcmVkaXNQb3J0LFxuICAgICAgICBBU1NFVF9CVUNLRVRfTkFNRTogdHJhbnNmb3JtZWRBc3NldEJ1Y2tldE5hbWUsXG4gICAgICAgIEdSQVBIUUxfVVJMOiB1cGRhdGVHcmFwaFFsQXBpRW5kcG9pbnQsXG4gICAgICAgIFNVTU1BUllfTExNX0NIQUlOX1RZUEU6IHN1bW1hcnlDaGFpblR5cGUsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5pbnB1dEFzc2V0QnVja2V0Py5ncmFudFJlYWQoZ2VuZXJhdGVTdW1tYXJ5bGFtYmRhKTtcbiAgICB0aGlzLnByb2Nlc3NlZEFzc2V0QnVja2V0Py5ncmFudFJlYWRXcml0ZShnZW5lcmF0ZVN1bW1hcnlsYW1iZGEpO1xuICAgIHRoaXMuaW5wdXRBc3NldEJ1Y2tldD8uZ3JhbnRSZWFkKGRvY3VtZW50UmVhZGVyTGFtYmRhKTtcbiAgICB0aGlzLnByb2Nlc3NlZEFzc2V0QnVja2V0Py5ncmFudFJlYWRXcml0ZShkb2N1bWVudFJlYWRlckxhbWJkYSk7XG5cblxuICAgIGNvbnN0IGVuYWJsZU9wZXJhdGlvbmFsTWV0cmljID1cbiAgICAgIHByb3BzLmVuYWJsZU9wZXJhdGlvbmFsTWV0cmljICE9PSB1bmRlZmluZWQgJiYgcHJvcHMuZW5hYmxlT3BlcmF0aW9uYWxNZXRyaWMgIT09IG51bGwgPyBwcm9wcy5lbmFibGVPcGVyYXRpb25hbE1ldHJpYyA6IHRydWU7XG5cbiAgICBpZiAoZW5hYmxlT3BlcmF0aW9uYWxNZXRyaWMpIHtcbiAgICAgIGNvbnN0IHNvbHV0aW9uSWQgPSBgZ2VuYWlfY2RrXyR7dmVyc2lvbn0vJHt0aGlzLmNvbnN0cnVjdG9yLm5hbWV9LyR7aWR9YDtcbiAgICAgIGRvY3VtZW50UmVhZGVyTGFtYmRhLmFkZEVudmlyb25tZW50KFxuICAgICAgICAnQVdTX1NES19VQV9BUFBfSUQnLCBzb2x1dGlvbklkLFxuICAgICAgKTtcbiAgICAgIGdlbmVyYXRlU3VtbWFyeWxhbWJkYS5hZGRFbnZpcm9ubWVudChcbiAgICAgICAgJ0FXU19TREtfVUFfQVBQX0lEJywgc29sdXRpb25JZCxcbiAgICAgICk7XG4gICAgICBpbnB1dFZhbGlkYXRvckxhbWJkYS5hZGRFbnZpcm9ubWVudChcbiAgICAgICAgJ0FXU19TREtfVUFfQVBQX0lEJywgc29sdXRpb25JZCxcbiAgICAgICk7XG4gICAgfTtcblxuXG4gICAgLy8gY3JlYXRlIGRhdGFzb3VyY2UgYXQgYXBwc3luY1xuICAgIGNvbnN0IFN1bW1hcnlTdGF0dXNEYXRhU291cmNlID0gbmV3IGFwcHN5bmMuTm9uZURhdGFTb3VyY2VcbiAgICAodGhpcywgJ25vbmVEYXRhU291cmNlJytzdGFnZSwge1xuICAgICAgYXBpOiBzdW1tYXJpemF0aW9uR3JhcGhxbEFwaSxcbiAgICAgIG5hbWU6ICdTdW1tYXJ5U3RhdHVzRGF0YVNvdXJjZScsXG4gICAgfSk7XG5cbiAgICBTdW1tYXJ5U3RhdHVzRGF0YVNvdXJjZS5jcmVhdGVSZXNvbHZlclxuICAgICgnc3VtbWFyeVJlc3BvbnNlcmVzb2x2ZXInK3N0YWdlLCB7XG4gICAgICB0eXBlTmFtZTogJ011dGF0aW9uJyxcbiAgICAgIGZpZWxkTmFtZTogJ3VwZGF0ZVN1bW1hcnlKb2JTdGF0dXMnLFxuICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogYXBwc3luYy5NYXBwaW5nVGVtcGxhdGUuZnJvbVN0cmluZyhcbiAgICAgICAgJ3sgXCJ2ZXJzaW9uXCI6IFwiMjAxNy0wMi0yOFwiLCBcInBheWxvYWRcIjogJHV0aWwudG9Kc29uKCRjb250ZXh0LmFyZ3MpIH0nLFxuICAgICAgKSxcblxuICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICckdXRpbC50b0pzb24oJGNvbnRleHQucmVzdWx0KScpLFxuICAgIH0pO1xuXG4gICAgLy8gc3RlcCBmdW5jdGlvbnNcbiAgICBjb25zdCBpbnB1dFZhbGlkYXRpb25UYXNrID0gbmV3IHNmblRhc2suTGFtYmRhSW52b2tlKHRoaXMsICdWYWxpZGF0ZSBJbnB1dCAnLCB7XG4gICAgICBsYW1iZGFGdW5jdGlvbjogaW5wdXRWYWxpZGF0b3JMYW1iZGEsXG4gICAgICByZXN1bHRQYXRoOiAnJC52YWxpZGF0aW9uX3Jlc3VsdCcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBkb2N1bWVudFJlYWRlclRhc2sgPSBuZXcgc2ZuVGFzay5MYW1iZGFJbnZva2UodGhpcywgJ1JlYWQgZG9jdW1lbnQgYW5kIGNoZWNrIHN1bW1hcnkgaW4gY2FjaGUgJywge1xuICAgICAgbGFtYmRhRnVuY3Rpb246IGRvY3VtZW50UmVhZGVyTGFtYmRhLFxuICAgICAgcmVzdWx0UGF0aDogJyQuZG9jdW1lbnRfcmVzdWx0JyxcbiAgICB9KTtcblxuXG4gICAgY29uc3QgZ2VuZXJhdGVTdW1tYXJ5VGFzayA9IG5ldyBzZm5UYXNrLkxhbWJkYUludm9rZSh0aGlzLCAnR2VuZXJhdGUgU3VtbWFyeSB3aXRoIGxsbScsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uOiBnZW5lcmF0ZVN1bW1hcnlsYW1iZGEsXG4gICAgICByZXN1bHRQYXRoOiAnJC5zdW1tYXJ5X3Jlc3VsdCcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBkbHE6IHNxcy5RdWV1ZSA9IG5ldyBzcXMuUXVldWUodGhpcywgJ2RscScsIHtcbiAgICAgIHF1ZXVlTmFtZTogJ3N1bW1hcnlkbHEnK3N0YWdlLFxuICAgICAgcmV0ZW50aW9uUGVyaW9kOiBEdXJhdGlvbi5kYXlzKDcpLFxuICAgICAgZW5mb3JjZVNTTDogdHJ1ZSxcbiAgICB9KTtcblxuXG4gICAgY29uc3Qgam9iRmFpbGVkPSBuZXcgc2ZuLkZhaWwodGhpcywgJ0ZhaWxlZCcsIHtcbiAgICAgIGNvbW1lbnQ6ICdBV1Mgc3VtbWFyeSBKb2IgZmFpbGVkJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGpvYlN1Y2Nlc3M9IG5ldyBzZm4uU3VjY2VlZCh0aGlzLCAnc3VjY2VlZGVkJywge1xuICAgICAgY29tbWVudDogJ0FXUyBzdW1tYXJ5IEpvYiBzdWNjZWVkZWQnLFxuICAgIH0pO1xuXG4gICAgLy8gc3RlcCBmdW5jdGlvbiBjaG9pY2Ugc3RlcHNcbiAgICBjb25zdCB2YWxpZGF0ZUlucHV0Q2hvaWNlID0gbmV3IHNmbi5DaG9pY2UodGhpcywgJ2lzIFZhbGlkIFBhcmFtZXRlcnM/Jywge1xuICAgICAgb3V0cHV0UGF0aDogJyQudmFsaWRhdGlvbl9yZXN1bHQuUGF5bG9hZC5maWxlcycsXG4gICAgfSk7XG5cbiAgICBjb25zdCBzdW1tYXJ5ZnJvbUNhY2hlQ2hvaWNlID0gbmV3IHNmbi5DaG9pY2UodGhpcywgJ2lzIFN1bW1hcnkgaW4gQ2FjaGU/Jywge1xuICAgIH0pO1xuXG4gICAgY29uc3QgZmlsZVN0YXR1c0ZvclN1bW1hcml6YXRpb24gPSBuZXcgc2ZuLkNob2ljZSh0aGlzLCAnaXMgZmlsZSBzdGF0dXMgcmVhZHkgZm9yIHN1bW1hcml6YXRpb24/Jywge1xuICAgICAgb3V0cHV0UGF0aDogJyQuZG9jdW1lbnRfcmVzdWx0LlBheWxvYWQnLFxuICAgIH0pO1xuXG5cbiAgICAvLyBzdGVwIGZ1bmN0aW9uLCBydW4gZmlsZXMgaW4gcGFyYWxsZWxcbiAgICBjb25zdCBydW5GaWxlc0lucGFyYWxsZWwgPSBuZXcgc2ZuLk1hcCh0aGlzLCAnUnVuIEZpbGVzIGluIFBhcmFsbGVsJywge1xuICAgICAgbWF4Q29uY3VycmVuY3k6IDEwMCxcbiAgICB9KS5pdGVyYXRvcihcbiAgICAgIGRvY3VtZW50UmVhZGVyVGFzay5uZXh0KFxuICAgICAgICBzdW1tYXJ5ZnJvbUNhY2hlQ2hvaWNlXG4gICAgICAgICAgLndoZW4oXG4gICAgICAgICAgICBzZm4uQ29uZGl0aW9uLmJvb2xlYW5FcXVhbHMoJyQuZG9jdW1lbnRfcmVzdWx0LlBheWxvYWQuaXNfc3VtbWFyeV9hdmFpbGFibGUnLCB0cnVlKSxcbiAgICAgICAgICAgIGpvYlN1Y2Nlc3MsXG4gICAgICAgICAgKS5vdGhlcndpc2UoXG4gICAgICAgICAgICBmaWxlU3RhdHVzRm9yU3VtbWFyaXphdGlvbi53aGVuKFxuICAgICAgICAgICAgICBzZm4uQ29uZGl0aW9uLnN0cmluZ01hdGNoZXMoJyQuZG9jdW1lbnRfcmVzdWx0LlBheWxvYWQuc3RhdHVzJywgJ0Vycm9yJyksXG4gICAgICAgICAgICAgIGpvYlN1Y2Nlc3MsXG4gICAgICAgICAgICApLm90aGVyd2lzZShcbiAgICAgICAgICAgICAgZ2VuZXJhdGVTdW1tYXJ5VGFzay5uZXh0KGpvYlN1Y2Nlc3MpLFxuICAgICAgICAgICAgKSxcbiAgICAgICAgICApLFxuICAgICAgKSxcbiAgICApO1xuXG4gICAgY29uc3QgbWF4TG9nR3JvdXBOYW1lTGVuZ3RoID0gMjU1O1xuICAgIGNvbnN0IGxvZ0dyb3VwUHJlZml4ID0gJy9hd3MvdmVuZGVkbG9ncy9zdGF0ZXMvY29uc3RydWN0cy8nO1xuICAgIGNvbnN0IG1heEdlbmVyYXRlZE5hbWVMZW5ndGggPSBtYXhMb2dHcm91cE5hbWVMZW5ndGggLSBsb2dHcm91cFByZWZpeC5sZW5ndGg7XG4gICAgY29uc3QgbmFtZVBhcnRzOiBzdHJpbmdbXSA9IFtcbiAgICAgIFN0YWNrLm9mKHNjb3BlKS5zdGFja05hbWUsIC8vIE5hbWUgb2YgdGhlIHN0YWNrXG4gICAgICBzY29wZS5ub2RlLmlkLCAvLyBDb25zdHJ1Y3QgSURcbiAgICAgICdTdGF0ZU1hY2hpbmVMb2dTdW1tYXJpemF0aW9uJywgLy8gTGl0ZXJhbCBzdHJpbmcgZm9yIGxvZyBncm91cCBuYW1lIHBvcnRpb25cbiAgICBdO1xuICAgIGNvbnN0IGxvZ0dyb3VwTmFtZSA9IGdlbmVyYXRlUGh5c2ljYWxOYW1lKGxvZ0dyb3VwUHJlZml4LCBuYW1lUGFydHMsIG1heEdlbmVyYXRlZE5hbWVMZW5ndGgpO1xuICAgIGNvbnN0IHN1bW1hcml6YXRpb25Mb2dHcm91cCA9IG5ldyBsb2dzLkxvZ0dyb3VwKHRoaXMsICdzdW1tYXJpemF0aW9uTG9nR3JvdXAnLCB7XG4gICAgICBsb2dHcm91cE5hbWU6IGxvZ0dyb3VwTmFtZSxcbiAgICB9KTtcblxuICAgIC8vIHN0ZXAgZnVuY3Rpb24gZGVmaW5pdGlvblxuICAgIGNvbnN0IGRlZmluaXRpb24gPSBpbnB1dFZhbGlkYXRpb25UYXNrLm5leHQoXG4gICAgICB2YWxpZGF0ZUlucHV0Q2hvaWNlXG4gICAgICAgIC53aGVuKFxuICAgICAgICAgIHNmbi5Db25kaXRpb24uYm9vbGVhbkVxdWFscygnJC52YWxpZGF0aW9uX3Jlc3VsdC5QYXlsb2FkLmlzVmFsaWQnLCBmYWxzZSksXG4gICAgICAgICAgam9iRmFpbGVkLFxuICAgICAgICApXG4gICAgICAgIC5vdGhlcndpc2UocnVuRmlsZXNJbnBhcmFsbGVsKSxcbiAgICApO1xuXG4gICAgLy8gc3RlcCBmdW5jdGlvblxuXG4gICAgY29uc3Qgc3VtbWFyaXphdGlvblN0ZXBGdW5jdGlvbiA9IG5ldyBzZm4uU3RhdGVNYWNoaW5lKHRoaXMsICdzdW1tYXJpemF0aW9uU3RlcEZ1bmN0aW9uJywge1xuICAgICAgZGVmaW5pdGlvbkJvZHk6IHNmbi5EZWZpbml0aW9uQm9keS5mcm9tQ2hhaW5hYmxlKGRlZmluaXRpb24pLFxuICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICBsb2dzOiB7XG4gICAgICAgIGRlc3RpbmF0aW9uOiBzdW1tYXJpemF0aW9uTG9nR3JvdXAsXG4gICAgICAgIGxldmVsOiBzZm4uTG9nTGV2ZWwuQUxMLFxuICAgICAgfSxcbiAgICAgIHRyYWNpbmdFbmFibGVkOiBlbmFibGVfeHJheSxcbiAgICB9KTtcbiAgICB0aGlzLnN0YXRlTWFjaGluZT1zdW1tYXJpemF0aW9uU3RlcEZ1bmN0aW9uO1xuICAgIC8vIGV2ZW50IGJyaWRnZSBkYXRhc291cmNlIGZvciBzdW1tYXJpemF0aW9uIGFwaVxuICAgIGNvbnN0IGV2ZW50QnJpZGdlRGF0YVNvdXJjZSA9IHN1bW1hcml6YXRpb25HcmFwaHFsQXBpLmFkZEV2ZW50QnJpZGdlRGF0YVNvdXJjZShcbiAgICAgICdldmVudEJyaWRnZURhdGFTb3VyY2UnLFxuICAgICAgdGhpcy5ldmVudEJyaWRnZUJ1cyxcbiAgICApO1xuXG4gICAgdGhpcy5ldmVudEJyaWRnZUJ1cy5ncmFudFB1dEV2ZW50c1RvKGV2ZW50QnJpZGdlRGF0YVNvdXJjZS5ncmFudFByaW5jaXBhbCk7XG5cbiAgICAvLyBhZGQgcmVzb2x2ZXIgb24gc3VtbWFyeSBncmFwaHFsIHNjaGVtYVxuICAgIGV2ZW50QnJpZGdlRGF0YVNvdXJjZS5jcmVhdGVSZXNvbHZlcignZ2VuZXJhdGVTdW1tYXJ5Jywge1xuICAgICAgdHlwZU5hbWU6ICdNdXRhdGlvbicsXG4gICAgICBmaWVsZE5hbWU6ICdnZW5lcmF0ZVN1bW1hcnknLFxuICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogYXBwc3luYy5NYXBwaW5nVGVtcGxhdGUuZnJvbVN0cmluZyhcbiAgICAgICAgYFxuICAgICAgICAge1xuICAgICAgICAgIFwidmVyc2lvblwiOiBcIjIwMTgtMDUtMjlcIixcbiAgICAgICAgICBcIm9wZXJhdGlvblwiOiBcIlB1dEV2ZW50c1wiLFxuICAgICAgICAgIFwiZXZlbnRzXCI6IFt7XG4gICAgICAgICAgICAgIFwic291cmNlXCI6IFwic3VtbWFyeVwiLFxuICAgICAgICAgICAgICBcImRldGFpbFwiOiAkdXRpbC50b0pzb24oJGNvbnRleHQuYXJndW1lbnRzKSxcbiAgICAgICAgICAgICAgXCJkZXRhaWxUeXBlXCI6IFwiZ2VuQUlkZW1vXCJcbiAgICAgICAgICB9XG4gICAgICAgICAgXVxuICB9IFxuICBgLFxuICAgICAgKSxcblxuICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IGFwcHN5bmMuTWFwcGluZ1RlbXBsYXRlLmZyb21TdHJpbmcoXG4gICAgICAgICcjaWYoJGN0eC5lcnJvcikkdXRpbGVycm9yKCRjdHguZXJyb3IubWVzc2FnZSwgJGN0eC5lcnJvci50eXBlLCAkY3R4LnJlc3VsdCkgI2VuZCAkdXRpbC50b0pzb24oJGN0eC5yZXN1bHQpJyxcbiAgICAgICksXG4gICAgfSk7XG5cbiAgICBuZXcgZXZlbnRzLlJ1bGUodGhpcywgJ1N1bW1hcnlNdXRhdGlvblJ1bGUnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ1N1bW1hcnkgTXV0YXRpb24gUnVsZScsXG4gICAgICBldmVudEJ1czogdGhpcy5ldmVudEJyaWRnZUJ1cyxcbiAgICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgICBzb3VyY2U6IFsnc3VtbWFyeSddLFxuICAgICAgfSxcbiAgICAgIHRhcmdldHM6IFtcbiAgICAgICAgbmV3IHRhcmdldHMuU2ZuU3RhdGVNYWNoaW5lKHN1bW1hcml6YXRpb25TdGVwRnVuY3Rpb24sIHtcbiAgICAgICAgICBkZWFkTGV0dGVyUXVldWU6IGRscSxcbiAgICAgICAgICByZXRyeUF0dGVtcHRzOiAxLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgfVxufVxuIl19