"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RagAppsyncStepfnOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const stepfn = require("aws-cdk-lib/aws-stepfunctions");
const stepfn_task = require("aws-cdk-lib/aws-stepfunctions-tasks");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
   * @summary The RagAppsyncStepfnOpensearch class.
   */
class RagAppsyncStepfnOpensearch extends constructs_1.Construct {
    /**
       * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
       * @param {cdk.App} scope - represents the scope for all the resources.
       * @param {string} id - this is a a scope-unique id.
       * @param {RagAppsyncStepfnOpensearchProps} props - user provided props for the construct.
       * @since 0.0.0
       * @access public
       */
    constructor(scope, id, props) {
        super(scope, id);
        // stage
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        // observability
        let lambda_tracing = lambda.Tracing.ACTIVE;
        let enable_xray = true;
        let api_log_config = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability == false) {
            enable_xray = false;
            lambda_tracing = lambda.Tracing.DISABLED;
            api_log_config = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        ;
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingProcessedAssetsBucketObj,
            bucketProps: props.bucketProcessedAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'ingestionConstructLogGroup');
        const role = new iam.Role(this, 'ingestionConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            versioned: true,
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(90),
                }],
        });
        // Bucket containing the inputs assets (documents - multiple modalities) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsBucket' + stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-assets-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsBucket' + stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // Bucket containing the processed assets (documents - text format) uploaded by the user
        let processedAssetsBucket;
        if (!props.existingProcessedAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'processed-assets-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [{
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        }],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'processedAssetsBucket' + stage, props.bucketProcessedAssetsProps);
            }
            processedAssetsBucket = tmpBucket;
            this.s3ProcessedAssetsBucket = tmpBucket;
        }
        else {
            processedAssetsBucket = props.existingProcessedAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3ProcessedAssetsBucketInterface = processedAssetsBucket;
        // GraphQL API
        const ingestion_graphql_api = new appsync.GraphqlApi(this, 'ingestionGraphqlApi', {
            name: 'ingestionGraphqlApi' + stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-rag-appsync-stepfn-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: enable_xray,
            logConfig: api_log_config,
        });
        this.graphqlApi = ingestion_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi ? ingestion_graphql_api.graphqlUrl : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi ? ingestion_graphql_api.apiId : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceIngestion', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateIngestionJobStatusResolver', {
            fieldName: 'updateIngestionJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2017-02-28",
                        "payload": $util.toJson($context.args)
                    }
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.ingestionBus = new events.EventBus(this, 'ingestionEventBus' + stage, {
                eventBusName: 'ingestionEventBus' + stage,
            });
        }
        else {
            this.ingestionBus = props.existingBusInterface;
        }
        // create httpdatasource with ingestion_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('ingestionEventBridgeDataSource' + stage, this.ingestionBus, {
            name: 'ingestionEventBridgeDataSource' + stage,
        });
        // Lambda function used to validate inputs in the step function
        const validate_input_function = new lambda.DockerImageFunction(this, 'lambda_function_validation_input' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/input_validation/src')),
            functionName: 'ingestion_input_validation_docker' + stage,
            description: 'Lambda function for validating input files formats',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        // Add GraphQl permissions to the IAM role for the Lambda function
        validate_input_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // The lambda will pull documents from the input bucket, transform them, and upload
        // the artifacts to the processed bucket
        // we don't use grant read here since it has no effect in case of existing buckets provided by the user
        const s3_transformer_job_function_role = new iam.Role(this, 's3_transformer_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:PutObjectRetention',
                's3:List*',
                's3:GetBucket*',
                's3:Abort*',
                's3:DeleteObject*',
                's3:PutObjectLegalHold',
                's3:PutObjectTagging',
                's3:PutObjectVersionTagging',
                's3:PutObject',
                's3:GetObject*'],
            resources: [
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        // Add GraphQl permissions to the IAM role for the Lambda function
        s3_transformer_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(s3_transformer_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const s3_transformer_job_function = new lambda.DockerImageFunction(this, 'lambda_function_s3_file_transformer' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/s3_file_transformer/src')),
            functionName: 's3_file_transformer_docker' + stage,
            description: 'Lambda function for converting files from their input format to text',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: s3_transformer_job_function_role,
            environment: {
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
            },
        });
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        const embeddings_job_function_role = new iam.Role(this, 'embeddings_job_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`],
                        })],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:DescribeNetworkInterfaces',
            ],
            resources: [
                '*',
            ],
        }));
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                's3:GetObject',
                's3:GetObject*',
                's3:GetBucket*',
                's3:List*',
            ],
            resources: [
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' + this.s3ProcessedAssetsBucketInterface?.bucketName + '/*',
            ],
        }));
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:*'],
            resources: [
                'arn:aws:es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName + '/*',
                'arn:aws:es:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':domain/' + props.existingOpensearchDomain.domainName,
            ],
        }));
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        embeddings_job_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['bedrock:*'],
            resources: [
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(embeddings_job_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(embeddings_job_function_role);
        }
        // Lambda function performing the embedding job
        const embeddings_job_function = new lambda.DockerImageFunction(this, 'lambda_function_embeddings_job' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-rag-appsync-stepfn-opensearch/embeddings_job/src')),
            functionName: 'embeddings_job_docker' + stage,
            description: 'Lambda function for creating documents chunks, embeddings and storing them in Amazon Opensearch',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: (0, utils_1.lambdaMemorySizeLimiter)(this, 1769 * 4),
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: embeddings_job_function_role,
            environment: {
                OUTPUT_BUCKET: this.s3ProcessedAssetsBucketInterface.bucketName,
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_DOMAIN_ENDPOINT: props.existingOpensearchDomain.domainEndpoint,
                OPENSEARCH_SECRET_ID: SecretId,
            },
        });
        const enableOperationalMetric = props.enableOperationalMetric !== undefined && props.enableOperationalMetric !== null ? props.enableOperationalMetric : true;
        if (enableOperationalMetric) {
            const solutionId = `genai_cdk_${utils_1.version}/${this.constructor.name}/${id}`;
            embeddings_job_function.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
            s3_transformer_job_function.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
            validate_input_function.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
        }
        ;
        // Add GraphQl permissions to the IAM role for the Lambda function
        embeddings_job_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'appsync:GraphQL',
            ],
            resources: [
                'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + updateGraphQlApiId + '/*',
            ],
        }));
        // Step function definition
        const inputValidationTask = new stepfn_task.LambdaInvoke(this, 'Validate Ingestion Input', {
            lambdaFunction: validate_input_function,
            resultPath: '$.validation_result',
        });
        const fileTransformationTask = new stepfn_task.LambdaInvoke(this, 'Download and transform document to raw text', {
            lambdaFunction: s3_transformer_job_function,
            resultPath: '$.s3_transformer_result',
        });
        const embeddingsTask = new stepfn_task.LambdaInvoke(this, 'Generate embeddings from processed documents and store them', {
            lambdaFunction: embeddings_job_function,
            resultPath: '$',
        });
        const validate_input_choice = new stepfn.Choice(this, 'Is Valid Ingestion Parameters?', {
            outputPath: '$.validation_result.Payload.files',
        });
        const run_files_in_parallel = new stepfn.Map(this, 'Map State', {
            maxConcurrency: 100,
        }).iterator(fileTransformationTask);
        const jobFailed = new stepfn.Fail(this, 'Job Failed', {
            cause: 'Validation job failed',
            error: 'DescribeJob returned FAILED',
        });
        const definition = inputValidationTask.next(validate_input_choice.when(stepfn.Condition.booleanEquals('$.validation_result.Payload.isValid', false), jobFailed).otherwise(run_files_in_parallel.next(embeddingsTask)));
        const maxLogGroupNameLength = 255;
        const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
        const maxGeneratedNameLength = maxLogGroupNameLength - logGroupPrefix.length;
        const nameParts = [
            aws_cdk_lib_1.Stack.of(scope).stackName,
            scope.node.id,
            'StateMachineLogRag', // Literal string for log group name portion
        ];
        const logGroupName = (0, utils_1.generatePhysicalName)(logGroupPrefix, nameParts, maxGeneratedNameLength);
        const ragLogGroup = new logs.LogGroup(this, 'ingestionStepFunctionLogGroup', {
            logGroupName: logGroupName,
        });
        const ingestion_step_function = new stepfn.StateMachine(this, 'IngestionStateMachine', {
            stateMachineName: 'IngestionStateMachine' + stage,
            definitionBody: stepfn.DefinitionBody.fromChainable(definition),
            timeout: aws_cdk_lib_1.Duration.minutes(30),
            logs: {
                destination: ragLogGroup,
                level: stepfn.LogLevel.ALL,
            },
            tracingEnabled: enable_xray,
        });
        this.stateMachine = ingestion_step_function;
        this.ingestionBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('ingestDocumentResolver', {
            fieldName: 'ingestDocuments',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                    {
                        "version": "2018-05-29",
                        "operation": "PutEvents",
                        "events": [{
                            "source": "ingestion",
                            "detail": $util.toJson($context.arguments),
                            "detailType": "genAIdemo"
                        }
                        ]
                    } 
                    `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                    #if($ctx.error)
                        $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                    #end
                        $util.toJson($ctx.result)
                    `),
        });
        const rule = new events.Rule(this, 'ingestionRule' + stage, {
            description: 'Rule to trigger ingestion function',
            eventBus: this.ingestionBus,
            eventPattern: {
                source: ['ingestion'],
            },
        });
        rule.addTarget(new targets.SfnStateMachine(this.stateMachine));
    }
}
exports.RagAppsyncStepfnOpensearch = RagAppsyncStepfnOpensearch;
_a = JSII_RTTI_SYMBOL_1;
RagAppsyncStepfnOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.RagAppsyncStepfnOpensearch", version: "0.1.34" };
//# sourceMappingURL=data:application/json;base64,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