"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QaAppsyncOpensearch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const s3_bucket_helper = require("../../../common/helpers/s3-bucket-helper");
const utils_1 = require("../../../common/helpers/utils");
const vpc_helper = require("../../../common/helpers/vpc-helper");
/**
 * @summary The QaAppsyncOpensearch class.
 */
class QaAppsyncOpensearch extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the RagAppsyncStepfnOpensearch class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {QaAppsyncOpensearchProps} props - user provided props for the construct.
     * @since 0.0.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // stage
        let stage = '-dev';
        if (props?.stage) {
            stage = props.stage;
        }
        // observability
        let lambda_tracing = lambda.Tracing.ACTIVE;
        let enable_xray = true;
        let api_log_config = {
            fieldLogLevel: appsync.FieldLogLevel.ALL,
            retention: logs.RetentionDays.TEN_YEARS,
        };
        if (props.observability == false) {
            enable_xray = false;
            lambda_tracing = lambda.Tracing.DISABLED;
            api_log_config = {
                fieldLogLevel: appsync.FieldLogLevel.NONE,
                retention: logs.RetentionDays.TEN_YEARS,
            };
        }
        vpc_helper.CheckVpcProps(props);
        s3_bucket_helper.CheckS3Props({
            existingBucketObj: props.existingInputAssetsBucketObj,
            bucketProps: props.bucketInputsAssetsProps,
        });
        if (props?.existingVpc) {
            this.vpc = props.existingVpc;
        }
        else {
            this.vpc = new ec2.Vpc(this, 'Vpc', props.vpcProps);
        }
        // Security group
        if (props?.existingSecurityGroup) {
            this.securityGroup = props.existingSecurityGroup;
        }
        else {
            this.securityGroup = new ec2.SecurityGroup(this, 'securityGroup', {
                vpc: this.vpc,
                allowAllOutbound: true,
                securityGroupName: 'securityGroup' + stage,
            });
        }
        // vpc flowloggroup
        const logGroup = new logs.LogGroup(this, 'qaConstructLogGroup');
        const role = new iam.Role(this, 'qaConstructRole', {
            assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
        });
        // vpc flowlogs
        new ec2.FlowLog(this, 'FlowLog', {
            resourceType: ec2.FlowLogResourceType.fromVpc(this.vpc),
            destination: ec2.FlowLogDestination.toCloudWatchLogs(logGroup, role),
        });
        // bucket for storing server access logging
        const serverAccessLogBucket = new s3.Bucket(this, 'serverAccessLogBucket' + stage, {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            versioned: true,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(90),
                },
            ],
        });
        // Bucket containing the inputs assets (documents - text format) uploaded by the user
        let inputAssetsBucket;
        if (!props.existingInputAssetsBucketObj) {
            let tmpBucket;
            if (!props.bucketInputsAssetsProps) {
                tmpBucket = new s3.Bucket(this, 'inputAssetsQABucket' + stage, {
                    blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
                    encryption: s3.BucketEncryption.S3_MANAGED,
                    bucketName: 'input-asset-qa-bucket' + stage + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    serverAccessLogsBucket: serverAccessLogBucket,
                    enforceSSL: true,
                    versioned: true,
                    lifecycleRules: [
                        {
                            expiration: aws_cdk_lib_1.Duration.days(90),
                        },
                    ],
                });
            }
            else {
                tmpBucket = new s3.Bucket(this, 'InputAssetsQABucket' + stage, props.bucketInputsAssetsProps);
            }
            inputAssetsBucket = tmpBucket;
            this.s3InputAssetsBucket = tmpBucket;
        }
        else {
            inputAssetsBucket = props.existingInputAssetsBucketObj;
        }
        // this is the one we manipulate, we know it exists
        this.s3InputAssetsBucketInterface = inputAssetsBucket;
        // GraphQL API
        const question_answering_graphql_api = new appsync.GraphqlApi(this, 'questionAnsweringGraphqlApi', {
            name: 'questionAnsweringGraphqlApi' + stage,
            definition: appsync.Definition.fromFile(path.join(__dirname, '../../../../resources/gen-ai/aws-qa-appsync-opensearch/schema.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool: props.cognitoUserPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.IAM,
                    },
                ],
            },
            xrayEnabled: enable_xray,
            logConfig: api_log_config,
        });
        this.graphqlApi = question_answering_graphql_api;
        // If the user provides a mergedApi endpoint, the lambda
        // functions will use this endpoint to send their status updates
        const updateGraphQlApiEndpoint = !props.existingMergedApi
            ? question_answering_graphql_api.graphqlUrl
            : props.existingMergedApi.attrGraphQlUrl;
        const updateGraphQlApiId = !props.existingMergedApi
            ? question_answering_graphql_api.apiId
            : props.existingMergedApi.attrApiId;
        const job_status_data_source = new appsync.NoneDataSource(this, 'NoneDataSourceQuestionAnswering', {
            api: this.graphqlApi,
            name: 'JobStatusDataSource',
        });
        job_status_data_source.createResolver('updateQAJobStatusResolver', {
            fieldName: 'updateQAJobStatus',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                          {
                              "version": "2017-02-28",
                              "payload": $util.toJson($context.args)
                          }
                          `),
            responseMappingTemplate: appsync.MappingTemplate.fromString('$util.toJson($context.result)'),
        });
        if (!props.existingBusInterface) {
            this.qaBus = new events.EventBus(this, 'questionAnsweringEventBus' + stage, {
                eventBusName: 'questionAnsweringEventBus' + stage,
            });
        }
        else {
            this.qaBus = props.existingBusInterface;
        }
        // create httpdatasource with question_answering_graphql_api
        const event_bridge_datasource = this.graphqlApi.addEventBridgeDataSource('questionAnsweringEventBridgeDataSource' + stage, this.qaBus, {
            name: 'questionAnsweringEventBridgeDataSource' + stage,
        });
        let SecretId = 'NONE';
        if (props.openSearchSecret) {
            SecretId = props.openSearchSecret.secretName;
        }
        // Lambda function used to validate inputs in the step function
        const question_answering_function_role = new iam.Role(this, 'question_answering_function_role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                'logs:CreateLogGroup',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            resources: [
                                `arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        // Minimum permissions for a Lambda function to execute while accessing a resource within a VPC
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:DeleteNetworkInterface',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
            ],
            resources: [
                'arn:aws:ec2:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':*/*',
            ],
        }));
        // Decribe only works if it's allowed on all resources.
        // Reference: https://docs.aws.amazon.com/lambda/latest/dg/configuration-vpc.html#vpc-permissions
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['ec2:DescribeNetworkInterfaces'],
            resources: ['*'],
        }));
        // The lambda will access the opensearch credentials
        if (props.openSearchSecret) {
            props.openSearchSecret.grantRead(question_answering_function_role);
        }
        // The lambda will pull processed files and create embeddings
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:GetObject*', 's3:GetBucket*', 's3:List*'],
            resources: [
                'arn:aws:s3:::' + this.s3InputAssetsBucketInterface?.bucketName,
                'arn:aws:s3:::' +
                    this.s3InputAssetsBucketInterface?.bucketName +
                    '/*',
            ],
        }));
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:*'],
            resources: [
                'arn:aws:es:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':domain/' +
                    props.existingOpensearchDomain.domainName +
                    '/*',
                'arn:aws:es:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':domain/' +
                    props.existingOpensearchDomain.domainName,
            ],
        }));
        // Add Amazon Bedrock permissions to the IAM role for the Lambda function
        question_answering_function_role.addToPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'bedrock:InvokeModel',
                'bedrock:InvokeModelWithResponseStream',
            ],
            resources: [
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model',
                'arn:aws:bedrock:' + aws_cdk_lib_1.Aws.REGION + '::foundation-model/*',
            ],
        }));
        cdk_nag_1.NagSuppressions.addResourceSuppressions(question_answering_function_role, [
            {
                id: 'AwsSolutions-IAM5',
                reason: 'AWSLambdaBasicExecutionRole is used.',
            },
        ], true);
        const question_answering_function = new lambda.DockerImageFunction(this, 'lambda_question_answering' + stage, {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../../../../lambda/aws-qa-appsync-opensearch/question_answering/src')),
            functionName: 'lambda_question_answering' + stage,
            description: 'Lambda function for question answering',
            vpc: this.vpc,
            tracing: lambda_tracing,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS },
            securityGroups: [this.securityGroup],
            memorySize: 1769 * 4,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            role: question_answering_function_role,
            environment: {
                GRAPHQL_URL: updateGraphQlApiEndpoint,
                INPUT_BUCKET: this.s3InputAssetsBucketInterface.bucketName,
                OPENSEARCH_DOMAIN_ENDPOINT: props.existingOpensearchDomain.domainEndpoint,
                OPENSEARCH_INDEX: props.openSearchIndexName,
                OPENSEARCH_SECRET_ID: SecretId,
            },
            ...(props.lambdaProvisionedConcurrency && {
                currentVersionOptions: {
                    provisionedConcurrentExecutions: props.lambdaProvisionedConcurrency,
                },
            }),
        });
        question_answering_function.currentVersion;
        const enableOperationalMetric = props.enableOperationalMetric !== undefined && props.enableOperationalMetric !== null ? props.enableOperationalMetric : true;
        if (enableOperationalMetric) {
            const solutionId = `genai_cdk_${utils_1.version}/${this.constructor.name}/${id}`;
            question_answering_function.addEnvironment('AWS_SDK_UA_APP_ID', solutionId);
        }
        // Add GraphQl permissions to the IAM role for the Lambda function
        question_answering_function.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['appsync:GraphQL'],
            resources: [
                'arn:aws:appsync:' +
                    aws_cdk_lib_1.Aws.REGION +
                    ':' +
                    aws_cdk_lib_1.Aws.ACCOUNT_ID +
                    ':apis/' +
                    updateGraphQlApiId +
                    '/*',
            ],
        }));
        this.qaBus.grantPutEventsTo(event_bridge_datasource.grantPrincipal);
        event_bridge_datasource.createResolver('QuestionAnsweringResolver', {
            fieldName: 'postQuestion',
            typeName: 'Mutation',
            requestMappingTemplate: appsync.MappingTemplate.fromString(`
                        {
                            "version": "2018-05-29",
                            "operation": "PutEvents",
                            "events": [{
                                "source": "questionanswering",
                                "detail": $util.toJson($context.arguments),
                                "detailType": "Question answering"
                            }
                            ]
                        } 
                        `),
            responseMappingTemplate: appsync.MappingTemplate.fromString(`
                        #if($ctx.error)
                            $util.error($ctx.error.message, $ctx.error.type, $ctx.result)
                        #end
                            $util.toJson($ctx.result)
                        `),
        });
        const rule = new events.Rule(this, 'QuestionAnsweringRule' + stage, {
            description: 'Rule to trigger question answering function',
            eventBus: this.qaBus,
            eventPattern: {
                source: ['questionanswering'],
            },
        });
        rule.addTarget(new targets.LambdaFunction(question_answering_function));
    }
}
exports.QaAppsyncOpensearch = QaAppsyncOpensearch;
_a = JSII_RTTI_SYMBOL_1;
QaAppsyncOpensearch[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.QaAppsyncOpensearch", version: "0.1.34" };
//# sourceMappingURL=data:application/json;base64,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