"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JumpStartSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const core_1 = require("aws-cdk-lib/core");
const jumpstart_constants_1 = require("./private/jumpstart-constants");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
/**
 * @summary The JumpStartSageMakerEndpoint class.
 */
class JumpStartSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.model = props.model;
        this.instanceType = props.instanceType;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.spec = this.model.bind();
        this.region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(this.region)) {
            throw new Error('Region is unresolved. You should explicitly specify the region in the environment.');
        }
        const instanceType = this.verifyInstanceType();
        const instanseBaseType = instanceType.split('.')[1];
        let model;
        if (this.spec.modelPackageArns) {
            if (this.environment) {
                throw new Error('Environment variables are not supported for model packages.');
            }
            model = this.getModelFromPackage(scope, id, props.vpcConfig);
        }
        else {
            const environment = this.buildEnvironment(instanceType);
            model = this.getModelFromArtifact(scope, id, instanceType, instanseBaseType, environment, props.vpcConfig);
        }
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType,
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${this.spec.modelId}-endpoint-${id}`, {
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            endpointName: props.endpointName,
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
    verifyInstanceType() {
        let instanceType = this.spec.defaultInstanceType;
        if (this.instanceType) {
            instanceType = this.instanceType.toString();
        }
        const supportedInstanceTypes = this.spec.instanceTypes;
        if (!supportedInstanceTypes.includes(instanceType)) {
            throw new Error(`The instance type ${instanceType} is not supported. Default instance type: ${this.spec.defaultInstanceType}. Supported instance types: ${supportedInstanceTypes.join(', ')}.`);
        }
        return instanceType;
    }
    buildEnvironment(instanceType) {
        const configEnvironment = this.spec.instanceVariants?.find((v) => v.instanceType === instanceType)?.environment;
        const environment = {
            ...(this.spec.environment ?? {}),
            ...configEnvironment,
            ...this.environment,
        };
        return environment;
    }
    getModelFromArtifact(scope, id, instanceType, instanceBaseType, environment, vpcConfig) {
        const key = this.spec.prepackedArtifactKey ?? this.spec.artifactKey;
        const bucket = jumpstart_constants_1.JumpStartConstants.JUMPSTART_LAUNCHED_REGIONS[this.region]?.contentBucket;
        if (!bucket) {
            throw new Error(`JumpStart is not available in the region ${this.region}.`);
        }
        const modelArtifactUrl = `s3://${bucket}/${key}`;
        const isArtifactCompressed = modelArtifactUrl.endsWith('.tar.gz');
        const imageUriKey = this.spec.instanceVariants
            ?.find((v) => v.instanceType === instanceBaseType)
            ?.imageUri?.replace('$', '');
        if (!imageUriKey) {
            throw new Error(`The image uri is not available for instance type ${instanceType}.`);
        }
        const image = this.spec.instanceAliases?.find((v) => v.region === this.region)?.aliases[imageUriKey];
        if (!image) {
            throw new Error(`The image uri is not available for instance type ${instanceType} in region ${this.region}.`);
        }
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: isArtifactCompressed ? {
                // True: Artifact is a tarball
                image,
                modelDataUrl: modelArtifactUrl,
                environment,
            } : {
                // False: Model is uncompressed
                image,
                modelDataSource: {
                    s3DataSource: {
                        compressionType: 'None',
                        s3DataType: 'S3Prefix',
                        s3Uri: modelArtifactUrl,
                    },
                },
                environment,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
            vpcConfig: vpcConfig,
        });
        return model;
    }
    getModelFromPackage(scope, id, vpcConfig) {
        const modelPackageArns = this.spec.modelPackageArns || {};
        const supportedRegions = Object.keys(modelPackageArns);
        if (!supportedRegions.includes(this.region)) {
            throw new Error(`The model package is not available in the region ${this.region}. Supported regions: ${supportedRegions.join(', ')}.`);
        }
        const modelPackageArn = modelPackageArns[this.region];
        const model = new sagemaker.CfnModel(scope, `${this.spec.modelId}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            enableNetworkIsolation: true,
            primaryContainer: {
                modelPackageName: modelPackageArn,
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.spec.modelId,
                },
                {
                    key: 'modelVersion',
                    value: this.spec.version,
                },
            ],
            vpcConfig: vpcConfig,
        });
        return model;
    }
}
exports.JumpStartSageMakerEndpoint = JumpStartSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
JumpStartSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.JumpStartSageMakerEndpoint", version: "0.1.34" };
//# sourceMappingURL=data:application/json;base64,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