"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomSageMakerEndpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const sagemaker_endpoint_base_1 = require("./sagemaker-endpoint-base");
class CustomSageMakerEndpoint extends sagemaker_endpoint_base_1.SageMakerEndpointBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.instanceType = props.instanceType;
        this.modelId = props.modelId;
        this.instanceCount = Math.max(1, props.instanceCount ?? 1);
        this.role = props.role ?? this.createSageMakerRole();
        this.grantPrincipal = this.role;
        this.modelDataUrl = props.modelDataUrl;
        this.startupHealthCheckTimeoutInSeconds = props.startupHealthCheckTimeoutInSeconds ?? 600;
        this.environment = props.environment;
        this.volumeSizeInGb = props.volumeSizeInGb ?? 512;
        this.modelDataDownloadTimeoutInSeconds = props.modelDataDownloadTimeoutInSeconds ?? 600;
        const image = props.container.bind(this, this.grantPrincipal).imageName;
        const modelIdStr = this.modelId.split('/').join('-').split('.').join('-');
        const isArtifactCompressed = this.modelDataUrl.endsWith('.tar.gz');
        const model = new sagemaker.CfnModel(scope, `${modelIdStr}-model-${id}`, {
            executionRoleArn: this.role.roleArn,
            primaryContainer: isArtifactCompressed ? {
                image,
                mode: 'SingleModel',
                modelDataUrl: this.modelDataUrl,
                environment: {
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    ...this.environment,
                },
            } : {
                image,
                mode: 'SingleModel',
                modelDataSource: {
                    s3DataSource: {
                        compressionType: 'None',
                        s3DataType: 'S3Prefix',
                        s3Uri: this.modelDataUrl,
                    },
                },
                environment: {
                    SAGEMAKER_REGION: cdk.Aws.REGION,
                    ...this.environment,
                },
            },
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
            vpcConfig: props.vpcConfig,
        });
        const endpointConfig = new sagemaker.CfnEndpointConfig(scope, `EndpointConfig-${id}`, {
            productionVariants: [
                {
                    instanceType: this.instanceType.toString(),
                    initialVariantWeight: 1,
                    initialInstanceCount: this.instanceCount,
                    variantName: 'AllTraffic',
                    volumeSizeInGb: this.volumeSizeInGb,
                    modelName: model.getAtt('ModelName').toString(),
                    containerStartupHealthCheckTimeoutInSeconds: this.startupHealthCheckTimeoutInSeconds,
                    modelDataDownloadTimeoutInSeconds: this.modelDataDownloadTimeoutInSeconds,
                },
            ],
        });
        endpointConfig.addDependency(model);
        const endpoint = new sagemaker.CfnEndpoint(scope, `${modelIdStr}-endpoint-${id}`, {
            endpointName: props.endpointName,
            endpointConfigName: endpointConfig.getAtt('EndpointConfigName').toString(),
            tags: [
                {
                    key: 'modelId',
                    value: this.modelId,
                },
            ],
        });
        endpoint.addDependency(endpointConfig);
        this.cfnModel = model;
        this.cfnEndpoint = endpoint;
        this.cfnEndpointConfig = endpointConfig;
        this.endpointArn = endpoint.ref;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['sagemaker:InvokeEndpoint'],
            resourceArns: [this.endpointArn],
        });
    }
}
exports.CustomSageMakerEndpoint = CustomSageMakerEndpoint;
_a = JSII_RTTI_SYMBOL_1;
CustomSageMakerEndpoint[_a] = { fqn: "@cdklabs/generative-ai-cdk-constructs.CustomSageMakerEndpoint", version: "0.1.34" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLXNhZ2VtYWtlci1lbmRwb2ludC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9wYXR0ZXJucy9nZW4tYWkvYXdzLW1vZGVsLWRlcGxveW1lbnQtc2FnZW1ha2VyL2N1c3RvbS1zYWdlbWFrZXItZW5kcG9pbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7Ozs7Ozs7Ozs7R0FXRztBQUNILG1DQUFtQztBQUNuQywyQ0FBMkM7QUFDM0MsdURBQXVEO0FBR3ZELHVFQUFrRTtBQWtCbEUsTUFBYSx1QkFBd0IsU0FBUSwrQ0FBcUI7SUFpQmhFLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBbUM7UUFDM0UsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDdkMsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLGFBQWEsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUMzRCxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7UUFDckQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBQ2hDLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsa0NBQWtDLEdBQUcsS0FBSyxDQUFDLGtDQUFrQyxJQUFJLEdBQUcsQ0FBQztRQUMxRixJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxJQUFJLEdBQUcsQ0FBQztRQUNsRCxJQUFJLENBQUMsaUNBQWlDLEdBQUcsS0FBSyxDQUFDLGlDQUFpQyxJQUFJLEdBQUcsQ0FBQztRQUV4RixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUN4RSxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMxRSxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRW5FLE1BQU0sS0FBSyxHQUFHLElBQUksU0FBUyxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsR0FBRyxVQUFVLFVBQVUsRUFBRSxFQUFFLEVBQUU7WUFDdkUsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ25DLGdCQUFnQixFQUFFLG9CQUFvQixDQUFDLENBQUMsQ0FBQztnQkFDdkMsS0FBSztnQkFDTCxJQUFJLEVBQUUsYUFBYTtnQkFDbkIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO2dCQUMvQixXQUFXLEVBQUU7b0JBQ1gsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNO29CQUNoQyxHQUFHLElBQUksQ0FBQyxXQUFXO2lCQUNwQjthQUNGLENBQUMsQ0FBQyxDQUFDO2dCQUNGLEtBQUs7Z0JBQ0wsSUFBSSxFQUFFLGFBQWE7Z0JBQ25CLGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osZUFBZSxFQUFFLE1BQU07d0JBQ3ZCLFVBQVUsRUFBRSxVQUFVO3dCQUN0QixLQUFLLEVBQUUsSUFBSSxDQUFDLFlBQVk7cUJBQ3pCO2lCQUNGO2dCQUNELFdBQVcsRUFBRTtvQkFDWCxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU07b0JBQ2hDLEdBQUcsSUFBSSxDQUFDLFdBQVc7aUJBQ3BCO2FBQ0Y7WUFDRCxJQUFJLEVBQUU7Z0JBQ0o7b0JBQ0UsR0FBRyxFQUFFLFNBQVM7b0JBQ2QsS0FBSyxFQUFFLElBQUksQ0FBQyxPQUFPO2lCQUNwQjthQUNGO1lBQ0QsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1NBQzNCLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLElBQUksU0FBUyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxFQUFFLEVBQUU7WUFDcEYsa0JBQWtCLEVBQUU7Z0JBQ2xCO29CQUNFLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRTtvQkFDMUMsb0JBQW9CLEVBQUUsQ0FBQztvQkFDdkIsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLGFBQWE7b0JBQ3hDLFdBQVcsRUFBRSxZQUFZO29CQUN6QixjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7b0JBQ25DLFNBQVMsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFFBQVEsRUFBRTtvQkFDL0MsMkNBQTJDLEVBQUUsSUFBSSxDQUFDLGtDQUFrQztvQkFDcEYsaUNBQWlDLEVBQUUsSUFBSSxDQUFDLGlDQUFpQztpQkFDMUU7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILGNBQWMsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFcEMsTUFBTSxRQUFRLEdBQUcsSUFBSSxTQUFTLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxHQUFHLFVBQVUsYUFBYSxFQUFFLEVBQUUsRUFBRTtZQUNoRixZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDaEMsa0JBQWtCLEVBQUUsY0FBYyxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLFFBQVEsRUFBRTtZQUMxRSxJQUFJLEVBQUU7Z0JBQ0o7b0JBQ0UsR0FBRyxFQUFFLFNBQVM7b0JBQ2QsS0FBSyxFQUFFLElBQUksQ0FBQyxPQUFPO2lCQUNwQjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUV2QyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQztRQUN0QixJQUFJLENBQUMsV0FBVyxHQUFHLFFBQVEsQ0FBQztRQUM1QixJQUFJLENBQUMsaUJBQWlCLEdBQUcsY0FBYyxDQUFDO1FBQ3hDLElBQUksQ0FBQyxXQUFXLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztJQUNsQyxDQUFDO0lBRU0sZUFBZSxDQUFDLFNBQThCO1FBQ25ELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ2QsT0FBTztTQUNSO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVNLFdBQVcsQ0FBQyxPQUF1QjtRQUN4QyxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxPQUFPLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQztZQUNyQyxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO1NBQ2pDLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBdkhILDBEQXdIQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgc2FnZW1ha2VyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zYWdlbWFrZXInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDb250YWluZXJJbWFnZSB9IGZyb20gJy4vY29udGFpbmVyLWltYWdlJztcbmltcG9ydCB7IFNhZ2VNYWtlckVuZHBvaW50QmFzZSB9IGZyb20gJy4vc2FnZW1ha2VyLWVuZHBvaW50LWJhc2UnO1xuaW1wb3J0IHsgU2FnZU1ha2VySW5zdGFuY2VUeXBlIH0gZnJvbSAnLi9zYWdlbWFrZXItaW5zdGFuY2UtdHlwZSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tU2FnZU1ha2VyRW5kcG9pbnRQcm9wcyB7XG4gIHJlYWRvbmx5IG1vZGVsSWQ6IHN0cmluZztcbiAgcmVhZG9ubHkgZW5kcG9pbnROYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZTogU2FnZU1ha2VySW5zdGFuY2VUeXBlO1xuICByZWFkb25seSBjb250YWluZXI6IENvbnRhaW5lckltYWdlO1xuICByZWFkb25seSBpbnN0YW5jZUNvdW50PzogbnVtYmVyO1xuICByZWFkb25seSByb2xlPzogaWFtLlJvbGU7XG4gIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcmVhZG9ubHkgc3RhcnR1cEhlYWx0aENoZWNrVGltZW91dEluU2Vjb25kcz86IG51bWJlcjtcbiAgcmVhZG9ubHkgbW9kZWxEYXRhRG93bmxvYWRUaW1lb3V0SW5TZWNvbmRzPzogbnVtYmVyO1xuICByZWFkb25seSB2b2x1bWVTaXplSW5HYj86IG51bWJlcjtcbiAgcmVhZG9ubHkgdnBjQ29uZmlnPzogc2FnZW1ha2VyLkNmbk1vZGVsLlZwY0NvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkO1xuICByZWFkb25seSBtb2RlbERhdGFVcmw6IHN0cmluZztcbn1cblxuZXhwb3J0IGNsYXNzIEN1c3RvbVNhZ2VNYWtlckVuZHBvaW50IGV4dGVuZHMgU2FnZU1ha2VyRW5kcG9pbnRCYXNlIGltcGxlbWVudHMgaWFtLklHcmFudGFibGUge1xuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuICBwdWJsaWMgcmVhZG9ubHkgZW5kcG9pbnRBcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGNmbk1vZGVsOiBzYWdlbWFrZXIuQ2ZuTW9kZWw7XG4gIHB1YmxpYyByZWFkb25seSBjZm5FbmRwb2ludDogc2FnZW1ha2VyLkNmbkVuZHBvaW50O1xuICBwdWJsaWMgcmVhZG9ubHkgY2ZuRW5kcG9pbnRDb25maWc6IHNhZ2VtYWtlci5DZm5FbmRwb2ludENvbmZpZztcblxuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VUeXBlPzogU2FnZU1ha2VySW5zdGFuY2VUeXBlO1xuICBwdWJsaWMgcmVhZG9ubHkgaW5zdGFuY2VDb3VudDogbnVtYmVyO1xuICBwdWJsaWMgcmVhZG9ubHkgcm9sZTogaWFtLlJvbGU7XG4gIHB1YmxpYyByZWFkb25seSBtb2RlbERhdGFVcmw6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IG1vZGVsSWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHZvbHVtZVNpemVJbkdiPzogbnVtYmVyO1xuICBwdWJsaWMgcmVhZG9ubHkgbW9kZWxEYXRhRG93bmxvYWRUaW1lb3V0SW5TZWNvbmRzOiBudW1iZXI7XG4gIHByaXZhdGUgcmVhZG9ubHkgc3RhcnR1cEhlYWx0aENoZWNrVGltZW91dEluU2Vjb25kczogbnVtYmVyO1xuICBwcml2YXRlIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ3VzdG9tU2FnZU1ha2VyRW5kcG9pbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmluc3RhbmNlVHlwZSA9IHByb3BzLmluc3RhbmNlVHlwZTtcbiAgICB0aGlzLm1vZGVsSWQgPSBwcm9wcy5tb2RlbElkO1xuICAgIHRoaXMuaW5zdGFuY2VDb3VudCA9IE1hdGgubWF4KDEsIHByb3BzLmluc3RhbmNlQ291bnQgPz8gMSk7XG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSA/PyB0aGlzLmNyZWF0ZVNhZ2VNYWtlclJvbGUoKTtcbiAgICB0aGlzLmdyYW50UHJpbmNpcGFsID0gdGhpcy5yb2xlO1xuICAgIHRoaXMubW9kZWxEYXRhVXJsID0gcHJvcHMubW9kZWxEYXRhVXJsO1xuICAgIHRoaXMuc3RhcnR1cEhlYWx0aENoZWNrVGltZW91dEluU2Vjb25kcyA9IHByb3BzLnN0YXJ0dXBIZWFsdGhDaGVja1RpbWVvdXRJblNlY29uZHMgPz8gNjAwO1xuICAgIHRoaXMuZW52aXJvbm1lbnQgPSBwcm9wcy5lbnZpcm9ubWVudDtcbiAgICB0aGlzLnZvbHVtZVNpemVJbkdiID0gcHJvcHMudm9sdW1lU2l6ZUluR2IgPz8gNTEyO1xuICAgIHRoaXMubW9kZWxEYXRhRG93bmxvYWRUaW1lb3V0SW5TZWNvbmRzID0gcHJvcHMubW9kZWxEYXRhRG93bmxvYWRUaW1lb3V0SW5TZWNvbmRzID8/IDYwMDtcblxuICAgIGNvbnN0IGltYWdlID0gcHJvcHMuY29udGFpbmVyLmJpbmQodGhpcywgdGhpcy5ncmFudFByaW5jaXBhbCkuaW1hZ2VOYW1lO1xuICAgIGNvbnN0IG1vZGVsSWRTdHIgPSB0aGlzLm1vZGVsSWQuc3BsaXQoJy8nKS5qb2luKCctJykuc3BsaXQoJy4nKS5qb2luKCctJyk7XG4gICAgY29uc3QgaXNBcnRpZmFjdENvbXByZXNzZWQgPSB0aGlzLm1vZGVsRGF0YVVybC5lbmRzV2l0aCgnLnRhci5neicpO1xuXG4gICAgY29uc3QgbW9kZWwgPSBuZXcgc2FnZW1ha2VyLkNmbk1vZGVsKHNjb3BlLCBgJHttb2RlbElkU3RyfS1tb2RlbC0ke2lkfWAsIHtcbiAgICAgIGV4ZWN1dGlvblJvbGVBcm46IHRoaXMucm9sZS5yb2xlQXJuLFxuICAgICAgcHJpbWFyeUNvbnRhaW5lcjogaXNBcnRpZmFjdENvbXByZXNzZWQgPyB7XG4gICAgICAgIGltYWdlLFxuICAgICAgICBtb2RlOiAnU2luZ2xlTW9kZWwnLFxuICAgICAgICBtb2RlbERhdGFVcmw6IHRoaXMubW9kZWxEYXRhVXJsLFxuICAgICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICAgIFNBR0VNQUtFUl9SRUdJT046IGNkay5Bd3MuUkVHSU9OLFxuICAgICAgICAgIC4uLnRoaXMuZW52aXJvbm1lbnQsXG4gICAgICAgIH0sXG4gICAgICB9IDoge1xuICAgICAgICBpbWFnZSxcbiAgICAgICAgbW9kZTogJ1NpbmdsZU1vZGVsJyxcbiAgICAgICAgbW9kZWxEYXRhU291cmNlOiB7XG4gICAgICAgICAgczNEYXRhU291cmNlOiB7XG4gICAgICAgICAgICBjb21wcmVzc2lvblR5cGU6ICdOb25lJyxcbiAgICAgICAgICAgIHMzRGF0YVR5cGU6ICdTM1ByZWZpeCcsXG4gICAgICAgICAgICBzM1VyaTogdGhpcy5tb2RlbERhdGFVcmwsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBTQUdFTUFLRVJfUkVHSU9OOiBjZGsuQXdzLlJFR0lPTixcbiAgICAgICAgICAuLi50aGlzLmVudmlyb25tZW50LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHRhZ3M6IFtcbiAgICAgICAge1xuICAgICAgICAgIGtleTogJ21vZGVsSWQnLFxuICAgICAgICAgIHZhbHVlOiB0aGlzLm1vZGVsSWQsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgdnBjQ29uZmlnOiBwcm9wcy52cGNDb25maWcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBlbmRwb2ludENvbmZpZyA9IG5ldyBzYWdlbWFrZXIuQ2ZuRW5kcG9pbnRDb25maWcoc2NvcGUsIGBFbmRwb2ludENvbmZpZy0ke2lkfWAsIHtcbiAgICAgIHByb2R1Y3Rpb25WYXJpYW50czogW1xuICAgICAgICB7XG4gICAgICAgICAgaW5zdGFuY2VUeXBlOiB0aGlzLmluc3RhbmNlVHlwZS50b1N0cmluZygpLFxuICAgICAgICAgIGluaXRpYWxWYXJpYW50V2VpZ2h0OiAxLFxuICAgICAgICAgIGluaXRpYWxJbnN0YW5jZUNvdW50OiB0aGlzLmluc3RhbmNlQ291bnQsXG4gICAgICAgICAgdmFyaWFudE5hbWU6ICdBbGxUcmFmZmljJyxcbiAgICAgICAgICB2b2x1bWVTaXplSW5HYjogdGhpcy52b2x1bWVTaXplSW5HYixcbiAgICAgICAgICBtb2RlbE5hbWU6IG1vZGVsLmdldEF0dCgnTW9kZWxOYW1lJykudG9TdHJpbmcoKSxcbiAgICAgICAgICBjb250YWluZXJTdGFydHVwSGVhbHRoQ2hlY2tUaW1lb3V0SW5TZWNvbmRzOiB0aGlzLnN0YXJ0dXBIZWFsdGhDaGVja1RpbWVvdXRJblNlY29uZHMsXG4gICAgICAgICAgbW9kZWxEYXRhRG93bmxvYWRUaW1lb3V0SW5TZWNvbmRzOiB0aGlzLm1vZGVsRGF0YURvd25sb2FkVGltZW91dEluU2Vjb25kcyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBlbmRwb2ludENvbmZpZy5hZGREZXBlbmRlbmN5KG1vZGVsKTtcblxuICAgIGNvbnN0IGVuZHBvaW50ID0gbmV3IHNhZ2VtYWtlci5DZm5FbmRwb2ludChzY29wZSwgYCR7bW9kZWxJZFN0cn0tZW5kcG9pbnQtJHtpZH1gLCB7XG4gICAgICBlbmRwb2ludE5hbWU6IHByb3BzLmVuZHBvaW50TmFtZSxcbiAgICAgIGVuZHBvaW50Q29uZmlnTmFtZTogZW5kcG9pbnRDb25maWcuZ2V0QXR0KCdFbmRwb2ludENvbmZpZ05hbWUnKS50b1N0cmluZygpLFxuICAgICAgdGFnczogW1xuICAgICAgICB7XG4gICAgICAgICAga2V5OiAnbW9kZWxJZCcsXG4gICAgICAgICAgdmFsdWU6IHRoaXMubW9kZWxJZCxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBlbmRwb2ludC5hZGREZXBlbmRlbmN5KGVuZHBvaW50Q29uZmlnKTtcblxuICAgIHRoaXMuY2ZuTW9kZWwgPSBtb2RlbDtcbiAgICB0aGlzLmNmbkVuZHBvaW50ID0gZW5kcG9pbnQ7XG4gICAgdGhpcy5jZm5FbmRwb2ludENvbmZpZyA9IGVuZHBvaW50Q29uZmlnO1xuICAgIHRoaXMuZW5kcG9pbnRBcm4gPSBlbmRwb2ludC5yZWY7XG4gIH1cblxuICBwdWJsaWMgYWRkVG9Sb2xlUG9saWN5KHN0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCkge1xuICAgIGlmICghdGhpcy5yb2xlKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgdGhpcy5yb2xlLmFkZFRvUG9saWN5KHN0YXRlbWVudCk7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRJbnZva2UoZ3JhbnRlZTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgZ3JhbnRlZSxcbiAgICAgIGFjdGlvbnM6IFsnc2FnZW1ha2VyOkludm9rZUVuZHBvaW50J10sXG4gICAgICByZXNvdXJjZUFybnM6IFt0aGlzLmVuZHBvaW50QXJuXSxcbiAgICB9KTtcbiAgfVxufSJdfQ==