"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getRepositories = exports.generateDLCContainerImages = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const node_child_process_1 = require("node:child_process");
const path = require("path");
const generate_utils_1 = require("./generate-utils");
const regionName = 'us-west-2';
const startVersionRegex = /^\d+\.\d+\.\d+-/;
const versionRegex = /-v\d+(\.\d+)*(-\d{4}-\d{2}-\d{2}-\d{2}-\d{2}-\d{2})?$/;
const repositories = [
    'huggingface-pytorch-tgi-inference',
    'huggingface-pytorch-inference',
    'huggingface-tensorflow-inference',
    'huggingface-pytorch-inference-neuronx',
    'djl-inference',
];
const DEEP_LEARNING_CONTAINER_IMAGE_PATH = path.join(__dirname, '../deep-learning-container-image.ts');
async function generateDLCContainerImages() {
    console.log('Getting DLC container image data');
    const output = (0, node_child_process_1.execSync)(`aws ecr get-authorization-token --output text --query 'authorizationData[].authorizationToken' --region ${regionName}`);
    const token = output.toString();
    const repositoryTagData = {};
    for (const repositoryName of repositories) {
        console.log(repositoryName);
        const pageSize = 1000;
        const hostname = '763104351884.dkr.ecr.us-west-2.amazonaws.com';
        let link = `/v2/${repositoryName}/tags/list?n=${pageSize}`;
        let tags = [];
        while (link) {
            const [data, response] = await generate_utils_1.GenerateUtils.downloadJSON({
                hostname,
                port: 443,
                method: 'GET',
                path: link,
                headers: {
                    Authorization: `Basic ${token.trim()}`,
                },
            });
            tags.push(...data.tags);
            link = response.headers.link;
            if (link) {
                console.log(link);
                link = link.substring(1, link.indexOf('>')).split(hostname)[1];
                console.log('Link:', link);
            }
        }
        tags = tags.filter((tag) => startVersionRegex.test(tag));
        if (repositoryName == 'huggingface-tensorflow-inference' ||
            repositoryName == 'huggingface-pytorch-inference' ||
            repositoryName == 'huggingface-pytorch-inference-neuronx') {
            tags = tags.filter((tag) => tag.includes('-transformers'));
        }
        else if (repositoryName == 'huggingface-pytorch-tgi-inference') {
            tags = tags.filter((tag) => tag.includes('-tgi'));
        }
        else if (repositoryName == 'djl-inference') {
            tags = tags.filter((tag) => tag.includes('-deepspeed') ||
                tag.includes('-fastertransformer') ||
                tag.includes('-neuronx'));
        }
        tags = tags.map((tag) => tag.replace(versionRegex, ''));
        tags = Array.from(new Set(tags)).sort();
        repositoryTagData[repositoryName] = tags;
        console.log(tags);
    }
    generateCode(repositoryTagData);
}
exports.generateDLCContainerImages = generateDLCContainerImages;
async function getRepositories() {
    const repositoryNames = new Set();
    const GITHUB_URL = 'https://raw.githubusercontent.com/aws/sagemaker-python-sdk/master/src/sagemaker/image_uri_config';
    const fileNames = [
        `${GITHUB_URL}/huggingface-llm.json`,
        `${GITHUB_URL}/huggingface.json`,
        `${GITHUB_URL}/huggingface-neuronx.json`,
        `${GITHUB_URL}/djl-deepspeed.json`,
        `${GITHUB_URL}/djl-fastertransformer.json`,
        `${GITHUB_URL}/djl-neuronx.json`,
    ];
    for (const fileName of fileNames) {
        const [data] = await generate_utils_1.GenerateUtils.downloadJSON(fileName);
        console.log('Processing file:', fileName);
        const versions = data.versions || data.inference.versions;
        for (const version of Object.keys(versions)) {
            const versionData = versions[version];
            const versionAliases = versionData.version_aliases;
            const items = [];
            if (versionAliases) {
                for (const versionAlias of Object.keys(versionAliases)) {
                    items.push(versionData[versionAliases[versionAlias]]);
                }
            }
            else {
                items.push(versionData);
            }
            for (const item of items) {
                const repositoryName = item.repository;
                repositoryNames.add(repositoryName);
            }
        }
    }
    console.log('Repositories:', new Array(...repositories));
    return repositoryNames;
}
exports.getRepositories = getRepositories;
function generateCode(repositoryTagData) {
    let imagesStr = '';
    for (const repositoryName of Object.keys(repositoryTagData).sort()) {
        const tags = repositoryTagData[repositoryName].sort();
        const repositoryNameStr = generate_utils_1.GenerateUtils.replaceAll(repositoryName, '-', '_').toUpperCase();
        for (const tagName of tags) {
            const tagNameStr = generate_utils_1.GenerateUtils.replaceAllBatch(tagName, ['\\.', '-'], '_').toUpperCase();
            const name = `${repositoryNameStr}_${tagNameStr}`;
            imagesStr +=
                '  ' +
                    `public static readonly ${name} = this.fromDeepLearningContainerImage('${repositoryName}', '${tagName}');\n`;
        }
    }
    const fileStr = `/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import { Stack } from 'aws-cdk-lib';
import * as ecr from 'aws-cdk-lib/aws-ecr';
import * as iam from 'aws-cdk-lib/aws-iam';
import { FactName } from 'aws-cdk-lib/region-info';
import { Construct } from 'constructs';
import { ContainerImage, ContainerImageConfig } from './container-image';

export class DeepLearningContainerImage extends ContainerImage {
${imagesStr}

  public static fromDeepLearningContainerImage(
    repositoryName: string,
    tag: string,
    accountId?: string,
  ): ContainerImage {
    return new DeepLearningContainerImage(repositoryName, tag, accountId);
  }

  constructor(
    private readonly repositoryName: string,
    private readonly tag: string,
    private readonly accountId?: string,
  ) {
    super();
  }

  public bind(
    scope: Construct,
    grantable: iam.IGrantable,
  ): ContainerImageConfig {
    const accountId =
      this.accountId ??
      Stack.of(scope).regionalFact(FactName.DLC_REPOSITORY_ACCOUNT);

    const repository = ecr.Repository.fromRepositoryAttributes(
      scope,
      'DeepLearningContainerRepository',
      {
        repositoryName: this.repositoryName,
        repositoryArn: ecr.Repository.arnForLocalRepository(
          this.repositoryName,
          scope,
          accountId,
        ),
      },
    );

    repository.grantPull(grantable);

    return { imageName: repository.repositoryUri + ':' + this.tag };
  }
}
`;
    generate_utils_1.GenerateUtils.writeFileSyncWithDirs(DEEP_LEARNING_CONTAINER_IMAGE_PATH, fileStr);
}
//# sourceMappingURL=data:application/json;base64,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