"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.lambdaMemorySizeLimiter = exports.recommendedMaximumLambdaMemorySize = exports.maximumLambdaMemorySizeContextItem = exports.generatePhysicalName = exports.version = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
/**
 * The version of this package
 */
// eslint-disable-next-line @typescript-eslint/no-require-imports
exports.version = require('../../../package.json').version;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @summary Creates a physical resource name in the style of the CDK (string+hash) - this value incorporates Stack ID or
 * the Stack Name, so it will remain static in multiple updates of a single stack, but will be different in a separate
 * stack instance.
 * @param {string} prefix - the prefix to use for the name
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @param {boolean} lower - whether to return the name in lowercase or mixed case
 * @param {IConstruct} resource - the resource that is calling this function (used to extract the stack Name and Node ID)
 * @returns {string} - a string with concatenated parts (truncated if necessary) + a hash of the full concatenated parts
 *
 */
function generatePhysicalName(prefix, parts, maxLength, lower, resource) {
    // The result will consist of:
    //    -The prefix - unaltered
    //    -The parts concatenated, but reduced in size to meet the maxLength limit for the overall name
    //    -A hyphen delimiter
    //    -The GUID portion of the stack arn
    const stackIdGuidLength = 36;
    const prefixLength = prefix.length;
    let maxPartsLength = maxLength - prefixLength - 1 - stackIdGuidLength; // 1 is the hyphen
    let uniqueStackIdPart = '';
    let uniqueStackIdPartLength = stackIdGuidLength;
    if (maxPartsLength > 2) {
        // Extract the Stack ID Guid
        uniqueStackIdPart = cdk.Fn.select(2, cdk.Fn.split('/', `${cdk.Aws.STACK_ID}`));
    }
    else if (resource) {
        const stack = cdk.Stack.of(resource);
        const hashLength = 8;
        const sha256 = crypto.createHash('sha256')
            .update(stack.stackName)
            .update(cdk.Names.nodeUniqueId(resource.node));
        uniqueStackIdPart = sha256.digest('hex').slice(0, hashLength);
        uniqueStackIdPartLength = hashLength;
        maxPartsLength = maxLength - prefixLength - 1 - hashLength; // 1 is the hyphen
    }
    else {
        throw new Error('The resource parameter is required for short names.');
    }
    let allParts = '';
    parts.forEach((part) => {
        allParts += part;
    });
    if (allParts.length > maxPartsLength) {
        const subStringLength = maxPartsLength / 2;
        allParts = allParts.substring(0, subStringLength) + allParts.substring(allParts.length - subStringLength);
    }
    if (prefix.length + allParts.length + uniqueStackIdPartLength + 1 /* hyphen */ > maxLength) {
        throw new Error(`The generated name is longer than the maximum length of ${maxLength}`);
    }
    return prefix.toLowerCase() + (lower ? allParts.toLowerCase() : allParts) + '-' + uniqueStackIdPart;
}
exports.generatePhysicalName = generatePhysicalName;
exports.maximumLambdaMemorySizeContextItem = 'maximumLambdaMemorySize';
exports.recommendedMaximumLambdaMemorySize = 7076;
function lambdaMemorySizeLimiter(construct, requestedMemorySizeInMegabytes) {
    const maximumLambaMemorySize = construct.node.tryGetContext(exports.maximumLambdaMemorySizeContextItem) === undefined ?
        exports.recommendedMaximumLambdaMemorySize :
        parseInt(construct.node.tryGetContext(exports.maximumLambdaMemorySizeContextItem));
    if (maximumLambaMemorySize < exports.recommendedMaximumLambdaMemorySize) {
        console.warn(`Maximum Lambda memorySize, ${maximumLambaMemorySize}, is less than the recommended ${exports.recommendedMaximumLambdaMemorySize}.`);
    }
    if (requestedMemorySizeInMegabytes > maximumLambaMemorySize) {
        console.warn(`Reducing Lambda memorySize, ${requestedMemorySizeInMegabytes} to ${maximumLambaMemorySize} for ${construct.constructor.name}`);
        return maximumLambaMemorySize;
    }
    else {
        return requestedMemorySizeInMegabytes;
    }
}
exports.lambdaMemorySizeLimiter = lambdaMemorySizeLimiter;
//# sourceMappingURL=data:application/json;base64,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