"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Layer = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const child_process_1 = require("child_process");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3assets = require("aws-cdk-lib/aws-s3-assets");
const constructs_1 = require("constructs");
class Layer extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { runtime, architecture, path, description, autoUpgrade, local } = props;
        const args = local ? [] : ['-t /asset-output/python'];
        if (autoUpgrade) {
            args.push('--upgrade');
        }
        const layerAsset = new s3assets.Asset(this, 'LayerAsset', {
            path,
            bundling: local ? {
                // If local is true use the host to install the requirements
                image: runtime.bundlingImage,
                local: {
                    tryBundle(outputDir) {
                        (0, child_process_1.execSync)(`${local} -m venv venv`);
                        (0, child_process_1.execSync)('source venv/bin/activate');
                        (0, child_process_1.execSync)(`pip install -r ${path}/requirements.txt -t ${outputDir}/python ${args.join(' ')}`);
                        return true;
                    },
                },
            } : {
                // Default: Docker is used to install the requirements
                image: runtime.bundlingImage,
                platform: architecture.dockerPlatform,
                command: [
                    'bash',
                    '-c',
                    `pip install -r requirements.txt ${args.join(' ')}`,
                ],
                outputType: cdk.BundlingOutput.AUTO_DISCOVER,
                securityOpt: 'no-new-privileges:true',
                network: 'host',
            },
        });
        const layer = new lambda.LayerVersion(this, 'Layer', {
            code: lambda.Code.fromBucket(layerAsset.bucket, layerAsset.s3ObjectKey),
            compatibleRuntimes: [runtime],
            compatibleArchitectures: [architecture],
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            description: description,
        });
        this.layer = layer;
    }
}
exports.Layer = Layer;
//# sourceMappingURL=data:application/json;base64,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