"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setMergedApiRole = exports.checkAppsyncMergedApiProps = exports.buildMergedAPI = void 0;
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appsync = require("aws-cdk-lib/aws-appsync");
const iam = require("aws-cdk-lib/aws-iam");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * build app sync merge api with source api associations
 *
 * @param AppsyncMergedApiProps The  props to be used by the construct
 * @param  apiType - MERGED, GRAPHQL
 * @returns App sync merge api
 */
function buildMergedAPI(scope, id, props) {
    if (props.existingMergeApi) {
        return props.existingMergeApi;
    }
    else {
        const mergeAPIname = props.cfnGraphQLApiProps?.name || 'appsyncmergeAPI';
        const apiType = props.cfnGraphQLApiProps?.apiType || 'MERGED';
        const fieldLogLevel = props?.fieldLogLevel || appsync.FieldLogLevel.NONE;
        const excludeVerboseContent = props?.excludeVerboseContent || false;
        const xRayEnabled = props?.xRayEnabled || false;
        let mergedApi = new appsync.CfnGraphQLApi(scope, id, {
            apiType: apiType,
            name: mergeAPIname,
            authenticationType: props.cfnGraphQLApiProps.authenticationType,
            userPoolConfig: props.cfnGraphQLApiProps?.userPoolConfig,
            additionalAuthenticationProviders: [{
                    authenticationType: 'AWS_IAM',
                }],
            logConfig: {
                cloudWatchLogsRoleArn: setAppsyncCloudWatchlogsRole(scope, props).roleArn,
                fieldLogLevel: fieldLogLevel,
                excludeVerboseContent: excludeVerboseContent,
            },
            xrayEnabled: xRayEnabled,
            mergedApiExecutionRoleArn: props.mergedApiRole.roleArn,
            ownerContact: props?.cfnGraphQLApiProps.ownerContact,
        });
        return mergedApi;
    }
}
exports.buildMergedAPI = buildMergedAPI;
function checkAppsyncMergedApiProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if (propsObject.existingMergeApi && propsObject.cfnGraphQLApiProps) {
        errorMessages += 'Error - Either provide existingMergeApi or cfnGraphQLApiProps, but not both.\n';
        errorFound = true;
    }
    if (!propsObject.existingMergeApi && !propsObject.cfnGraphQLApiProps) {
        errorMessages += 'Error - Atleast one is required either existingMergeApi or cfnGraphQLApiProps.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.checkAppsyncMergedApiProps = checkAppsyncMergedApiProps;
/**
 * @internal This is an internal core function and should not be called directly
 * by Solutions Constructs clients.
 * set the merge api role to access source api associations
 *
 * @param AppsyncMergedApiProps The  props to be used by the construct
 * @param mergedAPI app sync graphql api
 * @param mergedApiRole iam role
 * @returns App sync merge api role
 */
function setMergedApiRole(mergedApiID, sourceApiId, mergedApiRole) {
    mergedApiRole.addToPolicy(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ['appsync:SourceGraphQL',
            'appsync:StartSchemaMerge'],
        resources: [
            'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID
                + ':apis/' + sourceApiId + '/*',
            'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + mergedApiID + '/sourceApiAssociations/*',
            'arn:aws:appsync:' + aws_cdk_lib_1.Aws.REGION + ':' + aws_cdk_lib_1.Aws.ACCOUNT_ID + ':apis/' + sourceApiId + '/sourceApiAssociations/*',
        ],
    }));
    return mergedApiRole;
}
exports.setMergedApiRole = setMergedApiRole;
function setAppsyncCloudWatchlogsRole(scope, props) {
    const appsyncServicePrincipleRole = props.appsyncServicePrincipleRole || 'appsync.amazonaws.com';
    let appsynccloudWatchlogsRole = new iam.Role(scope, 'appsynccloudWatchlogsRole', {
        assumedBy: new iam.ServicePrincipal(appsyncServicePrincipleRole),
    });
    appsynccloudWatchlogsRole.addToPolicy(new iam.PolicyStatement({
        effect: iam.Effect.ALLOW,
        actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        resources: ['*'],
    }));
    return appsynccloudWatchlogsRole;
}
//# sourceMappingURL=data:application/json;base64,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