"""Dataset resource."""

from typing import Any, Optional

import httpx


class DatasetResource:
    """
    Dataset API endpoints.

    Provides access to customer datasets generated by precalc jobs.
    Allows viewing, querying, and deleting datasets.
    """

    def __init__(self, client: httpx.Client) -> None:
        """
        Initialize Dataset resource.

        Args:
            client: Configured httpx client
        """
        self._client = client

    def list_owners(self, mode: str = "compact") -> list[str] | list[dict[str, Any]]:
        """
        List dataset owners accessible to current user.

        Args:
            mode: 'compact' returns list of owner names,
                  'detailed' returns list with datasets per owner

        Returns:
            List of owner names if mode='compact',
            or list of owner details with datasets if mode='detailed'

        Raises:
            AuthenticationError: Invalid token
            ServerError: Server error occurred

        Example:
            >>> owners = client.dataset.list_owners()
            >>> print(owners)
            ['owner1', 'owner2', ...]

            >>> detailed = client.dataset.list_owners(mode='detailed')
            >>> print(detailed[0]['owner'])
            'owner1'
        """
        response = self._client.get("/v1/dataset", params={"mode": mode})
        response.raise_for_status()
        result = response.json()

        if mode == "detailed":
            return result.get("Owners", [])
        return result.get("Owners", [])

    def get_datasets(self, owner: str) -> list[str]:
        """
        Get list of datasets for a specific owner.

        Args:
            owner: Owner identifier

        Returns:
            List of dataset names

        Raises:
            AuthenticationError: Invalid token
            ForbiddenError: Not authorized to access this owner
            NotFoundError: Owner not found
            ServerError: Server error occurred

        Example:
            >>> datasets = client.dataset.get_datasets('owner1')
            >>> print(datasets)
            ['dataset1', 'dataset2', ...]
        """
        response = self._client.get(f"/v1/dataset/{owner}")
        response.raise_for_status()
        result = response.json()
        return result.get("Datasets", [])

    def get_metadata(self, owner: str, dataset: str) -> dict[str, Any]:
        """
        Get dataset specification/schema.

        Args:
            owner: Owner identifier
            dataset: Dataset name

        Returns:
            Dataset specification including schema and metadata

        Raises:
            AuthenticationError: Invalid token
            ForbiddenError: Not authorized to access this dataset
            NotFoundError: Dataset not found
            ServerError: Server error occurred

        Example:
            >>> spec = client.dataset.get_metadata('owner1', 'dataset1')
            >>> print(spec['schema'])
        """
        response = self._client.get(f"/v1/dataset/{owner}/{dataset}")
        response.raise_for_status()
        return response.json()

    def get_data(
        self,
        owner: str,
        dataset: str,
        select: Optional[str] = None,
        where: Optional[str] = None,
        order_by: Optional[str] = None,
        format: str = "json",
        limit: Optional[int] = None,
    ) -> dict[str, Any] | str:
        """
        Query dataset results with optional filtering and ordering.

        Args:
            owner: Owner identifier
            dataset: Dataset name
            select: Comma-separated list of columns to select (optional)
            where: SQL WHERE clause for filtering (optional)
            order_by: Column name for sorting (optional)
            format: 'json' or 'csv' (default: 'json')
            limit: Maximum number of rows to return (optional)

        Returns:
            Dataset results as dict (JSON) or CSV string

        Raises:
            AuthenticationError: Invalid token
            ForbiddenError: Not authorized to access this dataset
            NotFoundError: Dataset not found
            ValidationError: Invalid query parameters
            ServerError: Server error occurred

        Example:
            >>> # Get all data as JSON
            >>> data = client.dataset.get_data('owner1', 'dataset1')
            >>> print(data['data'])

            >>> # Get filtered data as CSV
            >>> csv = client.dataset.get_data(
            ...     'owner1', 'dataset1',
            ...     select='col1,col2',
            ...     where='col1 > 100',
            ...     format='csv'
            ... )
        """
        url = f"/v1/dataset/{owner}/{dataset}/data"

        params: dict[str, Any] = {"format": format}
        if select is not None:
            params["select"] = select
        if where is not None:
            params["where"] = where
        if order_by is not None:
            params["orderBy"] = order_by
        if limit is not None:
            params["limit"] = limit

        response = self._client.get(url, params=params)
        response.raise_for_status()

        if format == "csv":
            return response.text
        return response.json()

    def delete(self, owner: str, dataset: str) -> dict[str, Any]:
        """
        Delete a dataset.

        Args:
            owner: Owner identifier
            dataset: Dataset name

        Returns:
            Deletion response with success status and message

        Raises:
            AuthenticationError: Invalid token
            ForbiddenError: Not authorized to delete this dataset
            NotFoundError: Dataset not found
            ServerError: Server error occurred

        Example:
            >>> result = client.dataset.delete('owner1', 'old_dataset')
            >>> print(result['success'])
            'true'
        """
        response = self._client.delete(f"/v1/dataset/{owner}/{dataset}")
        response.raise_for_status()
        return response.json()
