"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'MyTable';
// DynamoDB table parameters
const TABLE_NAME = 'MyTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_NAME = 'MyGSI';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.BINARY };
const GSI_NON_KEY = 'gsiNonKey';
function* GSI_GENERATOR() {
    let n = 0;
    while (true) {
        const globalSecondaryIndexProps = {
            indexName: `${GSI_NAME}${n}`,
            partitionKey: { name: `${GSI_PARTITION_KEY.name}${n}`, type: GSI_PARTITION_KEY.type },
        };
        yield globalSecondaryIndexProps;
        n++;
    }
}
function* NON_KEY_ATTRIBUTE_GENERATOR(nonKeyPrefix) {
    let n = 0;
    while (true) {
        yield `${nonKeyPrefix}${n}`;
        n++;
    }
}
// DynamoDB local secondary index parameters
const LSI_NAME = 'MyLSI';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = 'lsiNonKey';
function* LSI_GENERATOR() {
    let n = 0;
    while (true) {
        const localSecondaryIndexProps = {
            indexName: `${LSI_NAME}${n}`,
            sortKey: { name: `${LSI_SORT_KEY.name}${n}`, type: LSI_SORT_KEY.type },
        };
        yield localSecondaryIndexProps;
        n++;
    }
}
describe('default properties', () => {
    test('hash key only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [{ AttributeName: 'hashKey', AttributeType: 'S' }],
            KeySchema: [{ AttributeName: 'hashKey', KeyType: 'HASH' }],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.RETAIN }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('removalPolicy is DESTROY', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, removalPolicy: core_1.RemovalPolicy.DESTROY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.DELETE }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('hash + range key', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('hash + range key can also be specified in props', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('point-in-time recovery is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('server-side encryption is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('stream is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('ttl is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('can specify new and old images', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_AND_OLD_IMAGES,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_AND_OLD_IMAGES' },
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            TableName: 'MyTable',
        });
    });
    test('can specify new images only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_IMAGE' },
            TableName: 'MyTable',
        });
    });
    test('can specify old images only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.OLD_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'OLD_IMAGE' },
            TableName: 'MyTable',
        });
    });
});
test('when specifying every property', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        readCapacity: 42,
        writeCapacity: 1337,
        pointInTimeRecovery: true,
        serverSideEncryption: true,
        billingMode: lib_1.BillingMode.PROVISIONED,
        stream: lib_1.StreamViewType.KEYS_ONLY,
        timeToLiveAttribute: 'timeToLive',
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 42,
            WriteCapacityUnits: 1337,
        },
        PointInTimeRecoverySpecification: { PointInTimeRecoveryEnabled: true },
        SSESpecification: { SSEEnabled: true },
        StreamSpecification: { StreamViewType: 'KEYS_ONLY' },
        TableName: 'MyTable',
        Tags: [{ Key: 'Environment', Value: 'Production' }],
        TimeToLiveSpecification: { AttributeName: 'timeToLive', Enabled: true },
    });
});
test('when specifying PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
        ],
        TableName: 'MyTable',
    });
});
test('error when specifying read or write capacity with a PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table B', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table C', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding a global secondary index with hash key only', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.ALL,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value, gsiNonKeyAttributeGenerator.next().value],
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['gsiNonKey0', 'gsiNonKey1'], ProjectionType: 'INCLUDE' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index on a table with PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    }).addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('error when adding a global secondary index with projection type INCLUDE, but without specifying non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
    })).toThrow(/non-key attributes should be specified when using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type ALL, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type KEYS_ONLY, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type INCLUDE, but with more than 20 non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    const gsiNonKeyAttributes = [];
    for (let i = 0; i < 21; i++) {
        gsiNonKeyAttributes.push(gsiNonKeyAttributeGenerator.next().value);
    }
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: gsiNonKeyAttributes,
    })).toThrow(/a maximum number of nonKeyAttributes across all of secondary indexes is 20/);
});
test('error when adding a global secondary index with read or write capacity on a PAY_PER_REQUEST table', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
    });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding multiple global secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiGenerator = GSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addGlobalSecondaryIndex(gsiGenerator.next().value);
    }
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey0', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey1', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey2', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey3', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey4', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI0',
                KeySchema: [
                    { AttributeName: 'gsiHashKey0', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI1',
                KeySchema: [
                    { AttributeName: 'gsiHashKey1', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI2',
                KeySchema: [
                    { AttributeName: 'gsiHashKey2', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI3',
                KeySchema: [
                    { AttributeName: 'gsiHashKey3', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI4',
                KeySchema: [
                    { AttributeName: 'gsiHashKey4', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index without specifying read and write capacity', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a local secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(LSI_NON_KEY);
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [lsiNonKeyAttributeGenerator.next().value, lsiNonKeyAttributeGenerator.next().value],
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['lsiNonKey0', 'lsiNonKey1'], ProjectionType: 'INCLUDE' },
            },
        ],
    });
});
test('error when adding more than 5 local secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiGenerator = LSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addLocalSecondaryIndex(lsiGenerator.next().value);
    }
    expect(() => table.addLocalSecondaryIndex(lsiGenerator.next().value))
        .toThrow(/a maximum number of local secondary index per table is 5/);
});
test('error when adding a local secondary index with the name of a global secondary index', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: 'SecondaryIndex',
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(() => table.addLocalSecondaryIndex({
        indexName: 'SecondaryIndex',
        sortKey: LSI_SORT_KEY,
    })).toThrow(/a duplicate index name, SecondaryIndex, is not allowed/);
});
test('error when validating construct if a local secondary index exists without a sort key of the table', () => {
    var _a;
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    const errors = core_1.ConstructNode.validate(table.node);
    expect(errors.length).toBe(1);
    expect((_a = errors[0]) === null || _a === void 0 ? void 0 : _a.message).toBe('a sort key of the table must be specified to add local secondary indexes');
});
test('can enable Read AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:ReadCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBReadCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('can enable Write AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:WriteCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBWriteCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('cannot enable AutoScaling twice on the same property', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/Read AutoScaling already enabled for this table/);
});
test('error when enabling AutoScaling on the PAY_PER_REQUEST table', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { billingMode: lib_1.BillingMode.PAY_PER_REQUEST, partitionKey: TABLE_PARTITION_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => {
        table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.autoScaleGlobalSecondaryIndexReadCapacity(GSI_NAME, {
        minCapacity: 1,
        maxCapacity: 5,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('error when specifying Read Auto Scaling with invalid scalingTargetValue < 10', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 5 });
    }).toThrow(/targetUtilizationPercent for DynamoDB scaling must be between 10 and 90 percent, got: 5/);
});
test('error when specifying Read Auto Scaling with invalid minimumCapacity', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => table.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 5 }))
        .toThrow(/minCapacity \(10\) should be lower than maxCapacity \(5\)/);
});
test('can autoscale on a schedule', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        readCapacity: 42,
        writeCapacity: 1337,
        partitionKey: { name: 'Hash', type: lib_1.AttributeType.STRING },
    });
    // WHEN
    const scaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 100 });
    scaling.scaleOnSchedule('SaveMoneyByNotScalingUp', {
        schedule: appscaling.Schedule.cron({}),
        maxCapacity: 10,
    });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        ScheduledActions: [
            {
                ScalableTargetAction: { 'MaxCapacity': 10 },
                Schedule: 'cron(* * * * ? *)',
                ScheduledActionName: 'SaveMoneyByNotScalingUp',
            },
        ],
    });
});
describe('metrics', () => {
    test('Can use metricConsumedReadCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedReadCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedReadCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricConsumedWriteCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedWriteCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedWriteCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricSystemErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSystemErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SystemErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricUserErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricUserErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'UserErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricConditionalCheckFailedRequests on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConditionalCheckFailedRequests())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConditionalCheckFailedRequests',
            statistic: 'Sum',
        });
    });
    test('Can use metricSuccessfulRequestLatency on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSuccessfulRequestLatency())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SuccessfulRequestLatency',
            statistic: 'Average',
        });
    });
});
describe('grants', () => {
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        testGrant(['action1', 'action2'], (p, t) => t.grant(p, 'dynamodb:action1', 'dynamodb:action2'));
    });
    test('"grantReadData" allows the principal to read data from the table', () => {
        testGrant(['BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan'], (p, t) => t.grantReadData(p));
    });
    test('"grantWriteData" allows the principal to write data to the table', () => {
        testGrant(['BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'], (p, t) => t.grantWriteData(p));
    });
    test('"grantReadWriteData" allows the principal to read/write data', () => {
        testGrant([
            'BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan',
            'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
        ], (p, t) => t.grantReadWriteData(p));
    });
    test('"grantFullAccess" allows the principal to perform any action on the table ("*")', () => {
        testGrant(['*'], (p, t) => t.grantFullAccess(p));
    });
    test('"Table.grantListStreams" allows principal to list all streams', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const user = new iam.User(stack, 'user');
        // WHEN
        lib_1.Table.grantListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantTableListStreams" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantTableListStreams(user)).toThrow(/DynamoDB Streams must be enabled on the table my-table/);
    });
    test('"grantTableListStreams" allows principal to list all streams for this table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantStreamRead" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantStreamRead(user)).toThrow(/DynamoDB Streams must be enabled on the table my-table/);
    });
    test('"grantStreamRead" allows principal to read and describe the table stream"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantStreamRead(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                    {
                        'Action': [
                            'dynamodb:DescribeStream',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'StreamArn',
                            ],
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('if table has an index grant gives access to the index', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
        table.addGlobalSecondaryIndex({ indexName: 'MyIndex', partitionKey: { name: 'Age', type: lib_1.AttributeType.NUMBER } });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            { 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] },
                            { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/index/*']] },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
});
describe('secondary indexes', () => {
    // See https://github.com/aws/aws-cdk/issues/4398
    test('attribute can be used as key attribute in one index, and non-key in another', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'pkey', type: lib_1.AttributeType.NUMBER },
        });
        // WHEN
        table.addGlobalSecondaryIndex({
            indexName: 'IndexA',
            partitionKey: { name: 'foo', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['bar'],
        });
        // THEN
        expect(() => table.addGlobalSecondaryIndex({
            indexName: 'IndexB',
            partitionKey: { name: 'baz', type: lib_1.AttributeType.STRING },
            sortKey: { name: 'bar', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['blah'],
        })).not.toThrow();
    });
});
describe('import', () => {
    test('report error when importing an external/existing table from invalid arn missing resource name', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1::table/';
        // WHEN
        expect(() => lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn)).toThrow(/ARN for DynamoDB table must be in the form: .../);
    });
    test('static fromTableArn(arn) allows importing an external/existing table from arn', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1:11111111:table/MyTable';
        const table = lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            tableArn,
                            { 'Ref': 'AWS::NoValue' },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe(tableArn);
        expect(stack.resolve(table.tableName)).toBe('MyTable');
    });
    test('static fromTableName(name) allows importing an external/existing table from table name', () => {
        const stack = new core_1.Stack();
        const tableName = 'MyTable';
        const table = lib_1.Table.fromTableName(stack, 'ImportedTable', tableName);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadWriteData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            'Ref': 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            'Ref': 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            'Ref': 'AWS::AccountId',
                                        },
                                        ':table/MyTable',
                                    ],
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe('arn:${Token[AWS::Partition.3]}:dynamodb:${Token[AWS::Region.4]}:${Token[AWS::AccountId.0]}:table/MyTable');
        expect(stack.resolve(table.tableName)).toBe(tableName);
    });
    describe('stream permissions on imported tables', () => {
        test('throw if no tableStreamArn is specified', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(() => table.grantTableListStreams(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
            expect(() => table.grantStreamRead(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
        });
        test('creates the correct list streams grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantTableListStreams(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
        test('creates the correct stream read grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantStreamRead(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                        {
                            Action: ['dynamodb:DescribeStream', 'dynamodb:GetRecords', 'dynamodb:GetShardIterator'],
                            Effect: 'Allow',
                            Resource: tableStreamArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
    });
});
describe('global', () => {
    test('create replicas', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-west-2',
            },
            Condition: 'TableStackRegionNotEqualseuwest2A03859E7',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-central-1',
            },
            Condition: 'TableStackRegionNotEqualseucentral199D46FC0',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toEqual({
            TableStackRegionNotEqualseuwest2A03859E7: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-west-2', { Ref: 'AWS::Region' }] },
                ],
            },
            TableStackRegionNotEqualseucentral199D46FC0: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-central-1', { Ref: 'AWS::Region' }] },
                ],
            },
        });
    });
    test('throws with PROVISIONED billing mode', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            billingMode: lib_1.BillingMode.PROVISIONED,
        })).toThrow(/`PAY_PER_REQUEST`/);
    });
    test('throws when stream is set and not set to NEW_AND_OLD_IMAGES', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            stream: lib_1.StreamViewType.OLD_IMAGE,
        })).toThrow(/`NEW_AND_OLD_IMAGES`/);
    });
    test('throws with replica in same region as stack', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'us-east-1' },
        });
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-1',
                'us-east-1',
                'eu-west-2',
            ],
        })).toThrow(/`replicationRegions` cannot include the region where this stack is deployed/);
    });
    test('no conditions when region is known', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'eu-west-1' },
        });
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toBeUndefined();
    });
});
function testGrant(expectedActions, invocation) {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
    const user = new iam.User(stack, 'user');
    // WHEN
    invocation(user, table);
    // THEN
    const action = expectedActions.length > 1 ? expectedActions.map(a => `dynamodb:${a}`) : `dynamodb:${expectedActions[0]}`;
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': action,
                    'Effect': 'Allow',
                    'Resource': [
                        { 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] },
                        { 'Ref': 'AWS::NoValue' },
                    ],
                },
            ],
            'Version': '2012-10-17',
        },
        'Users': [{ 'Ref': 'user2C2B57AE' }],
    });
}
//# sourceMappingURL=data:application/json;base64,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