from pathlib import Path
from typing import Optional, List, Sequence

from git import Repo

from flexlate.adder import Adder
from flexlate.add_mode import AddMode
from flexlate.config_manager import ConfigManager
from flexlate.constants import DEFAULT_MERGED_BRANCH_NAME, DEFAULT_TEMPLATE_BRANCH_NAME
from flexlate.finder.multi import MultiFinder
from flexlate.remover import Remover
from flexlate.render.multi import MultiRenderer
from flexlate.template.base import Template
from flexlate.template_data import TemplateData
from flexlate.transactions.transaction import FlexlateTransaction, TransactionType
from flexlate.transactions.undoer import Undoer
from flexlate.update.main import Updater


class Flexlate:
    def __init__(
        self,
        adder: Adder = Adder(),
        remover: Remover = Remover(),
        config_manager: ConfigManager = ConfigManager(),
        finder: MultiFinder = MultiFinder(),
        renderer: MultiRenderer = MultiRenderer(),
        updater: Updater = Updater(),
        undoer: Undoer = Undoer(),
    ):
        self.adder = adder
        self.remover = remover
        self.config_manager = config_manager
        self.finder = finder
        self.renderer = renderer
        self.updater = updater
        self.undoer = undoer

    def init_project(
        self,
        path: Path = Path("."),
        default_add_mode: AddMode = AddMode.LOCAL,
        merged_branch_name: str = DEFAULT_MERGED_BRANCH_NAME,
        template_branch_name: str = DEFAULT_TEMPLATE_BRANCH_NAME,
        user: bool = False,
    ):
        repo = Repo(path)
        self.adder.init_project_and_add_to_branches(
            repo,
            default_add_mode=default_add_mode,
            user=user,
            merged_branch_name=merged_branch_name,
            template_branch_name=template_branch_name,
        )

    def add_template_source(
        self,
        path: str,
        name: Optional[str] = None,
        target_version: Optional[str] = None,
        template_root: Path = Path("."),
        add_mode: Optional[AddMode] = None,
    ):
        transaction = FlexlateTransaction(
            type=TransactionType.ADD_SOURCE, target=name or path, out_root=template_root
        )
        project_config = self.config_manager.load_project_config(template_root)
        add_mode = add_mode or project_config.default_add_mode
        template = self.finder.find(path, version=target_version)
        if name:
            template.name = name
        repo = Repo(project_config.path)
        self.adder.add_template_source(
            repo,
            template,
            transaction,
            target_version=target_version,
            out_root=template_root,
            merged_branch_name=project_config.merged_branch_name,
            template_branch_name=project_config.template_branch_name,
            add_mode=add_mode,
            config_manager=self.config_manager,
        )

    def remove_template_source(
        self,
        template_name: str,
        template_root: Path = Path("."),
    ):
        transaction = FlexlateTransaction(
            type=TransactionType.REMOVE_SOURCE,
            target=template_name,
            out_root=template_root,
        )
        project_config = self.config_manager.load_project_config(template_root)
        repo = Repo(project_config.path)
        self.remover.remove_template_source(
            repo,
            template_name,
            transaction,
            out_root=template_root,
            merged_branch_name=project_config.merged_branch_name,
            template_branch_name=project_config.template_branch_name,
            add_mode=project_config.default_add_mode,
            config_manager=self.config_manager,
        )

    def apply_template_and_add(
        self,
        name: str,
        data: Optional[TemplateData] = None,
        out_root: Path = Path("."),
        add_mode: Optional[AddMode] = None,
        no_input: bool = False,
    ):
        transaction = FlexlateTransaction(
            type=TransactionType.ADD_OUTPUT, target=name, out_root=out_root, data=data
        )
        project_config = self.config_manager.load_project_config(out_root)
        add_mode = add_mode or project_config.default_add_mode
        template = self.config_manager.get_template_by_name(
            name, project_root=project_config.path
        )
        repo = Repo(project_config.path)
        self.adder.apply_template_and_add(
            repo,
            template,
            transaction,
            data=data,
            out_root=out_root,
            add_mode=add_mode,
            no_input=no_input,
            merged_branch_name=project_config.merged_branch_name,
            template_branch_name=project_config.template_branch_name,
            config_manager=self.config_manager,
            renderer=self.renderer,
            updater=self.updater,
        )

    def remove_applied_template_and_output(
        self,
        template_name: str,
        out_root: Path = Path("."),
    ):
        transaction = FlexlateTransaction(
            type=TransactionType.REMOVE_OUTPUT, target=template_name, out_root=out_root
        )
        project_config = self.config_manager.load_project_config(out_root)
        repo = Repo(project_config.path)

        self.remover.remove_applied_template_and_output(
            repo,
            template_name,
            transaction,
            out_root=out_root,
            add_mode=project_config.default_add_mode,
            merged_branch_name=project_config.merged_branch_name,
            template_branch_name=project_config.template_branch_name,
            config_manager=self.config_manager,
            updater=self.updater,
            renderer=self.renderer,
        )

    def update(
        self,
        names: Optional[List[str]] = None,
        data: Optional[Sequence[TemplateData]] = None,
        no_input: bool = False,
        project_path: Path = Path("."),
    ):
        transaction = FlexlateTransaction(
            type=TransactionType.UPDATE,
            target=", ".join(names) if names is not None else None,
            data=data,
        )
        project_config = self.config_manager.load_project_config(project_path)
        templates: List[Template]
        if names:
            # User wants to update targeted template sources
            templates = [
                self.config_manager.get_template_by_name(
                    name, project_root=project_config.path
                )
                for name in names
            ]
        else:
            # User wants to update all templates
            renderables = self.config_manager.get_all_renderables(
                project_root=project_config.path
            )
            templates = [renderable.template for renderable in renderables]
        self.updater.update_passed_templates_to_target_versions(
            templates,
            project_root=project_config.path,
            finder=self.finder,
            config_manager=self.config_manager,
        )
        updates = self.updater.get_updates_for_templates(
            templates,
            data=data,
            config_manager=self.config_manager,
            project_root=project_config.path,
        )

        repo = Repo(project_config.path)
        self.updater.update(
            repo,
            updates,
            transaction,
            no_input=no_input,
            merged_branch_name=project_config.merged_branch_name,
            template_branch_name=project_config.template_branch_name,
            renderer=self.renderer,
            config_manager=self.config_manager,
        )

    def undo(self, num_operations: int = 1, project_path: Path = Path(".")):
        project_config = self.config_manager.load_project_config(project_path)
        repo = Repo(project_config.path)
        self.undoer.undo_transactions(
            repo,
            num_transactions=num_operations,
            merged_branch_name=project_config.merged_branch_name,
            template_branch_name=project_config.template_branch_name,
        )

    # TODO: list template sources, list applied templates
    # TODO: Update target versions in template sources
