import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { IResource, RemovalPolicy, Resource, SecretValue } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { RotationSchedule, RotationScheduleOptions } from './rotation-schedule';
/**
 * A secret in AWS Secrets Manager.
 *
 * @stability stable
 */
export interface ISecret extends IResource {
    /**
     * The customer-managed encryption key that is used to encrypt this secret, if any.
     *
     * When not specified, the default
     * KMS key for the account and region is being used.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The ARN of the secret in AWS Secrets Manager.
     *
     * Will return the full ARN if available, otherwise a partial arn.
     * For secrets imported by the deprecated `fromSecretName`, it will return the `secretName`.
     *
     * @stability stable
     * @attribute true
     */
    readonly secretArn: string;
    /**
     * The full ARN of the secret in AWS Secrets Manager, which is the ARN including the Secrets Manager-supplied 6-character suffix.
     *
     * This is equal to `secretArn` in most cases, but is undefined when a full ARN is not available (e.g., secrets imported by name).
     *
     * @stability stable
     */
    readonly secretFullArn?: string;
    /**
     * The name of the secret.
     *
     * For "owned" secrets, this will be the full resource name (secret name + suffix), unless the
     * '@aws-cdk/aws-secretsmanager:parseOwnedSecretName' feature flag is set.
     *
     * @stability stable
     */
    readonly secretName: string;
    /**
     * Retrieve the value of the stored secret as a `SecretValue`.
     *
     * @stability stable
     * @attribute true
     */
    readonly secretValue: SecretValue;
    /**
     * Interpret the secret as a JSON object and return a field's value from it as a `SecretValue`.
     *
     * @stability stable
     */
    secretValueFromJson(key: string): SecretValue;
    /**
     * Grants reading the secret value to some role.
     *
     * @param grantee the principal being granted permission.
     * @param versionStages the version stages the grant is limited to.
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable, versionStages?: string[]): iam.Grant;
    /**
     * Grants writing and updating the secret value to some role.
     *
     * @param grantee the principal being granted permission.
     * @stability stable
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Adds a rotation schedule to the secret.
     *
     * @stability stable
     */
    addRotationSchedule(id: string, options: RotationScheduleOptions): RotationSchedule;
    /**
     * Adds a statement to the IAM resource policy associated with this secret.
     *
     * If this secret was created in this stack, a resource policy will be
     * automatically created upon the first call to `addToResourcePolicy`. If
     * the secret is imported, then this is a no-op.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Denies the `DeleteSecret` action to all principals within the current account.
     *
     * @stability stable
     */
    denyAccountRootDelete(): void;
    /**
     * Attach a target to this secret.
     *
     * @param target The target to attach.
     * @returns An attached secret
     * @stability stable
     */
    attach(target: ISecretAttachmentTarget): ISecret;
}
/**
 * The properties required to create a new secret in AWS Secrets Manager.
 *
 * @stability stable
 */
export interface SecretProps {
    /**
     * An optional, human-friendly description of the secret.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
    /**
     * The customer-managed encryption key to use for encrypting the secret value.
     *
     * @default - A default KMS key for the account and region is used.
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Configuration for how to generate a secret value.
     *
     * @default - 32 characters with upper-case letters, lower-case letters, punctuation and numbers (at least one from each
     * category), per the default values of ``SecretStringGenerator``.
     * @stability stable
     */
    readonly generateSecretString?: SecretStringGenerator;
    /**
     * A name for the secret.
     *
     * Note that deleting secrets from SecretsManager does not happen immediately, but after a 7 to
     * 30 days blackout period. During that period, it is not possible to create another secret that shares the same name.
     *
     * @default - A name is generated by CloudFormation.
     * @stability stable
     */
    readonly secretName?: string;
    /**
     * Policy to apply when the secret is removed from this stack.
     *
     * @default - Not set.
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * A list of regions where to replicate this secret.
     *
     * @default - Secret is not replicated
     * @stability stable
     */
    readonly replicaRegions?: ReplicaRegion[];
}
/**
 * Secret replica region.
 *
 * @stability stable
 */
export interface ReplicaRegion {
    /**
     * The name of the region.
     *
     * @stability stable
     */
    readonly region: string;
    /**
     * The customer-managed encryption key to use for encrypting the secret value.
     *
     * @default - A default KMS key for the account and region is used.
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * Attributes required to import an existing secret into the Stack.
 *
 * One ARN format (`secretArn`, `secretCompleteArn`, `secretPartialArn`) must be provided.
 *
 * @stability stable
 */
export interface SecretAttributes {
    /**
     * The encryption key that is used to encrypt the secret, unless the default SecretsManager key is used.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * (deprecated) The ARN of the secret in SecretsManager.
     *
     * Cannot be used with `secretCompleteArn` or `secretPartialArn`.
     *
     * @deprecated use `secretCompleteArn` or `secretPartialArn` instead.
     */
    readonly secretArn?: string;
    /**
     * The complete ARN of the secret in SecretsManager.
     *
     * This is the ARN including the Secrets Manager 6-character suffix.
     * Cannot be used with `secretArn` or `secretPartialArn`.
     *
     * @stability stable
     */
    readonly secretCompleteArn?: string;
    /**
     * The partial ARN of the secret in SecretsManager.
     *
     * This is the ARN without the Secrets Manager 6-character suffix.
     * Cannot be used with `secretArn` or `secretCompleteArn`.
     *
     * @stability stable
     */
    readonly secretPartialArn?: string;
}
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
declare abstract class SecretBase extends Resource implements ISecret {
    abstract readonly encryptionKey?: kms.IKey;
    abstract readonly secretArn: string;
    abstract readonly secretName: string;
    protected abstract readonly autoCreatePolicy: boolean;
    private policy?;
    /**
     * The full ARN of the secret in AWS Secrets Manager, which is the ARN including the Secrets Manager-supplied 6-character suffix.
     *
     * This is equal to `secretArn` in most cases, but is undefined when a full ARN is not available (e.g., secrets imported by name).
     *
     * @stability stable
     */
    get secretFullArn(): string | undefined;
    /**
     * Grants reading the secret value to some role.
     *
     * @stability stable
     */
    grantRead(grantee: iam.IGrantable, versionStages?: string[]): iam.Grant;
    /**
     * Grants writing and updating the secret value to some role.
     *
     * @stability stable
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Retrieve the value of the stored secret as a `SecretValue`.
     *
     * @stability stable
     */
    get secretValue(): SecretValue;
    /**
     * Interpret the secret as a JSON object and return a field's value from it as a `SecretValue`.
     *
     * @stability stable
     */
    secretValueFromJson(jsonField: string): SecretValue;
    /**
     * Adds a rotation schedule to the secret.
     *
     * @stability stable
     */
    addRotationSchedule(id: string, options: RotationScheduleOptions): RotationSchedule;
    /**
     * Adds a statement to the IAM resource policy associated with this secret.
     *
     * If this secret was created in this stack, a resource policy will be
     * automatically created upon the first call to `addToResourcePolicy`. If
     * the secret is imported, then this is a no-op.
     *
     * @stability stable
     */
    addToResourcePolicy(statement: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Denies the `DeleteSecret` action to all principals within the current account.
     *
     * @stability stable
     */
    denyAccountRootDelete(): void;
    /**
     * Provides an identifier for this secret for use in IAM policies.
     *
     * If there is a full ARN, this is just the ARN;
     * if we have a partial ARN -- due to either importing by secret name or partial ARN --
     * then we need to add a suffix to capture the full ARN's format.
     *
     * @stability stable
     */
    protected get arnForPolicies(): string;
    /**
     * Attach a target to this secret.
     *
     * @param target The target to attach.
     * @returns An attached secret
     * @stability stable
     */
    attach(target: ISecretAttachmentTarget): ISecret;
}
/**
 * Creates a new secret in AWS SecretsManager.
 *
 * @stability stable
 */
export declare class Secret extends SecretBase {
    /**
     * @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn`
     */
    static fromSecretArn(scope: Construct, id: string, secretArn: string): ISecret;
    /**
     * Imports a secret by complete ARN.
     *
     * The complete ARN is the ARN with the Secrets Manager-supplied suffix.
     *
     * @stability stable
     */
    static fromSecretCompleteArn(scope: Construct, id: string, secretCompleteArn: string): ISecret;
    /**
     * Imports a secret by partial ARN.
     *
     * The partial ARN is the ARN without the Secrets Manager-supplied suffix.
     *
     * @stability stable
     */
    static fromSecretPartialArn(scope: Construct, id: string, secretPartialArn: string): ISecret;
    /**
     * (deprecated) Imports a secret by secret name;
     *
     * the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     *
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope: Construct, id: string, secretName: string): ISecret;
    /**
     * Imports a secret by secret name.
     *
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     *
     * @stability stable
     */
    static fromSecretNameV2(scope: Construct, id: string, secretName: string): ISecret;
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     * @stability stable
     */
    static fromSecretAttributes(scope: Construct, id: string, attrs: SecretAttributes): ISecret;
    /**
     * The customer-managed encryption key that is used to encrypt this secret, if any.
     *
     * When not specified, the default
     * KMS key for the account and region is being used.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The ARN of the secret in AWS Secrets Manager.
     *
     * Will return the full ARN if available, otherwise a partial arn.
     * For secrets imported by the deprecated `fromSecretName`, it will return the `secretName`.
     *
     * @stability stable
     */
    readonly secretArn: string;
    /**
     * The name of the secret.
     *
     * For "owned" secrets, this will be the full resource name (secret name + suffix), unless the
     * '@aws-cdk/aws-secretsmanager:parseOwnedSecretName' feature flag is set.
     *
     * @stability stable
     */
    readonly secretName: string;
    private replicaRegions;
    /**
     * @stability stable
     */
    protected readonly autoCreatePolicy = true;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: SecretProps);
    /**
     * (deprecated) Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id: string, options: AttachedSecretOptions): SecretTargetAttachment;
    /**
     * Adds a replica region for the secret.
     *
     * @param region The name of the region.
     * @param encryptionKey The customer-managed encryption key to use for encrypting the secret value.
     * @stability stable
     */
    addReplicaRegion(region: string, encryptionKey?: kms.IKey): void;
}
/**
 * A secret attachment target.
 *
 * @stability stable
 */
export interface ISecretAttachmentTarget {
    /**
     * Renders the target specifications.
     *
     * @stability stable
     */
    asSecretAttachmentTarget(): SecretAttachmentTargetProps;
}
/**
 * The type of service or database that's being associated with the secret.
 *
 * @stability stable
 */
export declare enum AttachmentTargetType {
    /**
     * (deprecated) A database instance.
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    INSTANCE = "AWS::RDS::DBInstance",
    /**
     * (deprecated) A database cluster.
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    CLUSTER = "AWS::RDS::DBCluster",
    /**
     * AWS::RDS::DBInstance
     */
    RDS_DB_INSTANCE = "AWS::RDS::DBInstance",
    /**
     * AWS::RDS::DBCluster
     */
    RDS_DB_CLUSTER = "AWS::RDS::DBCluster",
    /**
     * AWS::RDS::DBProxy.
     *
     * @stability stable
     */
    RDS_DB_PROXY = "AWS::RDS::DBProxy",
    /**
     * AWS::Redshift::Cluster.
     *
     * @stability stable
     */
    REDSHIFT_CLUSTER = "AWS::Redshift::Cluster",
    /**
     * AWS::DocDB::DBInstance.
     *
     * @stability stable
     */
    DOCDB_DB_INSTANCE = "AWS::DocDB::DBInstance",
    /**
     * AWS::DocDB::DBCluster.
     *
     * @stability stable
     */
    DOCDB_DB_CLUSTER = "AWS::DocDB::DBCluster"
}
/**
 * Attachment target specifications.
 *
 * @stability stable
 */
export interface SecretAttachmentTargetProps {
    /**
     * The id of the target to attach the secret to.
     *
     * @stability stable
     */
    readonly targetId: string;
    /**
     * The type of the target to attach the secret to.
     *
     * @stability stable
     */
    readonly targetType: AttachmentTargetType;
}
/**
 * (deprecated) Options to add a secret attachment to a secret.
 *
 * @deprecated use `secret.attach()` instead
 */
export interface AttachedSecretOptions {
    /**
     * (deprecated) The target to attach the secret to.
     *
     * @deprecated
     */
    readonly target: ISecretAttachmentTarget;
}
/**
 * Construction properties for an AttachedSecret.
 *
 * @stability stable
 */
export interface SecretTargetAttachmentProps extends AttachedSecretOptions {
    /**
     * The secret to attach to the target.
     *
     * @stability stable
     */
    readonly secret: ISecret;
}
/**
 * @stability stable
 */
export interface ISecretTargetAttachment extends ISecret {
    /**
     * Same as `secretArn`.
     *
     * @stability stable
     * @attribute true
     */
    readonly secretTargetAttachmentSecretArn: string;
}
/**
 * An attached secret.
 *
 * @stability stable
 */
export declare class SecretTargetAttachment extends SecretBase implements ISecretTargetAttachment {
    /**
     * @stability stable
     */
    static fromSecretTargetAttachmentSecretArn(scope: Construct, id: string, secretTargetAttachmentSecretArn: string): ISecretTargetAttachment;
    /**
     * The customer-managed encryption key that is used to encrypt this secret, if any.
     *
     * When not specified, the default
     * KMS key for the account and region is being used.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The ARN of the secret in AWS Secrets Manager.
     *
     * Will return the full ARN if available, otherwise a partial arn.
     * For secrets imported by the deprecated `fromSecretName`, it will return the `secretName`.
     *
     * @stability stable
     */
    readonly secretArn: string;
    /**
     * The name of the secret.
     *
     * For "owned" secrets, this will be the full resource name (secret name + suffix), unless the
     * '@aws-cdk/aws-secretsmanager:parseOwnedSecretName' feature flag is set.
     *
     * @stability stable
     */
    readonly secretName: string;
    /**
     * Same as `secretArn`.
     *
     * @stability stable
     * @attribute true
     */
    readonly secretTargetAttachmentSecretArn: string;
    /**
     * @stability stable
     */
    protected readonly autoCreatePolicy = true;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SecretTargetAttachmentProps);
}
/**
 * Configuration to generate secrets such as passwords automatically.
 *
 * @stability stable
 */
export interface SecretStringGenerator {
    /**
     * Specifies that the generated password shouldn't include uppercase letters.
     *
     * @default false
     * @stability stable
     */
    readonly excludeUppercase?: boolean;
    /**
     * Specifies whether the generated password must include at least one of every allowed character type.
     *
     * @default true
     * @stability stable
     */
    readonly requireEachIncludedType?: boolean;
    /**
     * Specifies that the generated password can include the space character.
     *
     * @default false
     * @stability stable
     */
    readonly includeSpace?: boolean;
    /**
     * A string that includes characters that shouldn't be included in the generated password.
     *
     * The string can be a minimum
     * of ``0`` and a maximum of ``4096`` characters long.
     *
     * @default no exclusions
     * @stability stable
     */
    readonly excludeCharacters?: string;
    /**
     * The desired length of the generated password.
     *
     * @default 32
     * @stability stable
     */
    readonly passwordLength?: number;
    /**
     * Specifies that the generated password shouldn't include punctuation characters.
     *
     * @default false
     * @stability stable
     */
    readonly excludePunctuation?: boolean;
    /**
     * Specifies that the generated password shouldn't include lowercase letters.
     *
     * @default false
     * @stability stable
     */
    readonly excludeLowercase?: boolean;
    /**
     * Specifies that the generated password shouldn't include digits.
     *
     * @default false
     * @stability stable
     */
    readonly excludeNumbers?: boolean;
    /**
     * A properly structured JSON string that the generated password can be added to.
     *
     * The ``generateStringKey`` is
     * combined with the generated random string and inserted into the JSON structure that's specified by this parameter.
     * The merged JSON string is returned as the completed SecretString of the secret. If you specify ``secretStringTemplate``
     * then ``generateStringKey`` must be also be specified.
     *
     * @stability stable
     */
    readonly secretStringTemplate?: string;
    /**
     * The JSON key name that's used to add the generated password to the JSON structure specified by the ``secretStringTemplate`` parameter.
     *
     * If you specify ``generateStringKey`` then ``secretStringTemplate``
     * must be also be specified.
     *
     * @stability stable
     */
    readonly generateStringKey?: string;
}
export {};
