import json
import math
from typing import Any, Dict

import pytest
from kisters.network_store.model_library.util import (  # element_to_dict,
    all_links,
    all_nodes,
    element_from_dict,
    elements_mapping,
)

elements = [
    {
        "uid": "channel1",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": {
                "roughness": 10.0,
                "cross_section": [
                    {"z": 0, "lr": -5},
                    {"z": 10, "lr": -5},
                    {"z": 0, "lr": 5},
                    {"z": 10, "lr": 5},
                ],
            },
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel2",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": [
                {
                    "distance": 25.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
                {
                    "distance": 75.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
            ],
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel3",
        "source_uid": "junction",
        "target_uid": "storage",
        "hydrologic_routing": {
            "muskingum_station": {"model": "muskingum", "k": 100.0, "x": 0.4},
            "reservoir_station": {"model": "reservoir", "p": 2.0, "m": 1.0},
        },
        "length": 100.0,
        "display_name": "channel",
        "element_class": "Channel",
        "user_metadata": {
            "ArbitraryKey": "1",
            "ArbitraryKey2": 1.0,
            "ArbitraryKey3": 1,
            "ArbitraryKey4": True,
            "ArbitraryKey5": 1.1,
        },
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel4",
        "source_uid": "junction",
        "target_uid": "storage",
        "hydrologic_routing": {
            "reservoir_station": {
                "model": "reservoir",
                "p": 2.0,
                "m": 1.0,
            }
        },
        "length": 100.0,
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "delay",
        "created": "2020-03-04T11:02:16.790000+00:00",
        "source_uid": "junction",
        "target_uid": "storage",
        "transit_time": 10.0,
        "display_name": "delay",
        "element_class": "Delay",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "flow_controlled_structure",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": -1.0,
        "max_flow": 1.0,
        "display_name": "flow_controlled_structure",
        "element_class": "FlowControlledStructure",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "drain",
        "source_uid": "junction",
        "target_uid": "storage",
        "flow_model": "free",
        "coefficient": 1.0,
        "level": 0.0,
        "min_area": 0.0,
        "max_area": 10.0,
        "display_name": "drain",
        "element_class": "Drain",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pipe",
        "source_uid": "junction",
        "target_uid": "storage",
        "diameter": 1.0,
        "length": 10.0,
        "roughness": 10.0,
        "model": "hazen-williams",
        "check_valve": False,
        "display_name": "pipe",
        "element_class": "Pipe",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump",
        "source_uid": "junction",
        "target_uid": "storage",
        "speed": [
            {"flow": 1, "head": 1, "speed": 1},
            {"flow": 3, "head": 3, "speed": 1},
        ],
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine",
        "source_uid": "junction",
        "target_uid": "storage",
        "speed": [
            {"flow": 1, "head": 1, "speed": 1},
            {"flow": 3, "head": 3, "speed": 1},
        ],
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump2",
        "source_uid": "junction",
        "target_uid": "storage",
        "speed": [
            {"flow": 1, "head": 1, "speed": 1},
            {"flow": 3, "head": 3, "speed": 1},
        ],
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "max_speed": 1.0,
        "max_head": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine2",
        "source_uid": "junction",
        "target_uid": "storage",
        "speed": [
            {"flow": 1, "head": 1, "speed": 1},
            {"flow": 3, "head": 3, "speed": 1},
        ],
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "max_speed": 1.0,
        "max_head": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "valve",
        "source_uid": "junction",
        "target_uid": "storage",
        "diameter": 10.0,
        "model": "prv",
        "coefficient": 1.0,
        "setting": 0.0,
        "display_name": "valve",
        "element_class": "Valve",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "weir",
        "source_uid": "junction",
        "target_uid": "storage",
        "flow_model": "free",
        "coefficient": 1.0,
        "min_crest_level": 0.0,
        "max_crest_level": 0.0,
        "crest_width": 10.0,
        "display_name": "weir",
        "element_class": "Weir",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "flow_boundary",
        "location": {"x": 0.0, "y": 0.0, "z": 0.0},
        "display_name": "flow_boundary",
        "schematic_location": {"x": 0.0, "y": 0.0, "z": 0.0},
        "element_class": "FlowBoundary",
        "collection": "nodes",
        "domain": "water",
    },
    {
        "uid": "junction",
        "location": {"x": 0.0, "y": 1.0, "z": 0.0},
        "display_name": "junction",
        "schematic_location": {"x": 0.0, "y": 1.0, "z": 0.0},
        "element_class": "Junction",
        "collection": "nodes",
        "domain": "water",
    },
    {
        "uid": "level_boundary",
        "location": {"x": 1.0, "y": 0.0, "z": 0.0},
        "display_name": "level_boundary",
        "schematic_location": {"x": 1.0, "y": 0.0, "z": 0.0},
        "element_class": "LevelBoundary",
        "collection": "nodes",
        "domain": "water",
    },
    {
        "uid": "storage",
        "location": {"x": 1.0, "y": 1.0, "z": 0.0},
        "level_volume": [
            {"level": 0.0, "volume": 0.0},
            {"level": 10.0, "volume": 10.0},
            {"level": 20.0, "volume": 20.0},
            {"level": 30.0, "volume": 30.0},
        ],
        "level_volume_interp": "bspline",
        "level_capacity": [
            {"level": 0.0, "capacity": 0.0},
            {"level": 10.0, "capacity": 10.0},
            {"level": 20.0, "capacity": 20.0},
            {"level": 30.0, "capacity": 30.0},
        ],
        "level_capacity_interp": "bspline",
        "display_name": "storage",
        "schematic_location": {"x": 1.0, "y": 1.0, "z": 0.0},
        "element_class": "Storage",
        "flow_boundary": False,
        "collection": "nodes",
        "domain": "water",
    },
]

bad_elements = [
    {
        "uid": "channel1",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": {
                "distance": 50.0,
                "roughness": 10.0,
                "cross_section": [
                    {"z": 0, "lr": -5},
                    {"z": 10, "lr": -5},
                    {"z": 0, "lr": 5},
                    {"z": 10, "lr": 5},
                ],
            },
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel2",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": [
                {
                    "distance": 25.0,
                    "roughness": 10.0,
                    "cross_section": [{"z": 0, "lr": -5}, {"z": 10, "lr": 5}],
                },
                {
                    "distance": 75.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
            ],
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel2b",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": {
                "roughness": 10.0,
                "cross_section": [
                    {"z": 10, "lr": -5},
                    {"z": 0, "lr": 5},
                    {"z": 0, "lr": -5},
                    {"z": 10, "lr": 5},
                ],
            },
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel3",
        "source_uid": "junction",
        "target_uid": "storage",
        "hydrologic_routing": {
            "model": "muskingum",
            "stations": {"model": "muskingum", "k": 100.0, "x": 100.0},
        },
        "length": 100.0,
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel4",
        "source_uid": "junction",
        "target_uid": "storage",
        "hydrologic_routing": {
            "model": "muskingum-cunge",
            "stations": {
                "muskingum-cunge": {"model": "muskingum-cunge", "k": 100.0, "x": 100.0}
            },
        },
        "length": 100.0,
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel5",
        "source_uid": "junction",
        "target_uid": "storage",
        "hydrologic_routing": {
            "model": "muskingum",
            "stations": {"muskingum": {"model": "muskingum", "k": 100.0, "x": 100.0}},
        },
        "length": 100.0,
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel6",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": [
                {
                    "distance": 50.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
                {
                    "distance": 50.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
            ],
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel7",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": [
                {
                    "distance": 25.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
                {
                    "distance": 101.0,
                    "roughness": 10.0,
                    "cross_section": [
                        {"z": 0, "lr": -5},
                        {"z": 10, "lr": -5},
                        {"z": 0, "lr": 5},
                        {"z": 10, "lr": 5},
                    ],
                },
            ],
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel8",
        "source_uid": "junction",
        "target_uid": "storage",
        "hydrologic_routing": {
            "model": "muskingum",
            "stations": {"muskingum": {"model": "muskingum", "k": 100.0, "x": 100.0}},
        },
        "length": 100.0,
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "channel9",
        "source_uid": "junction",
        "target_uid": "storage",
        "length": 100.0,
        "hydraulic_routing": {
            "model": "saint-venant",
            "roughness_model": "chezy",
            "stations": {
                "roughness": 10.0,
                "cross_section": [
                    {"z": 0, "lr": -5},
                    {"z": 0, "lr": -5},
                    {"z": 0, "lr": 5},
                    {"z": 0, "lr": 5},
                ],
            },
        },
        "display_name": "channel",
        "element_class": "Channel",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump1",
        "source_uid": "junction",
        "target_uid": "storage",
        "max_speed": math.inf,
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine1",
        "source_uid": "junction",
        "target_uid": "storage",
        "max_speed": math.inf,
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump2",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 2.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine2",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 2.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
        "collection": "links",
    },
    {
        "uid": "pump3",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 2.0,
        "max_power": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine3",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 2.0,
        "max_power": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump4",
        "source_uid": "junction",
        "target_uid": "storage",
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine4",
        "source_uid": "junction",
        "target_uid": "storage",
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump5",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "min_head": 2.0,
        "max_head": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine5",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "min_head": 2.0,
        "max_head": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "pump5",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "min_speed": 2.0,
        "max_speed": 1.0,
        "display_name": "pump",
        "element_class": "Pump",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "turbine5",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 1.0,
        "max_flow": 1.0,
        "min_power": 1.0,
        "max_power": 1.0,
        "min_speed": 2.0,
        "max_speed": 1.0,
        "display_name": "turbine",
        "element_class": "Turbine",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "flow_controlled_structure",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 0.0,
        "max_flow": math.inf,
        "display_name": "flow_controlled_structure",
        "element_class": "FlowControlledStructure",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "flow_controlled_structure",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": -math.inf,
        "max_flow": 0.0,
        "display_name": "flow_controlled_structure",
        "element_class": "FlowControlledStructure",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "flow_controlled_structure",
        "source_uid": "junction",
        "target_uid": "storage",
        "min_flow": 2.0,
        "max_flow": 1.0,
        "display_name": "flow_controlled_structure",
        "element_class": "FlowControlledStructure",
        "collection": "links",
        "domain": "water",
    },
    {
        "uid": "storage",
        "location": {"x": 1.0, "y": 1.0, "z": 0.0},
        "level_volume": [
            {"level": 0.0, "volume": 10.0},
            {"level": 10.0, "volume": 10.0},
        ],
        "display_name": "storage",
        "schematic_location": {"x": 1.0, "y": 1.0, "z": 0.0},
        "element_class": "Storage",
        "domain": "water",
    },
    {
        "uid": "storage",
        "location": {"x": 1.0, "y": 1.0, "z": 0.0},
        "level_volume": [
            {"level": 0.0, "volume": 11.0},
            {"level": 10.0, "volume": 10.0},
        ],
        "display_name": "storage",
        "schematic_location": {"x": 1.0, "y": 1.0, "z": 0.0},
        "element_class": "Storage",
        "domain": "water",
    },
    {
        "uid": "weir",
        "source_uid": "junction",
        "target_uid": "storage",
        "flow_model": "free",
        "coefficient": 1.0,
        "min_crest_level": 1.0,
        "max_crest_level": 0.0,
        "crest_width": 10.0,
        "display_name": "weir",
        "element_class": "Weir",
        "domain": "water",
    },
]


@pytest.mark.parametrize("element", elements)
def test_parse(element: Dict[str, Any]) -> None:
    instance = element_from_dict(element)
    reserialised = json.loads(instance.json(exclude_none=True, exclude_unset=True))
    assert element == reserialised


@pytest.mark.parametrize("element", bad_elements)
def test_parse_bad(element: Dict[str, Any]) -> None:
    with pytest.raises(ValueError):
        element_from_dict(element)


def test_util_entry_point() -> None:
    assert elements_mapping["water"]["links"]
    assert elements_mapping["water"]["nodes"]
    assert set(elements_mapping["water"]["links"].values()).issubset(set(all_links))
    assert set(elements_mapping["water"]["nodes"].values()).issubset(set(all_nodes))


@pytest.mark.parametrize("element", elements)
def test_correct_element_class(element: Any) -> None:
    instance = element_from_dict(element)
    assert instance.element_class == instance.__class__.__name__
