"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Namer = exports.Format = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
/**
 * Too many naming conventions?
 *
 * @stability stable
 */
var Format;
(function (Format) {
    Format["KEBAB"] = "kebab";
    Format["PASCAL"] = "pascal";
    Format["SNAKE"] = "snake";
    Format["CAMEL"] = "camel";
})(Format = exports.Format || (exports.Format = {}));
/**
 * @stability stable
 */
class Namer {
    /**
     * Create a namer.
     *
     * @param parts an array of strings to be composed into a name.
     * @param props modify the behavior of namer.
     * @stability stable
     */
    constructor(parts, props) {
        this.props = props;
        const defaultedDeleteCharacters = (props === null || props === void 0 ? void 0 : props.deleteCharacters) || '-_';
        const rawDeleteCharactersRe = '[' + [...defaultedDeleteCharacters].map(escapeRegex).join('|') + ']';
        const deleteCharactersRe = new RegExp(rawDeleteCharactersRe, 'gi');
        if (props === null || props === void 0 ? void 0 : props.illegalCharacters) {
            throw new Error('not implemented');
        }
        if (props === null || props === void 0 ? void 0 : props.uniqueSeed) {
            const shasum = crypto.createHash('sha1');
            if (props.uniqueSeed instanceof String) {
                shasum.update(props.uniqueSeed);
            }
            else {
                // everthing is an object in typescript
                shasum.update(JSON.stringify(props.uniqueSeed));
            }
            this._unique = shasum.digest('hex').substring(0, 7);
        }
        this._parts = parts;
        this._parts = this.splitParts();
        // Ignore empty parts
        this.parts = this._parts
            .filter((element, _index, _array) => {
            return element != '';
        })
            // Make sure the parts are lower cased
            // Remove illegal characters from parts: - _
            // Enforce maxPartLength
            .map((element, _index, _array) => {
            const r = element.toLowerCase().replace(deleteCharactersRe, '');
            if (!(props === null || props === void 0 ? void 0 : props.maxPartLength)) {
                // no max
                return r;
            }
            if (r.length <= props.maxPartLength) {
                // already short enough
                return r;
            }
            if (props.maxPartLengthTruncateHead) {
                // trim the head
                return r.slice(r.length - props.maxPartLength);
            }
            else {
                // trim the tail
                return r.slice(0, props.maxPartLength);
            }
        });
    }
    splitParts() {
        // Split kebab and snake
        this._parts.forEach((part, index) => {
            const splitParts = part.split(/[-_ ]/);
            this._parts.splice(index, 1, ...splitParts);
        });
        // Split camel or pascal
        // TODO: need to dry this out.
        this._parts.forEach((part, index) => {
            if (part.toUpperCase() == part) {
                return;
            }
            const splitParts = part.split(/(?=[A-Z])/);
            this._parts.splice(index, 1, ...splitParts);
        });
        this._parts.forEach((part, index) => {
            if (part.toUpperCase() == part) {
                return;
            }
            const splitParts = part.split(/(?=[A-Z])/);
            this._parts.splice(index, 1, ...splitParts);
        });
        return this._parts;
    }
    /**
     * @stability stable
     */
    get partsWithUnique() {
        return this._unique ? [...this.parts, this._unique] : this.parts;
    }
    /**
     * @stability stable
     */
    enforceMaxLength(raw) {
        var _b, _c;
        if (((_b = this.props) === null || _b === void 0 ? void 0 : _b.maxLength) === undefined) {
            return raw;
        }
        if (raw.length <= this.props.maxLength) {
            return raw;
        }
        if ((_c = this.props) === null || _c === void 0 ? void 0 : _c.maxLengthTruncateHead) {
            return raw.slice(raw.length - this.props.maxLength);
        }
        if (this._unique) {
            return raw.slice(0, this.props.maxLength - this._unique.length) + this._unique;
        }
        return raw.slice(0, this.props.maxLength);
    }
    /**
     * camelCase.
     *
     * @stability stable
     */
    get camel() {
        if (!this._camel) {
            this._camel = this.enforceMaxLength(this.partsWithUnique
                .map((value, index, _array) => {
                return index == 0 ? value : value.charAt(0).toUpperCase() + value.slice(1);
            })
                .reduce((previousValue, currentValue, _currentIndex, _array) => {
                return `${previousValue}${currentValue}`;
            }));
        }
        return this._camel;
    }
    /**
     * kebab-case.
     *
     * @stability stable
     */
    get kebab() {
        if (!this._kebab) {
            this._kebab = this.enforceMaxLength(this.partsWithUnique.reduce((previousValue, currentValue, _currentIndex, _array) => {
                return `${previousValue}-${currentValue}`;
            }));
        }
        return this._kebab;
    }
    /**
     * PascalCase.
     *
     * @stability stable
     */
    get pascal() {
        if (!this._pascal) {
            this._pascal = this.enforceMaxLength(this.partsWithUnique
                .map((value, _index, _array) => {
                return value.charAt(0).toUpperCase() + value.slice(1);
            })
                .reduce((previousValue, currentValue, _currentIndex, _array) => {
                return `${previousValue}${currentValue}`;
            }));
        }
        return this._pascal;
    }
    /**
     * snake_case.
     *
     * @stability stable
     */
    get snake() {
        if (!this._snake) {
            this._snake = this.enforceMaxLength(this.partsWithUnique.reduce((previousValue, currentValue, _currentIndex, _array) => {
                return `${previousValue}_${currentValue}`;
            }));
        }
        return this._snake;
    }
    /**
     * Create a new Namer with the added prefix.
     *
     * @param prefix the prefix to add.
     * @param props properties to over-ride the parent props.
     * @stability stable
     */
    addPrefix(prefix, props) {
        const p = prefix instanceof Namer ? prefix.parts : prefix;
        return new Namer([...p, ...this.parts], { ...this.props, ...props });
    }
    /**
     * Create a new Namer with the added suffix.
     *
     * @param suffix the suffix to add.
     * @param props properties to over-ride the parent props.
     * @stability stable
     */
    addSuffix(suffix, props) {
        const s = suffix instanceof Namer ? suffix.parts : suffix;
        return new Namer([...this.parts, ...s], { ...this.props, ...props });
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any, @typescript-eslint/explicit-module-boundary-types
    /**
     * Create a new Namer with a unique suffix.
     *
     * @param uniqueItem : any value to use as the seed for generating a unique hash.
     * @stability stable
     */
    unique(uniqueItem) {
        return this.addSuffix([], { uniqueSeed: uniqueItem });
    }
    /**
     * @stability stable
     */
    toString() {
        var _b;
        switch ((_b = this.props) === null || _b === void 0 ? void 0 : _b.defaultFormat) {
            case Format.CAMEL:
                return this.camel;
            case Format.KEBAB:
                return this.kebab;
            case Format.PASCAL:
                return this.pascal;
            case Format.SNAKE:
                return this.snake;
            default:
                throw new Error('default stringer not defined');
        }
    }
}
exports.Namer = Namer;
_a = JSII_RTTI_SYMBOL_1;
Namer[_a] = { fqn: "multi-convention-namer.Namer", version: "0.0.0" };
// https://stackoverflow.com/a/3561711
function escapeRegex(s) {
    return s.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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