import React, { useState, useEffect, useCallback, useMemo, useRef } from 'react';
import Popover from 'react-popover';
import { createUseStyles } from 'react-jss';
import { useStoreState } from 'pullstate';
import { ExtensionStore } from '../../stores/ExtensionStore';
import { TextField } from '../TextField';
import { checkVariableNameValid } from '../../utils/Helpers';
const useStyles = createUseStyles({
    textField: {
        width: '200px'
    },
    instructions: {
        padding: '8px',
        background: '#f5f5f5',
        fontSize: '8pt',
        color: '#808080'
    },
    proceedButton: {
        alignItems: 'center',
        padding: '4px',
        lineHeight: 0,
        cursor: 'pointer'
    },
    proceedIcon: {
        color: '#2196F390',
        fontSize: '16px',
        lineHeight: '24px',
        '&:hover': {
            color: '#2196F3'
        }
    },
    error: {
        color: '#d32f2f'
    }
});
export const AddToNotebookPopover = ({ children, did, type }) => {
    const classes = useStyles();
    const textFieldRef = useRef();
    const [open, setOpen] = useState(false);
    const [error, setError] = useState();
    const [varName, setVarName] = useState('');
    const activeNotebookPanel = useStoreState(ExtensionStore, s => s.activeNotebookPanel);
    const activeNotebookAttachments = useStoreState(ExtensionStore, s => s.activeNotebookAttachment);
    const existingAttachmentVariableNames = useMemo(() => (activeNotebookAttachments ? activeNotebookAttachments.map(a => a.variableName) : []), [activeNotebookAttachments]);
    const didAttached = useMemo(() => (activeNotebookAttachments ? !!activeNotebookAttachments.find(a => a.did === did) : false), [activeNotebookAttachments]);
    const setActiveNotebookAttachments = (attachments) => {
        ExtensionStore.update(s => {
            s.activeNotebookAttachment = attachments;
        });
    };
    const escFunction = useCallback(event => {
        if (event.keyCode === 27) {
            setOpen(false);
        }
    }, []);
    useEffect(() => {
        document.addEventListener('keydown', escFunction, false);
        return () => {
            document.removeEventListener('keydown', escFunction, false);
        };
    }, []);
    const addAttachment = () => {
        if (!checkVariableNameValid(varName) || checkVariableNameExists(varName)) {
            return;
        }
        setOpen(false);
        const attachment = {
            did,
            type,
            variableName: varName
        };
        const notebookAttachments = activeNotebookAttachments ? [...activeNotebookAttachments, attachment] : [attachment];
        setActiveNotebookAttachments(notebookAttachments);
    };
    const handleKeyPress = (e) => {
        if (e.key === 'Enter') {
            addAttachment();
        }
    };
    const checkVariableNameExists = (variableName) => {
        return existingAttachmentVariableNames.includes(variableName);
    };
    const handleTextChange = (variableName) => {
        setVarName(variableName);
        if (!!variableName && !checkVariableNameValid(variableName)) {
            setError('Invalid variable name');
        }
        else if (checkVariableNameExists(variableName)) {
            setError('Variable name exists');
        }
        else {
            setError(undefined);
        }
    };
    const proceedButton = (React.createElement("div", { className: classes.proceedButton, onClick: addAttachment },
        React.createElement("i", { className: `${classes.proceedIcon} material-icons` }, "arrow_forward")));
    const popoverBody = (React.createElement(React.Fragment, null,
        React.createElement(TextField, { ref: textFieldRef, className: classes.textField, outlineColor: "#FFFFFF", placeholder: "Enter a variable name", onKeyPress: handleKeyPress, after: proceedButton, value: varName, onChange: e => handleTextChange(e.target.value) }),
        React.createElement("div", { className: classes.instructions },
            !error && React.createElement("span", null, "Press Enter to proceed, Esc to cancel"),
            !!error && React.createElement("span", { className: classes.error }, error))));
    const openPopover = () => {
        setOpen(true);
        setVarName('');
        setTimeout(() => {
            textFieldRef.current.focus();
        }, 10);
    };
    return (React.createElement(React.Fragment, null, !!activeNotebookPanel && !didAttached && (React.createElement(Popover, { enterExitTransitionDurationMs: 0, isOpen: open, preferPlace: "below", body: popoverBody, onOuterAction: () => setOpen(false) },
        React.createElement("div", { onClick: openPopover }, children)))));
};
