"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const punycode_1 = require("punycode/");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
const user_pool_resource_server_1 = require("./user-pool-resource-server");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
_a = JSII_RTTI_SYMBOL_1;
UserPoolOperation[_a] = { fqn: "@aws-cdk/aws-cognito.UserPoolOperation", version: "1.156.0" };
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-email-sender.html
 */
UserPoolOperation.CUSTOM_EMAIL_SENDER = new UserPoolOperation('customEmailSender');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-sms-sender.html
 */
UserPoolOperation.CUSTOM_SMS_SENDER = new UserPoolOperation('customSmsSender');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    addResourceServer(id, options) {
        return new user_pool_resource_server_1.UserPoolResourceServer(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this.triggers = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const signIn = this.signInConfiguration(props);
        if (props.customSenderKmsKey) {
            const kmsKey = props.customSenderKmsKey;
            this.triggers.kmsKeyId = kmsKey.keyArn;
        }
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                let trigger;
                switch (t) {
                    case 'customSmsSender':
                    case 'customEmailSender':
                        if (!this.triggers.kmsKeyId) {
                            throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                        }
                        trigger = props.lambdaTriggers[t];
                        const version = 'V1_0';
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = {
                                lambdaArn: trigger.functionArn,
                                lambdaVersion: version,
                            };
                        }
                        break;
                    default:
                        trigger = props.lambdaTriggers[t];
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = trigger.functionArn;
                        }
                        break;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.emailBody,
            emailSubject: (_d = props.userInvitation) === null || _d === void 0 ? void 0 : _d.emailSubject,
            smsMessage: (_e = props.userInvitation) === null || _e === void 0 ? void 0 : _e.smsMessage,
        };
        const selfSignUpEnabled = (_f = props.selfSignUpEnabled) !== null && _f !== void 0 ? _f : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        if (props.email && props.emailSettings) {
            throw new Error('you must either provide "email" or "emailSettings", but not both');
        }
        const emailConfiguration = props.email ? props.email._bind(this) : undefinedIfNoKeys({
            from: encodePuny((_g = props.emailSettings) === null || _g === void 0 ? void 0 : _g.from),
            replyToEmailAddress: encodePuny((_h = props.emailSettings) === null || _h === void 0 ? void 0 : _h.replyTo),
        });
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.any({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsAuthenticationMessage: this.mfaMessage(props),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration,
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
            deviceConfiguration: props.deviceTracking,
        });
        userPool.applyRemovalPolicy(props.removalPolicy);
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        let userPoolArn = core_1.Stack.of(scope).formatArn({
            service: 'cognito-idp',
            resource: 'userpool',
            resourceName: userPoolId,
        });
        return UserPool.fromUserPoolArn(scope, id, userPoolArn);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        const arnParts = core_1.Stack.of(scope).splitArn(userPoolArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arnParts.resourceName) {
            throw new Error('invalid user pool ARN');
        }
        const userPoolId = arnParts.resourceName;
        class ImportedUserPool extends UserPoolBase {
            constructor() {
                super(scope, id, {
                    account: arnParts.account,
                    region: arnParts.region,
                });
                this.userPoolArn = userPoolArn;
                this.userPoolId = userPoolId;
            }
        }
        return new ImportedUserPool();
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolOperation(operation);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTrigger);
            }
            throw error;
        }
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation.operationName} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        switch (operation.operationName) {
            case 'customEmailSender':
            case 'customSmsSender':
                if (!this.triggers.kmsKeyId) {
                    throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                }
                this.triggers[operation.operationName] = {
                    lambdaArn: fn.functionArn,
                    lambdaVersion: 'V1_0',
                };
                break;
            default:
                this.triggers[operation.operationName] = fn.functionArn;
        }
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: core_1.Lazy.string({ produce: () => this.userPoolArn }),
        });
    }
    mfaMessage(props) {
        const CODE_TEMPLATE = '{####}';
        const MAX_LENGTH = 140;
        const message = props.mfaMessage;
        if (message && !core_1.Token.isUnresolved(message)) {
            if (!message.includes(CODE_TEMPLATE)) {
                throw new Error(`MFA message must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (message.length > MAX_LENGTH) {
                throw new Error(`MFA message must be between ${CODE_TEMPLATE.length} and ${MAX_LENGTH} characters`);
            }
        }
        return message;
    }
    verificationMessageConfiguration(props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailStyle) !== null && _d !== void 0 ? _d : VerificationEmailStyle.CODE;
        const emailSubject = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.emailSubject) !== null && _f !== void 0 ? _f : 'Verify your new account';
        const smsMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.smsMessage) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_m = (_l = props.userVerification) === null || _l === void 0 ? void 0 : _l.emailBody) !== null && _m !== void 0 ? _m : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _c;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_c = props.signInAliases) !== null && _c !== void 0 ? _c : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _c, _d, _e;
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
                snsRegion: props.snsRegion,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = ((_c = props.signInAliases) === null || _c === void 0 ? void 0 : _c.phone) === true || ((_d = props.autoVerify) === null || _d === void 0 ? void 0 : _d.phone) === true;
        const roleRequired = (mfaConfiguration === null || mfaConfiguration === void 0 ? void 0 : mfaConfiguration.includes('SMS_MFA')) || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = core_1.Names.uniqueId(this).slice(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = (_e = props.smsRole) !== null && _e !== void 0 ? _e : new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
            snsRegion: props.snsRegion,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _c, _d, _e, _f, _g, _h;
        const tempPasswordValidity = (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_d = props.passwordPolicy.minLength) !== null && _d !== void 0 ? _d : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireLowercase,
            requireUppercase: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireUppercase,
            requireNumbers: (_g = props.passwordPolicy) === null || _g === void 0 ? void 0 : _g.requireDigits,
            requireSymbols: (_h = props.passwordPolicy) === null || _h === void 0 ? void 0 : _h.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _c, _d;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_c = attr.mutable) !== null && _c !== void 0 ? _c : true,
                    required: (_d = attr.required) !== null && _d !== void 0 ? _d : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _c, _d, _e, _f, _g, _h, _j, _k;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.min) === null || _d === void 0 ? void 0 : _d.toString(),
                    maxValue: (_f = (_e = attrConfig.numberConstraints) === null || _e === void 0 ? void 0 : _e.max) === null || _f === void 0 ? void 0 : _f.toString(),
                };
                const stringConstraints = {
                    minLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.minLen) === null || _h === void 0 ? void 0 : _h.toString(),
                    maxLength: (_k = (_j = attrConfig.stringConstraints) === null || _j === void 0 ? void 0 : _j.maxLen) === null || _k === void 0 ? void 0 : _k.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _c;
        const accountRecovery = (_c = props.accountRecovery) !== null && _c !== void 0 ? _c : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
_b = JSII_RTTI_SYMBOL_1;
UserPool[_b] = { fqn: "@aws-cdk/aws-cognito.UserPool", version: "1.156.0" };
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
function encodePuny(input) {
    return input !== undefined ? punycode_1.toASCII(input) : input;
}
//# sourceMappingURL=data:application/json;base64,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