import sys
from collections.abc import Sequence, Mapping
from configparser import RawConfigParser
from datetime import timedelta
from enum import Enum
from json import dumps
from stat import S_IRUSR, S_IWUSR
from typing import Iterable, Iterator, Dict, Any, Optional, List
from uuid import UUID

import typer
import click
from click import Context
from prettytable import PrettyTable
from sys import version as python_version
from time import perf_counter
from typer import Typer
from zanshinsdk import Client, AlertState, AlertSeverity, __version__ as sdk_version
from zanshinsdk.client import Roles, ScanTargetKind, _CONFIG_DIR, _CONFIG_FILE

from zanshincli import __version__ as cli_version


class OrderedCommands(click.Group):
    def list_commands(self, ctx: Context) -> Iterable[str]:
        return self.commands.keys()

class OutputFormat(str, Enum):
    """
    Used to specify command-line parameters indicating output format.
    """
    JSON = "json"
    TABLE = "table"
    CSV = "csv"
    HTML = "html"


###################################################
# Exchandler
###################################################

def zanshin_exchandler(_, value, __):
    print(value)

sys.excepthook = zanshin_exchandler


###################################################
# Utils
###################################################

def format_field(value: Any) -> str:
    """
    Function that formats a single field for output on a table or CSV output, in order to deal with nested arrays or
    objects in the JSON outputs of the API.
    :param value: the value to format
    :return: a string that is fit for console output
    """
    if isinstance(value, Sequence) and not isinstance(value, (str, bytes)):
        if all(isinstance(x, (str, bytes, int, float)) for x in value):
            return ", ".join([str(x) for x in value])
        return dumps(value)

    if isinstance(value, Mapping):
        return dumps(value)

    return value


def output_iterable(iterator: Iterator[Dict], empty: Any = None) -> None:
    """
    Function that iterates over a series of dicts representing JSON objects returned by API list operations, and which
    outputs them using typer.echo in the specified format. Will use streaming processing for JSON, all others need to
    load all responses in memory in a PrettyTable prior to output, which could be problematic for large number of
    entries.
    :param iterator: the iterator containing the JSON objects
    :return: None
    """
    global global_options

    global_options['entries'] = 0
    if global_options['format'] is OutputFormat.JSON:
        for entry in iterator:
            typer.echo(dumps(entry, indent=4))
            global_options['entries'] += 1
    else:
        table = PrettyTable()
        for entry in iterator:
            if not table.field_names:
                table.field_names = sorted(entry.keys())
            else:
                for k in entry.keys():
                    if k not in table.field_names:
                        table.add_column(k, [empty] * global_options['entries'])
            table.add_row([format_field(entry.get(fn, empty)) for fn in table.field_names])
            global_options['entries'] += 1
        if global_options['format'] is OutputFormat.TABLE:
            typer.echo(table.get_string())
        elif global_options['format'] is OutputFormat.CSV:
            typer.echo(table.get_csv_string())
        elif global_options['format'] is OutputFormat.HTML:
            typer.echo(table.get_html_string())
        else:
            raise NotImplementedError(f"unexpected format type {global_options['format']}")

def dump_json(out: Dict) -> None:
    typer.echo(dumps(out, indent=4))

###################################################
# Main App
###################################################

global_options: dict = {'entries': 1}
main_app: Typer = typer.Typer(cls=OrderedCommands)


@main_app.callback()
def global_options_callback(ctx: typer.Context,
                            profile: str = typer.Option("default",
                                                        help="Configuration file section to read API key and configutation from"),
                            output_format: OutputFormat = typer.Option(OutputFormat.JSON, '--format',
                                                                       help="Output format to use for list operations",
                                                                       case_sensitive=False),
                            verbose: bool = typer.Option(True, help="Print timiing and other information to stderr")):
    """
    Command-line utility to interact with the Zanshin SaaS service offered by Tenchi Security
    (https://tenchisecurity.com), go to https://github.com/tenchi-security/zanshin-cli for license, source code and
    documentation
    """
    if verbose:
        # print summary of data processed and elapsed time at the end of the execution
        start_time = perf_counter()

        def print_elapsed_time():
            typer.echo(
                f"zanshin: {global_options['entries']} object(s) processed in {timedelta(seconds=perf_counter() - start_time)}",
                err=True)

        ctx.call_on_close(print_elapsed_time)

    global_options['verbose'] = verbose
    global_options['profile'] = profile
    global_options['format'] = output_format


@main_app.command()
def init():
    """
    Update settings on configuration file.
    """
    cfg = RawConfigParser()
    cfg.read(_CONFIG_FILE)
    typer.echo("This command will allow you to set up profiles in the configuration file.")
    profile = typer.prompt("Please enter the profile name to use", default=global_options['profile'])
    if cfg.has_section(profile):
        typer.confirm("Profile already exists. Overwrite?", abort=True)
    else:
        cfg.add_section(profile)
    cfg.set(profile, "api_key", typer.prompt("Please enter the API key", hide_input=True))
    _CONFIG_DIR.mkdir(parents=True, exist_ok=True)
    with _CONFIG_FILE.open('w') as f:
        cfg.write(f)
    _CONFIG_FILE.chmod(S_IRUSR | S_IWUSR)

@main_app.command()
def version():
    """
    Display the program and Python versions in use.
    """
    typer.echo(f'Zanshin CLI v{cli_version}')
    typer.echo(f'Zanshin Python SDK v{sdk_version}')
    typer.echo(f'Python {python_version}')

###################################################
# Account App
###################################################

account_app = typer.Typer(cls=OrderedCommands)
main_app.add_typer(account_app, name="account",
                   help="Operations on user the API key owner has direct access to")

@account_app.command(name='me')
def account_me():
    """
    Returns the details of the user account that owns the API key used by this Connection instance as per
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_me())

###################################################
# Account Invites App
###################################################

invites_app = typer.Typer(cls=OrderedCommands)
account_app.add_typer(invites_app, name="invites",
                      help="Operations on invites from account the API key owner has direct access to")

@invites_app.command(name='list')
def account_invite_list():
    """
    Iterates over the invites of current logged user.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_invites())

@invites_app.command(name='get')
def account_invite_get(invite_id: UUID = typer.Argument(..., help="UUID of the invite")):
    """
    Gets an specific invitation details, it only works if the invitation was made for the current logged user.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_invite(invite_id))

@invites_app.command(name='accept')
def account_invite_accept(invite_id: UUID = typer.Argument(..., help="UUID of the invite")):
    """
    Accepts an inivitation with the informed ID, it only works if the user accepting the invitation is the user that received the invitation.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_invite(invite_id))

###################################################
# Account API key App
###################################################

api_key_app = typer.Typer(cls=OrderedCommands)
account_app.add_typer(api_key_app, name="api_key",
                      help="Operations on API keys from account the API key owner has direct access to")

@api_key_app.command(name='list')
def account_api_key_list():
    """
    Iterates over the API keys of current logged user.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_api_keys())

@api_key_app.command(name='create')
def account_api_key_create(name: str = typer.Argument(..., help="Name of the new API key")):
    """
    Creates a new API key for the current logged user, API Keys can be used to interact with the zanshin api directly on behalf of that user.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.create_api_key(name))

@api_key_app.command(name='delete')
def account_api_key_delete(api_key_id: UUID = typer.Argument(..., help="UUID of the invite to delete")):
    """
    Deletes a given API key by its id, it will only work if the informed ID belongs to the current logged user.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.delete_api_key(api_key_id))

###################################################
# Organization App
###################################################

organization_app = typer.Typer(cls=OrderedCommands)
main_app.add_typer(organization_app, name="organization",
                   help="Operations on organizations the API key owner has direct access to")


@organization_app.command(name='list')
def organization_list():
    """
    Lists the organizations this user has direct access to as a member.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organizations())

@organization_app.command(name='get')
def organization_get(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Gets an organization given its ID.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization(organization_id))

@organization_app.command(name='update')
def organization_update(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    name: Optional[str] = typer.Argument(None, help="Name of the organization"),
    picture: Optional[str] = typer.Argument(None, help="Picture of the organization"),
    email: Optional[str] = typer.Argument(None, help="Contact e-mail of the organization")
    ):
    """
    Gets an organization given its ID.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.update_organization(organization_id, name, picture, email))

###################################################
# Organization Member App
###################################################

organization_member_app = typer.Typer(cls=OrderedCommands)
organization_app.add_typer(organization_member_app, name="member",
                   help="Operations on members of organization the API key owner has direct access to")

@organization_member_app.command(name='list')
def organization_member_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the members of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_members(organization_id))

@organization_member_app.command(name='get')
def organization_member_get(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_id: UUID = typer.Argument(..., help="UUID of the organization member")
    ):
    """
    Get organization member.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization_member(organization_id, organization_member_id))

@organization_member_app.command(name='update')
def organization_member_update(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_id: UUID = typer.Argument(..., help="UUID of the organization member"),
    role: Optional[List[Roles]] = typer.Option([x.value for x in Roles],
                                                help="Role of the organization member",
                                                case_sensitive=False)
    ):
    """
    Update organization member.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.update_organization_member(organization_id, organization_member_id, role))

@organization_member_app.command(name='delete')
def organization_member_delete(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_id: UUID = typer.Argument(..., help="UUID of the organization member")
    ):
    """
    Delete organization member.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.delete_organization_member(organization_id, organization_member_id))

###################################################
# Organization Member Invite App
###################################################

organization_member_invite_app = typer.Typer(cls=OrderedCommands)
organization_member_app.add_typer(organization_member_invite_app, name="invite",
                   help="Operations on member invites of organization the API key owner has direct access to")

@organization_member_invite_app.command(name='list')
def organization_member_invite_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the member invites of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_members_invites(organization_id))

@organization_member_invite_app.command(name='create')
def organization_member_invite_create(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_invite_email: str = typer.Argument(..., help="E-mail of the organization member"),
    organization_member_invite_role: Optional[List[Roles]] = typer.Option([x.value for x in Roles],
                                                                           help="Role of the organization member",
                                                                           case_sensitive=False)
    ):
    """
    Create organization member invite.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.create_organization_members_invite(organization_id, organization_member_invite_email, organization_member_invite_role))

@organization_member_invite_app.command(name='get')
def organization_member_invite_get(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_invite_email: str = typer.Argument(..., help="E-mail of the organization member invite")
    ):
    """
    Get organization member invite.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization_member(organization_id, organization_member_invite_email))

@organization_member_invite_app.command(name='delete')
def organization_member_invite_delete(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_invite_email: str = typer.Argument(..., help="E-mail of the organization member")
    ):
    """
    Delete organization member invite.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.delete_organization_member_invite(organization_id, organization_member_invite_email))

@organization_member_invite_app.command(name='resend')
def organization_member_invite_resend(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_member_invite_email: str = typer.Argument(..., help="E-mail of the organization member")
    ):
    """
    Resend organization member invitation.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.resend_organization_member_invite(organization_id, organization_member_invite_email))

###################################################
# Organization Follower App
###################################################

organization_follower_app = typer.Typer(cls=OrderedCommands)
organization_app.add_typer(organization_follower_app, name="follower",
                   help="Operations on followers of organization the API key owner has direct access to")

@organization_follower_app.command(name='list')
def organization_follower_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the followers of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_followers(organization_id))

@organization_follower_app.command(name='stop')
def organization_follower_stop(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_follower_id: UUID = typer.Argument(..., help="UUID of the organization follower")
    ):
    """
    Stops one organization follower of another.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.stop_organization_follower(organization_id, organization_follower_id))

###################################################
# Organization Follower Request App
###################################################

organization_follower_request_app = typer.Typer(cls=OrderedCommands)
organization_follower_app.add_typer(organization_follower_request_app, name="request",
                   help="Operations on follower requests of organization the API key owner has direct access to")

@organization_follower_request_app.command(name='list')
def organization_follower_request_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the follower requests of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_followers(organization_id))

@organization_follower_request_app.command(name='create')
def organization_follower_request_create(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    token: UUID = typer.Argument(..., help="Token of the follower request")
    ):
    """
    Create organization follower request.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.create_organization_follower_request(organization_id, token))

@organization_follower_request_app.command(name='get')
def organization_follower_request_get(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    token: UUID = typer.Argument(..., help="Token of the follower request")
    ):
    """
    Get organization follower request.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization_follower_request(organization_id, token))

@organization_follower_request_app.command(name='delete')
def organization_follower_request_delete(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    token: UUID = typer.Argument(..., help="Token of the follower request")
    ):
    """
    Delete organization follower request.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.delete_organization_follower_request(organization_id, token))

###################################################
# Organization Following App
###################################################

organization_following_app = typer.Typer(cls=OrderedCommands)
organization_app.add_typer(organization_following_app, name="following",
                   help="Operations on following of organization the API key owner has direct access to")

@organization_following_app.command(name='list')
def organization_following_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the following of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_following(organization_id))

@organization_following_app.command(name='stop')
def organization_following_stop(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    organization_following_id: UUID = typer.Argument(..., help="UUID of the organization following")
    ):
    """
    Stops one organization following of another.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.stop_organization_following(organization_id, organization_following_id))

###################################################
# Organization Following Request App
###################################################

organization_following_request_app = typer.Typer(cls=OrderedCommands)
organization_following_app.add_typer(organization_following_request_app, name="request",
                   help="Operations on following requests of organization the API key owner has direct access to")

@organization_following_request_app.command(name='list')
def organization_following_request_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the following requests of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_following_requests(organization_id))

@organization_following_request_app.command(name='get')
def organization_following_request_get(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    following_id: UUID = typer.Argument(..., help="UUID of the following request")
    ):
    """
    Returns a request received by an organization to follow another.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization_following_request(organization_id, following_id))

@organization_following_request_app.command(name='accept')
def organization_following_request_accept(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    following_id: UUID = typer.Argument(..., help="UUID of the following request")
    ):
    """
    Accepts a request to follow another organization.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.accept_organization_following_request(organization_id, following_id))

@organization_following_request_app.command(name='decline')
def organization_following_request_decline(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    following_id: UUID = typer.Argument(..., help="UUID of the following request")
    ):
    """
    Declines a request to follow another organization.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.decline_organization_following_request(organization_id, following_id))

###################################################
# Organization Scan Target App
###################################################

organization_scan_target_app = typer.Typer(cls=OrderedCommands)
organization_app.add_typer(organization_scan_target_app, name="scan_target",
                           help="Operations on scan targets from organizations the API key owner has direct access to")

@organization_scan_target_app.command(name='list')
def organization_scan_target_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization")):
    """
    Lists the scan targets of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_scan_targets(organization_id))

@organization_scan_target_app.command(name='create')
def organization_scan_target_create(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    kind: ScanTargetKind = typer.Argument(..., help="kind of the scan target"),
    name: str = typer.Argument(..., help="name of the scan target"),
    credential: str = typer.Argument(..., help="credential of the scan target"),
    schedule: str = typer.Argument("0 0 * * *", help="schedule of the scan target")
    ):
    """
    Create a new scan target in organization.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.create_organization_scan_target(organization_id, kind, name, credential, schedule))

@organization_scan_target_app.command(name='get')
def organization_scan_target_get(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target")
    ):
    """
    Get scan target of organization.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization_scan_target(organization_id, scan_target_id))

@organization_scan_target_app.command(name='update')
def organization_scan_target_update(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target"),
    name: Optional[str] = typer.Argument(None, help="name of the scan target"),
    schedule: Optional[str] = typer.Argument(None, help="schedule of the scan target")
    ):
    """
    Update scan target of organization.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.update_organization_scan_target(organization_id, scan_target_id, name, schedule))

@organization_scan_target_app.command(name='delete')
def organization_scan_target_delete(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target")
    ):
    """
    Delete scan target of organization.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.delete_organization_scan_target(organization_id, scan_target_id))

@organization_scan_target_app.command(name='check')
def organization_scan_target_check(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target")
    ):
    """
    Check scan target.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.check_organization_scan_target(organization_id, scan_target_id))

###################################################
# Organization Scan Target Scan App
###################################################

organization_scan_target_scan_app = typer.Typer(cls=OrderedCommands)
organization_scan_target_app.add_typer(organization_scan_target_scan_app, name="scan",
                           help="Operations on scan targets from organizations the API key owner has direct access to")

@organization_scan_target_scan_app.command(name='start')
def organization_scan_target_scan_start(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target")
    ):
    """
    Starts a scan on the specified scan target.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.start_organization_scan_target_scan(organization_id, scan_target_id))

@organization_scan_target_scan_app.command(name='list')
def organization_scan_target_scan_list(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target")
    ):
    """
    Lists the scan target scans of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_organization_scan_target_scans(organization_id, scan_target_id))

@organization_scan_target_scan_app.command(name='get')
def organization_scan_target_scan_get(
    organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
    scan_target_id: UUID = typer.Argument(..., help="UUID of the scan target"),
    scan_id: UUID = typer.Argument(..., help="UUID of the scan")
    ):
    """
    Get scan of scan target.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_organization_scan_target_scan(organization_id, scan_target_id, scan_id))

###################################################
# Alert
###################################################

alert_app = typer.Typer(cls=OrderedCommands)
main_app.add_typer(alert_app, name="alert",
                   help="Operations on alerts the API key owner has direct access to")

@alert_app.command(name='list')
def alert_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
               scan_target_id: Optional[List[UUID]] = typer.Option(None, help="Only list alerts from the specified scan targets."),
               state: Optional[List[AlertState]] = typer.Option(
                        [x.value for x in AlertState if x != AlertState.CLOSED],
                        help="Only list alerts in the specified states.", case_sensitive=False),
               severity: Optional[List[AlertSeverity]] = typer.Option([x.value for x in AlertSeverity],
                                                                       help="Only list alerts with the specified severities",
                                                                       case_sensitive=False)):
    """
    List alerts from a given organization, with optional filters by scan target, state or severity.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(
        client.iter_alerts(organization_id=organization_id, scan_target_ids=scan_target_id, states=state,
                           severities=severity))

@alert_app.command(name='list_following')
def alert_following_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                         following_ids: Optional[List[UUID]] = typer.Option(None, help="Only list alerts from the specified scan targets."),
                         state: Optional[List[AlertState]] = typer.Option(
                                [x.value for x in AlertState if x != AlertState.CLOSED],
                                help="Only list alerts in the specified states.", case_sensitive=False),
                         severity: Optional[List[AlertSeverity]] = typer.Option([x.value for x in AlertSeverity],
                                                                                help="Only list alerts with the specified severities",
                                                                                case_sensitive=False)):
    """
    List following alerts from a given organization, with optional filters by following ids, state or severity.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(
        client.iter_following_alerts(organization_id=organization_id, following_ids=following_ids, states=state,
                                     severities=severity))

@alert_app.command(name='grouped_list')
def grouped_alert_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                       scan_target_id: Optional[List[UUID]] = typer.Option(None, help="Only list alerts from the specified scan targets."),
                       state: Optional[List[AlertState]] = typer.Option(
                                   [x.value for x in AlertState if x != AlertState.CLOSED],
                                   help="Only list alerts in the specified states.", case_sensitive=False),
                       severity: Optional[List[AlertSeverity]] = typer.Option([x.value for x in AlertSeverity],
                                                                               help="Only list alerts with the specified severities",
                                                                               case_sensitive=False)):
    """
    List grouped alerts from a given organization, with optional filters by scan target, state or severity.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(
        client.iter_grouped_alerts(organization_id=organization_id, scan_target_ids=scan_target_id, states=state,
                                   severities=severity))

@alert_app.command(name='grouped_list_following')
def grouped_alert_following_list(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                       following_ids: Optional[List[UUID]] = typer.Option(None, help="Only list alerts from the specified scan targets."),
                       state: Optional[List[AlertState]] = typer.Option(
                                   [x.value for x in AlertState if x != AlertState.CLOSED],
                                   help="Only list alerts in the specified states.", case_sensitive=False),
                       severity: Optional[List[AlertSeverity]] = typer.Option([x.value for x in AlertSeverity],
                                                                               help="Only list alerts with the specified severities",
                                                                               case_sensitive=False)):
    """
    List grouped following alerts from a given organization, with optional filters by scan target, state or severity.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(
        client.iter_grouped_following_alerts(organization_id=organization_id, following_ids=following_ids, states=state,
                                             severities=severity))

@alert_app.command(name='get')
def alert_get(alert_id: UUID = typer.Argument(..., help="UUID of the alert to look up")):
    """
    Returns details about a specified alert
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_alert(alert_id))

@alert_app.command(name='list_history')
def alert_list_history(alert_id: UUID = typer.Argument(..., help="UUID of the alert to look up")):
    """
    Lists the alert history of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_alert_history(alert_id))

@alert_app.command(name='list_comment')
def alert_list_comment(alert_id: UUID = typer.Argument(..., help="UUID of the alert to look up")):
    """
    Lists the alert comments of organization this user has direct access to.
    """
    client = Client(profile=global_options['profile'])
    output_iterable(client.iter_alert_comments(alert_id))

###################################################
# Summary
###################################################

summary_app = typer.Typer(cls=OrderedCommands)
main_app.add_typer(summary_app, name="summary",
                   help="Operations on summaries the API key owner has direct access to")

@summary_app.command(name='alert')
def summary_alert(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                  scan_target_id: Optional[List[UUID]] = typer.Option(None, help="Only summarize alerts from the specified scan targets, defaults to all.")):
    """
    Gets a summary of the current state of alerts for an organization, both in total and broken down by scan target.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_alert_summaries(organization_id, scan_target_id))

@summary_app.command(name='alert_following')
def summary_alert_following(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                            following_ids: Optional[List[UUID]] = typer.Option(None, help="Only summarize alerts from the specified following, defaults to all.")):
    """
    Gets a summary of the current state of alerts for followed organizations.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_following_alert_summaries(organization_id, following_ids))

@summary_app.command(name='scan')
def summary_scan(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                 scan_target_ids: Optional[List[UUID]] = typer.Option(None, help="Only summarize alerts from the specified scan targets, defaults to all."),
                 days: Optional[int] = typer.Option(7, help="Number of days to go back in time in historical search")
                ):
    """
    Returns summaries of scan results over a period of time, summarizing number of alerts that changed states.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_scan_summaries(organization_id, scan_target_ids, days))

@summary_app.command(name='scan_following')
def summary_scan_following(organization_id: UUID = typer.Argument(..., help="UUID of the organization"),
                           following_ids: Optional[List[UUID]] = typer.Option(None, help="Only summarize alerts from the specified following, defaults to all."),
                           days: Optional[int] = typer.Option(7, help="Number of days to go back in time in historical search")
                          ):
    """
    Returns summaries of scan results over a period of time, summarizing number of alerts that changed states.
    """
    client = Client(profile=global_options['profile'])
    dump_json(client.get_scan_summaries(organization_id, following_ids, days))

if __name__ == "__main__":
    main_app()
