# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyband',
 'pyband.messages',
 'pyband.messages.cosmos',
 'pyband.messages.cosmos.bank',
 'pyband.messages.cosmos.bank.v1beta1',
 'pyband.messages.cosmos.distribution',
 'pyband.messages.cosmos.distribution.v1beta1',
 'pyband.messages.cosmos.gov',
 'pyband.messages.cosmos.gov.v1beta1',
 'pyband.messages.cosmos.staking',
 'pyband.messages.cosmos.staking.v1beta1',
 'pyband.messages.oracle',
 'pyband.messages.oracle.v1',
 'pyband.proto',
 'pyband.proto.cosmos',
 'pyband.proto.cosmos.auth',
 'pyband.proto.cosmos.auth.v1beta1',
 'pyband.proto.cosmos.authz',
 'pyband.proto.cosmos.authz.v1beta1',
 'pyband.proto.cosmos.bank',
 'pyband.proto.cosmos.bank.v1beta1',
 'pyband.proto.cosmos.base',
 'pyband.proto.cosmos.base.abci',
 'pyband.proto.cosmos.base.abci.v1beta1',
 'pyband.proto.cosmos.base.kv',
 'pyband.proto.cosmos.base.kv.v1beta1',
 'pyband.proto.cosmos.base.query',
 'pyband.proto.cosmos.base.query.v1beta1',
 'pyband.proto.cosmos.base.reflection',
 'pyband.proto.cosmos.base.reflection.v1beta1',
 'pyband.proto.cosmos.base.reflection.v2alpha1',
 'pyband.proto.cosmos.base.snapshots',
 'pyband.proto.cosmos.base.snapshots.v1beta1',
 'pyband.proto.cosmos.base.store',
 'pyband.proto.cosmos.base.store.v1beta1',
 'pyband.proto.cosmos.base.tendermint',
 'pyband.proto.cosmos.base.tendermint.v1beta1',
 'pyband.proto.cosmos.base.v1beta1',
 'pyband.proto.cosmos.capability',
 'pyband.proto.cosmos.capability.v1beta1',
 'pyband.proto.cosmos.crisis',
 'pyband.proto.cosmos.crisis.v1beta1',
 'pyband.proto.cosmos.crypto',
 'pyband.proto.cosmos.crypto.ed25519',
 'pyband.proto.cosmos.crypto.multisig',
 'pyband.proto.cosmos.crypto.multisig.v1beta1',
 'pyband.proto.cosmos.crypto.secp256k1',
 'pyband.proto.cosmos.crypto.secp256r1',
 'pyband.proto.cosmos.distribution',
 'pyband.proto.cosmos.distribution.v1beta1',
 'pyband.proto.cosmos.evidence',
 'pyband.proto.cosmos.evidence.v1beta1',
 'pyband.proto.cosmos.feegrant',
 'pyband.proto.cosmos.feegrant.v1beta1',
 'pyband.proto.cosmos.genutil',
 'pyband.proto.cosmos.genutil.v1beta1',
 'pyband.proto.cosmos.gov',
 'pyband.proto.cosmos.gov.v1beta1',
 'pyband.proto.cosmos.mint',
 'pyband.proto.cosmos.mint.v1beta1',
 'pyband.proto.cosmos.params',
 'pyband.proto.cosmos.params.v1beta1',
 'pyband.proto.cosmos.slashing',
 'pyband.proto.cosmos.slashing.v1beta1',
 'pyband.proto.cosmos.staking',
 'pyband.proto.cosmos.staking.v1beta1',
 'pyband.proto.cosmos.tx',
 'pyband.proto.cosmos.tx.signing',
 'pyband.proto.cosmos.tx.signing.v1beta1',
 'pyband.proto.cosmos.tx.v1beta1',
 'pyband.proto.cosmos.upgrade',
 'pyband.proto.cosmos.upgrade.v1beta1',
 'pyband.proto.cosmos.vesting',
 'pyband.proto.cosmos.vesting.v1beta1',
 'pyband.proto.cosmos_proto',
 'pyband.proto.gogoproto',
 'pyband.proto.google',
 'pyband.proto.google.api',
 'pyband.proto.ics23',
 'pyband.proto.oracle',
 'pyband.proto.oracle.v1',
 'pyband.proto.tendermint',
 'pyband.proto.tendermint.abci',
 'pyband.proto.tendermint.crypto',
 'pyband.proto.tendermint.libs',
 'pyband.proto.tendermint.libs.bits',
 'pyband.proto.tendermint.p2p',
 'pyband.proto.tendermint.types',
 'pyband.proto.tendermint.version',
 'pyband.wallet']

package_data = \
{'': ['*']}

install_requires = \
['bech32==1.2.0',
 'betterproto==2.0.0b5',
 'bip32==0.0.8',
 'ecdsa==0.15',
 'grpclib==0.4.3',
 'ledgerblue==0.1.42',
 'mnemonic==0.19',
 'python-dateutil==2.8.2']

setup_kwargs = {
    'name': 'pyband',
    'version': '0.3.2',
    'description': 'Python library for BandChain',
    'long_description': '<div align="center">\n    <h2>PyBand</h2>\n    <blockquote>BandChain Python Library</blockquote>\n</div>\n\nPyband is a library that is used to interact with BandChain through the `gRPC` protocol. Querying data and sending\ntransaction can be done here!\n\n## ⭐️ Features\n\nThis helper library allows users to interact with BandChain.\n\nPyBand supports the following features:\n\n- Getting the information of a specific oracle script, data source, and request ID.\n- Getting the account information of specific address.\n- Getting the latest request for a specific oracle script with its matching calldata and validator ask_count and\n  min_count.\n- Querying all the reporters associated with a specific validator.\n- Seeing what client_id you are using and getting BandChain\'s latest block data.\n- Able to send transaction in 3 modes: block mode, async mode, and sync mode.\n\n## 📦 Installation\n\nThis library is available on [PyPI](https://pypi.org/project/pyband/)\n\n```bash\npip install pyband\n```\n\n## 💎 Example Usage\n\nThe example below shows how this library can be used to get the result of the latest request for the price of any\ncryptocurrency. In this example, we will get the latest price of BTC on BandChain\'s testnet.\n\nThe specified parameters are:\n\n- `oracleScriptID`: 111\n- `calldata`: The hex string representing the [OBI](<https://github.com/bandprotocol/bandchain/wiki/Oracle-Binary-Encoding-(OBI)>)-encoded value of `{\'symbols\': [\'BTC\'], \'multiplier\': 100000000}`\n- `minCount`: 10\n- `askCount`: 16\n\n```python\nimport asyncio\n\nfrom pyband import Client, PyObi\n\n\nasync def main():\n    grpc_url = "laozi-testnet6.bandchain.org"\n    c = Client.from_endpoint(grpc_url, 443)\n\n    oid = 111\n    calldata = "00000001000000034254430000000005f5e100"\n    min_count = 10\n    ask_count = 16\n\n    req_info = await c.get_latest_request(oid, calldata, min_count, ask_count)\n    oracle_script = await c.get_oracle_script(oid)\n    obi = PyObi(oracle_script.schema)\n\n    # Converts the calldata into a readable syntax\n    print(obi.decode_input(bytes.fromhex(calldata)))\n\n    # Prints the result\n    print(obi.decode_output(req_info.request.result.result))\n\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\nBelow is the results of the example above.\n\n```\n{\'symbols\': [\'BTC\'], \'multiplier\': 100000000}\n{\'rates\': [1936488410000]}\n```\n\nThis example shows how to send a transaction on BandChain using block mode.\n\n```python\nimport asyncio\nimport os\n\nfrom pyband import Client, Transaction, Wallet\nfrom pyband.messages.cosmos.bank.v1beta1 import MsgSend\nfrom pyband.proto.cosmos.base.v1beta1 import Coin\n\n\nasync def main():\n    # Create a GRPC connection\n    grpc_url = "laozi-testnet6.bandchain.org"\n    c = Client.from_endpoint(grpc_url, 443)\n\n    # Convert a mnemonic to a wallet\n    wallet = Wallet.from_mnemonic(os.getenv("MNEMONIC"))\n    sender = wallet.get_address().to_acc_bech32()\n\n    # Prepare a transaction\'s properties\n    msg_send = MsgSend(\n        from_address=sender,\n        to_address="band19ajhdg6maw0ja0a7qd9sq7nm4ym9f4wjg8r96w",\n        amount=[Coin(amount="1000000", denom="uband")],\n    )\n\n    account = await c.get_account(sender)\n    account_num = account.account_number\n    sequence = account.sequence\n\n    fee = [Coin(amount="50000", denom="uband")]\n    chain_id = await c.get_chain_id()\n\n    # Step 4 Construct a transaction\n    txn = (\n        Transaction()\n        .with_messages(msg_send)\n        .with_sequence(sequence)\n        .with_account_num(account_num)\n        .with_chain_id(chain_id)\n        .with_gas(2000000)\n        .with_fee(fee)\n        .with_memo("")\n    )\n\n    # Sign and broadcast a transaction\n    tx_block = await c.send_tx_block_mode(wallet.sign_and_build(txn))\n\n    # Converting to JSON for readability\n    print(tx_block.to_json(indent=4))\n\n\nif __name__ == "__main__":\n    asyncio.run(main())\n```\n\n## 🧀 Notes\n\nFor more examples, please go to [`examples`](/examples/request_data_example.py).\n',
    'author': 'Band Protocol',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://bandprotocol.com/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
