"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
require("@aws-cdk/assert/jest");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const mediastore = require("aws-cdk-lib/aws-mediastore");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const s3 = require("aws-cdk-lib/aws-s3");
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
// --------------------------------------------------------------
// Test the default deployment pattern variables
// --------------------------------------------------------------
test('Test the default deployment pattern variables', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {});
    // Assertion
    expect(cloudFrontToMediaStore.cloudFrontWebDistribution).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.mediaStoreContainer).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontLoggingBucket).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginRequestPolicy).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).not.toEqual(undefined);
    expect(cloudFrontToMediaStore.cloudFrontFunction).not.toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment without HTTP security headers
// --------------------------------------------------------------
test('Test the deployment without HTTP security headers', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        insertHttpSecurityHeaders: false
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontFunction).toEqual(undefined);
});
test('Test the deployment with securityHeadersBehavior instead of HTTP security headers', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        insertHttpSecurityHeaders: false,
        responseHeadersPolicyProps: {
            securityHeadersBehavior: {
                strictTransportSecurity: {
                    accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(63072),
                    includeSubdomains: true,
                    override: true,
                    preload: true
                },
                contentSecurityPolicy: {
                    contentSecurityPolicy: "upgrade-insecure-requests; default-src 'none';",
                    override: true
                },
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike("AWS::CloudFront::ResponseHeadersPolicy", {
        ResponseHeadersPolicyConfig: {
            SecurityHeadersConfig: {
                ContentSecurityPolicy: {
                    ContentSecurityPolicy: "upgrade-insecure-requests; default-src 'none';",
                    Override: true
                },
                StrictTransportSecurity: {
                    AccessControlMaxAgeSec: 63072,
                    IncludeSubdomains: true,
                    Override: true,
                    Preload: true
                }
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontFunction).toEqual(undefined);
});
test("throw exception if insertHttpSecurityHeaders and responseHeadersPolicyProps are provided", () => {
    const stack = new cdk.Stack();
    expect(() => {
        new lib_1.CloudFrontToMediaStore(stack, "test-cloudfront-mediastore", {
            insertHttpSecurityHeaders: true,
            responseHeadersPolicyProps: {
                securityHeadersBehavior: {
                    strictTransportSecurity: {
                        accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(63072),
                        includeSubdomains: true,
                        override: false,
                        preload: true
                    }
                }
            }
        });
    }).toThrowError();
});
// --------------------------------------------------------------
// Test the deployment with existing MediaStore container
// --------------------------------------------------------------
test('Test the deployment with existing MediaStore container', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, 'MyMediaStoreContainer', {
        containerName: 'MyMediaStoreContainer'
    });
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        existingMediaStoreContainerObj: mediaStoreContainer
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'MyMediaStoreContainer',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer'
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided MediaStore properties
// --------------------------------------------------------------
test('Test the deployment with the user provided MediaStore properties', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    const cloudFrontToMediaStore = new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        mediaStoreContainerProps: {
            containerName: 'MyMediaStoreContainer',
            policy: '{}',
            lifecyclePolicy: '{}',
            corsPolicy: [],
            metricPolicy: {
                containerLevelMetrics: 'DISABLED'
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'redirect-to-https'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D'
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        ContainerName: 'MyMediaStoreContainer',
        Policy: '{}',
        LifecyclePolicy: '{}',
        CorsPolicy: [],
        MetricPolicy: {
            ContainerLevelMetrics: 'DISABLED'
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(cloudFrontToMediaStore.cloudFrontOriginAccessIdentity).toEqual(undefined);
});
// --------------------------------------------------------------
// Test the deployment with the user provided CloudFront properties
// --------------------------------------------------------------
test('Test the deployment with the user provided CloudFront properties', () => {
    // Initial setup
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'test-cloudfront-mediastore', {
        cloudFrontDistributionProps: {
            defaultBehavior: {
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.HTTPS_ONLY,
                allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
                cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD
            }
        }
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE'
                ],
                CachePolicyId: '658327ea-f89d-4fab-a63d-7e88639e58f6',
                CachedMethods: [
                    'GET',
                    'HEAD'
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "testcloudfrontmediastoreSetHttpSecurityHeaders9995A63D",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                OriginRequestPolicyId: {
                    Ref: 'testcloudfrontmediastoreCloudfrontOriginRequestPolicyA1D988D3'
                },
                TargetOriginId: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                ViewerProtocolPolicy: 'https-only'
            },
            Enabled: true,
            HttpVersion: 'http2',
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    'Fn::GetAtt': [
                        'testcloudfrontmediastoreCloudfrontLoggingBucketA3A51E6A',
                        'RegionalDomainName'
                    ]
                }
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only'
                    },
                    DomainName: {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '/',
                                    {
                                        'Fn::Select': [
                                            1,
                                            {
                                                'Fn::Split': [
                                                    '://',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'testcloudfrontmediastoreMediaStoreContainerF60A96BB',
                                                            'Endpoint'
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: 'testcloudfrontmediastoreCloudFrontDistributionOrigin1BBFA2A4D',
                    OriginCustomHeaders: [
                        {
                            HeaderName: 'User-Agent',
                            HeaderValue: {
                                Ref: 'testcloudfrontmediastoreCloudFrontOriginAccessIdentity966405A0'
                            }
                        }
                    ]
                }
            ]
        }
    });
    expect(stack).toHaveResourceLike('AWS::MediaStore::Container', {
        AccessLoggingEnabled: true,
        ContainerName: {
            Ref: 'AWS::StackName'
        },
        CorsPolicy: [
            {
                AllowedHeaders: ['*'],
                AllowedMethods: ['GET'],
                AllowedOrigins: ['*'],
                ExposeHeaders: ["*"],
                MaxAgeSeconds: 3000
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::OriginRequestPolicy', {
        OriginRequestPolicyConfig: {
            Comment: 'Policy for Constructs CloudFrontDistributionForMediaStore',
            CookiesConfig: {
                CookieBehavior: 'none'
            },
            HeadersConfig: {
                HeaderBehavior: 'whitelist',
                Headers: [
                    'Access-Control-Allow-Origin',
                    'Access-Control-Request-Method',
                    'Access-Control-Request-Header',
                    'Origin'
                ]
            },
            Name: {
                'Fn::Join': [
                    '',
                    [
                        {
                            Ref: 'AWS::StackName'
                        },
                        '-',
                        {
                            Ref: 'AWS::Region'
                        },
                        '-CloudFrontDistributionForMediaStore'
                    ]
                ]
            },
            QueryStringsConfig: {
                QueryStringBehavior: 'all'
            }
        }
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::CloudFrontOriginAccessIdentity', {
        CloudFrontOriginAccessIdentityConfig: {
            Comment: {
                'Fn::Join': [
                    '',
                    [
                        'access-identity-',
                        {
                            Ref: 'AWS::Region',
                        },
                        '-',
                        {
                            Ref: 'AWS::StackName',
                        }
                    ]
                ]
            }
        }
    });
});
// --------------------------------------------------------------
// Cloudfront logging bucket with destroy removal policy and auto delete objects
// --------------------------------------------------------------
test('Cloudfront logging bucket with destroy removal policy and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.CloudFrontToMediaStore(stack, 'cloudfront-mediatstore', {
        cloudFrontLoggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "cloudfrontmediatstoreCloudfrontLoggingBucket2565C68A"
        }
    });
});
// --------------------------------------------------------------
// Cloudfront logging bucket error providing existing log bucket and logBucketProps
// --------------------------------------------------------------
test('Cloudfront logging bucket error when providing existing log bucket and logBucketProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const logBucket = new s3.Bucket(stack, 'cloudfront-log-bucket', {});
    const app = () => {
        new lib_1.CloudFrontToMediaStore(stack, 'cloudfront-s3', {
            cloudFrontDistributionProps: {
                logBucket
            },
            cloudFrontLoggingBucketProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: true
            }
        });
    };
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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