# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fakeredis', 'fakeredis.commands_mixins']

package_data = \
{'': ['*']}

install_requires = \
['redis<4.5', 'sortedcontainers>=2.4.0,<3.0.0']

extras_require = \
{'aioredis': ['aioredis>=2.0.1,<3.0.0'], 'lua': ['lupa>=1.13,<2.0']}

setup_kwargs = {
    'name': 'fakeredis',
    'version': '2.0.0',
    'description': 'Fake implementation of redis API for testing purposes.',
    'long_description': 'fakeredis: A fake version of a redis-py\n=======================================\n[![badge](https://img.shields.io/pypi/v/fakeredis)](https://pypi.org/project/fakeredis/)\n[![CI](https://github.com/cunla/fakeredis-py/actions/workflows/test.yml/badge.svg)](https://github.com/cunla/fakeredis-py/actions/workflows/test.yml)\n[![badge](https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/cunla/b756396efb895f0e34558c980f1ca0c7/raw/fakeredis-py.json)](https://github.com/cunla/fakeredis-py/actions/workflows/test.yml)\n[![badge](https://img.shields.io/pypi/dm/fakeredis)](https://pypi.org/project/fakeredis/)\n[![badge](https://img.shields.io/pypi/l/fakeredis)](./LICENSE)\n--------------------\n[Intro](#intro) | [How to Use](#how-to-use) | [Contributing](.github/CONTRIBUTING.md) | [Guides](#guides)\n| [Sponsoring](#sponsor)\n\n# Intro\n\nfakeredis is a pure-Python implementation of the redis-py python client\nthat simulates talking to a redis server. This was created for a single\npurpose: **to write tests**. Setting up redis is not hard, but\nmany times you want to write tests that do not talk to an external server\n(such as redis). This module now allows tests to simply use this\nmodule as a reasonable substitute for redis.\n\nFor a list of supported/unsupported redis commands, see [REDIS_COMMANDS.md](./REDIS_COMMANDS.md).\n\n# Installation\n\nTo install fakeredis-py, simply:\n\n``` bash\n$ pip install fakeredis\n```\n\nYou will need [lupa](https://pypi.org/project/lupa/) if you want to run Lua scripts\n(this includes features like ``redis.lock.Lock``, which are implemented in Lua).\nIf you install fakeredis with ``pip install fakeredis[lua]`` it will be automatically installed.\n\n# How to Use\n\nFakeRedis can imitate Redis server version 6.x or 7.x.\nIf you do not specify the version, version 7 is used by default.\n\nThe intent is for fakeredis to act as though you\'re talking to a real\nredis server. It does this by storing state internally.\nFor example:\n\n```\n>>> import fakeredis\n>>> r = fakeredis.FakeStrictRedis(version=6)\n>>> r.set(\'foo\', \'bar\')\nTrue\n>>> r.get(\'foo\')\n\'bar\'\n>>> r.lpush(\'bar\', 1)\n1\n>>> r.lpush(\'bar\', 2)\n2\n>>> r.lrange(\'bar\', 0, -1)\n[2, 1]\n```\n\nThe state is stored in an instance of `FakeServer`. If one is not provided at\nconstruction, a new instance is automatically created for you, but you can\nexplicitly create one to share state:\n\n```\n>>> import fakeredis\n>>> server = fakeredis.FakeServer()\n>>> r1 = fakeredis.FakeStrictRedis(server=server)\n>>> r1.set(\'foo\', \'bar\')\nTrue\n>>> r2 = fakeredis.FakeStrictRedis(server=server)\n>>> r2.get(\'foo\')\n\'bar\'\n>>> r2.set(\'bar\', \'baz\')\nTrue\n>>> r1.get(\'bar\')\n\'baz\'\n>>> r2.get(\'bar\')\n\'baz\'\n```\n\nIt is also possible to mock connection errors, so you can effectively test\nyour error handling. Simply set the connected attribute of the server to\n`False` after initialization.\n\n```\n>>> import fakeredis\n>>> server = fakeredis.FakeServer()\n>>> server.connected = False\n>>> r = fakeredis.FakeStrictRedis(server=server)\n>>> r.set(\'foo\', \'bar\')\nConnectionError: FakeRedis is emulating a connection error.\n>>> server.connected = True\n>>> r.set(\'foo\', \'bar\')\nTrue\n```\n\nFakeredis implements the same interface as `redis-py`, the popular\nredis client for python, and models the responses of redis 6.x or 7.x.\n\n## Use to test django-rq\n\nThere is a need to override `django_rq.queues.get_redis_connection` with\na method returning the same connection.\n\n```python\nfrom fakeredis import FakeRedisConnSingleton\n\ndjango_rq.queues.get_redis_connection = FakeRedisConnSingleton()\n```\n\n### Limitations\n\nApart from unimplemented commands, there are a number of cases where fakeredis\nwon\'t give identical results to real redis. The following are differences that\nare unlikely to ever be fixed; there are also differences that are fixable\n(such as commands that do not support all features) which should be filed as\nbugs in GitHub.\n\n1. Hyperloglogs are implemented using sets underneath. This means that the\n   `type` command will return the wrong answer, you can\'t use `get` to retrieve\n   the encoded value, and counts will be slightly different (they will in fact be\n   exact).\n\n2. When a command has multiple error conditions, such as operating on a key of\n   the wrong type and an integer argument is not well-formed, the choice of\n   error to return may not match redis.\n\n3. The `incrbyfloat` and `hincrbyfloat` commands in redis use the C `long\n   double` type, which typically has more precision than Python\'s `float`\n   type.\n\n4. Redis makes guarantees about the order in which clients blocked on blocking\n   commands are woken up. Fakeredis does not honour these guarantees.\n\n5. Where redis contains bugs, fakeredis generally does not try to provide exact\n   bug-compatibility. It\'s not practical for fakeredis to try to match the set\n   of bugs in your specific version of redis.\n\n6. There are a number of cases where the behaviour of redis is undefined, such\n   as the order of elements returned by set and hash commands. Fakeredis will\n   generally not produce the same results, and in Python versions before 3.6\n   may produce different results each time the process is re-run.\n\n7. SCAN/ZSCAN/HSCAN/SSCAN will not necessarily iterate all items if items are\n   deleted or renamed during iteration. They also won\'t necessarily iterate in\n   the same chunk sizes or the same order as redis.\n\n8. DUMP/RESTORE will not return or expect data in the RDB format. Instead, the\n   `pickle` module is used to mimic an opaque and non-standard format.\n   **WARNING**: Do not use RESTORE with untrusted data, as a malicious pickle\n   can execute arbitrary code.\n\n### Local development environment\n\nTo ensure parity with the real redis, there are a set of integration tests\nthat mirror the unittests. For every unittest that is written, the same\ntest is run against a real redis instance using a real redis-py client\ninstance. In order to run these tests you must have a redis server running\non localhost, port 6379 (the default settings). **WARNING**: the tests will\ncompletely wipe your database!\n\nFirst install poetry if you don\'t have it, and then install all the dependencies:\n\n```   \npip install poetry\npoetry install\n``` \n\nTo run all the tests:\n\n```\npoetry run pytest -v\n```\n\nIf you only want to run tests against fake redis, without a real redis::\n\n```\npoetry run pytest -m fake\n```\n\nBecause this module is attempting to provide the same interface as `redis-py`,\nthe python bindings to redis, a reasonable way to test this to take each\nunittest and run it against a real redis server. fakeredis and the real redis\nserver should give the same result. To run tests against a real redis instance\ninstead:\n\n```\npoetry run pytest -m real\n```\n\nIf redis is not running, and you try to run tests against a real redis server,\nthese tests will have a result of \'s\' for skipped.\n\nThere are some tests that test redis blocking operations that are somewhat\nslow. If you want to skip these tests during day to day development,\nthey have all been tagged as \'slow\' so you can skip them by running:\n\n```\npoetry run pytest -m "not slow"\n```\n\n# Contributing\n\nContributions are welcome. Please see the [contributing guide](.github/CONTRIBUTING.md) for more details.\nIf you\'d like to help out, you can start with any of the issues labeled with `Help wanted`.\n\n# Guides\n\n### Implementing support for a command\n\nCreating a new command support should be done in the `FakeSocket` class (in `_fakesocket.py`) by creating the method\nand using `@command` decorator (which should be the command syntax, you can use existing samples on the file).\n\nFor example:\n\n```python\nclass FakeSocket(BaseFakeSocket, FakeLuaSocket):\n    # ...\n    @command(...)\n    def zscore(self, key, member):\n        try:\n            return self._encodefloat(key.value[member], False)\n        except KeyError:\n            return None\n```\n\n#### Implement a test for it\n\nThere are multiple scenarios for test, with different versions of redis server, redis-py, etc.\nThe tests not only assert the validity of output but runs the same test on a real redis-server and compares the output\nto the real server output.\n\n- Create tests in the `test_fakeredis6.py` if the command is supported in redis server 6.x.\n- Alternatively, create the test in `test_fakeredis7.py` if the command is supported only on redis server 7.x.\n- If support for the command was introduced in a certain version of redis-py (\n  see [redis-py release notes](https://github.com/redis/redis-py/releases/tag/v4.3.4)) you can use the\n  decorator `@testtools.run_test_if_redispy_ver` on your tests. example:\n\n```python\n@testtools.run_test_if_redispy_ver(\'above\', \'4.2.0\')  # This will run for redis-py 4.2.0 or above.\ndef test_expire_should_not_expire__when_no_expire_is_set(r):\n    r.set(\'foo\', \'bar\')\n    assert r.get(\'foo\') == b\'bar\'\n    assert r.expire(\'foo\', 1, xx=True) == 0\n```\n\n#### Updating `REDIS_COMMANDS.md`\n\nLastly, run from the root of the project the script to regenerate `REDIS_COMMANDS.md`:\n\n```\npython scripts/supported.py > REDIS_COMMANDS.md    \n```\n\n# Sponsor\n\nfakeredis-py is developed for free.\n\nYou can support this project by becoming a sponsor using [this link](https://github.com/sponsors/cunla).\n\nAlternatively, you can buy me coffee using this\nlink: [!["Buy Me A Coffee"](https://www.buymeacoffee.com/assets/img/custom_images/orange_img.png)](https://buymeacoffee.com/danielmoran)\n',
    'author': 'James Saryerwinnie',
    'author_email': 'js@jamesls.com',
    'maintainer': 'Daniel Moran',
    'maintainer_email': 'daniel.maruani@gmail.com',
    'url': 'https://github.com/cunla/fakeredis-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
