"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Key = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
const perms = require("./private/perms");
class KeyBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Collection of aliases added to the key
         *
         * Tracked to determine whether or not the aliasName should be added to the end of its ID
         */
        this.aliases = [];
    }
    /**
     * Defines a new alias for the key.
     *
     * @stability stable
     */
    addAlias(aliasName) {
        const aliasId = this.aliases.length > 0 ? `Alias${aliasName}` : 'Alias';
        const alias = new alias_1.Alias(this, aliasId, { aliasName, targetKey: this });
        this.aliases.push(alias);
        return alias;
    }
    /**
     * Adds a statement to the KMS key resource policy.
     *
     * @param statement The policy statement to add.
     * @param allowNoOp If this is set to `false` and there is no policy defined (i.e. external key), the operation will fail. Otherwise, it will no-op.
     * @stability stable
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = core_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return { statementAdded: false };
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Grant the indicated permissions on this key to the given principal.
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        // KMS verifies whether the principals included in its key policy actually exist.
        // This is a problem if the stack the grantee is part of depends on the key stack
        // (as it won't exist before the key policy is attempted to be created).
        // In that case, make the account the resource policy principal
        const granteeStackDependsOnKeyStack = this.granteeStackDependsOnKeyStack(grantee);
        const principal = granteeStackDependsOnKeyStack
            ? new iam.AccountPrincipal(granteeStackDependsOnKeyStack)
            : grantee.grantPrincipal;
        const crossAccountAccess = this.isGranteeFromAnotherAccount(grantee);
        const crossRegionAccess = this.isGranteeFromAnotherRegion(grantee);
        const crossEnvironment = crossAccountAccess || crossRegionAccess;
        const grantOptions = {
            grantee,
            actions,
            resource: this,
            resourceArns: [this.keyArn],
            resourceSelfArns: crossEnvironment ? undefined : ['*'],
        };
        if (this.trustAccountIdentities) {
            return iam.Grant.addToPrincipalOrResource(grantOptions);
        }
        else {
            return iam.Grant.addToPrincipalAndResource({
                ...grantOptions,
                // if the key is used in a cross-environment matter,
                // we can't access the Key ARN (they don't have physical names),
                // so fall back to using '*'. ToDo we need to make this better... somehow
                resourceArns: crossEnvironment ? ['*'] : [this.keyArn],
                resourcePolicyPrincipal: principal,
            });
        }
    }
    /**
     * Grant decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, ...perms.DECRYPT_ACTIONS);
    }
    /**
     * Grant encryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, ...perms.ENCRYPT_ACTIONS);
    }
    /**
     * Grant encryption and decryption permissions using this key to the given principal.
     *
     * @stability stable
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, ...[...perms.DECRYPT_ACTIONS, ...perms.ENCRYPT_ACTIONS]);
    }
    /**
     * Checks whether the grantee belongs to a stack that will be deployed
     * after the stack containing this key.
     *
     * @param grantee the grantee to give permissions to
     * @returns the account ID of the grantee stack if its stack does depend on this stack,
     *   undefined otherwise
     */
    granteeStackDependsOnKeyStack(grantee) {
        const grantPrincipal = grantee.grantPrincipal;
        if (!(grantPrincipal instanceof constructs_1.Construct)) {
            return undefined;
        }
        // this logic should only apply to newly created
        // (= not imported) resources
        if (!this.principalIsANewlyCreatedResource(grantPrincipal)) {
            return undefined;
        }
        // return undefined;
        const keyStack = core_1.Stack.of(this);
        const granteeStack = core_1.Stack.of(grantPrincipal);
        if (keyStack === granteeStack) {
            return undefined;
        }
        return granteeStack.dependencies.includes(keyStack)
            ? granteeStack.account
            : undefined;
    }
    principalIsANewlyCreatedResource(principal) {
        // yes, this sucks
        // this is just a temporary stopgap to stem the bleeding while we work on a proper fix
        return principal instanceof iam.Role ||
            principal instanceof iam.User ||
            principal instanceof iam.Group;
    }
    isGranteeFromAnotherRegion(grantee) {
        if (!(grantee instanceof constructs_1.Construct)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.region !== identityStack.region;
    }
    isGranteeFromAnotherAccount(grantee) {
        if (!(grantee instanceof constructs_1.Construct)) {
            return false;
        }
        const bucketStack = core_1.Stack.of(this);
        const identityStack = core_1.Stack.of(grantee);
        return bucketStack.account !== identityStack.account;
    }
}
/**
 * Defines a KMS key.
 *
 * @stability stable
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        super(scope, id);
        const defaultKeyPoliciesFeatureEnabled = core_1.FeatureFlags.of(this).isEnabled(cxapi.KMS_DEFAULT_KEY_POLICIES);
        this.policy = (_a = props.policy) !== null && _a !== void 0 ? _a : new iam.PolicyDocument();
        if (defaultKeyPoliciesFeatureEnabled) {
            if (props.trustAccountIdentities === false) {
                throw new Error('`trustAccountIdentities` cannot be false if the @aws-cdk/aws-kms:defaultKeyPolicies feature flag is set');
            }
            this.trustAccountIdentities = true;
            // Set the default key policy if one hasn't been provided by the user.
            if (!props.policy) {
                this.addDefaultAdminPolicy();
            }
        }
        else {
            this.trustAccountIdentities = (_b = props.trustAccountIdentities) !== null && _b !== void 0 ? _b : false;
            if (this.trustAccountIdentities) {
                this.addDefaultAdminPolicy();
            }
            else {
                this.addLegacyAdminPolicy();
            }
        }
        let pendingWindowInDays;
        if (props.pendingWindow) {
            pendingWindowInDays = props.pendingWindow.toDays();
            if (pendingWindowInDays < 7 || pendingWindowInDays > 30) {
                throw new Error(`'pendingWindow' value must between 7 and 30 days. Received: ${pendingWindowInDays}`);
            }
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
            pendingWindowInDays: pendingWindowInDays,
        });
        this.keyArn = resource.attrArn;
        this.keyId = resource.ref;
        resource.applyRemovalPolicy(props.removalPolicy);
        ((_c = props.admins) !== null && _c !== void 0 ? _c : []).forEach((p) => this.grantAdmin(p));
        if (props.alias !== undefined) {
            this.addAlias(props.alias);
        }
    }
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope the construct that will "own" the imported key.
     * @param id the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     * @stability stable
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor(keyId) {
                super(scope, id);
                this.keyArn = keyArn;
                this.policy = undefined;
                // defaulting true: if we are importing the key the key policy is
                // undefined and impossible to change here; this means updating identity
                // policies is really the only option
                this.trustAccountIdentities = true;
                this.keyId = keyId;
            }
        }
        const keyResourceName = core_1.Stack.of(scope).parseArn(keyArn).resourceName;
        if (!keyResourceName) {
            throw new Error(`KMS key ARN must be in the format 'arn:aws:kms:<region>:<account>:key/<keyId>', got: '${keyArn}'`);
        }
        return new Import(keyResourceName);
    }
    /**
     * Grant admins permissions using this key to the given principal.
     *
     * Key administrators have permissions to manage the key (e.g., change permissions, revoke), but do not have permissions
     * to use the key in cryptographic operations (e.g., encrypt, decrypt).
     *
     * @stability stable
     */
    grantAdmin(grantee) {
        return this.grant(grantee, ...perms.ADMIN_ACTIONS);
    }
    /**
     * Adds the default key policy to the key. This policy gives the AWS account (root user) full access to the CMK,
     * which reduces the risk of the CMK becoming unmanageable and enables IAM policies to allow access to the CMK.
     * This is the same policy that is default when creating a Key via the KMS API or Console.
     * @see https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     */
    addDefaultAdminPolicy() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['kms:*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Grants the account admin privileges -- not full account access -- plus the GenerateDataKey action.
     * The GenerateDataKey action was added for interop with S3 in https://github.com/aws/aws-cdk/issues/3458.
     *
     * This policy is discouraged and deprecated by the '@aws-cdk/aws-kms:defaultKeyPolicies' feature flag.
     *
     * @link https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default
     * @deprecated
     */
    addLegacyAdminPolicy() {
        // This is equivalent to `[...perms.ADMIN_ACTIONS, 'kms:GenerateDataKey']`,
        // but keeping this explicit ordering for backwards-compatibility (changing the ordering causes resource updates)
        const actions = [
            'kms:Create*',
            'kms:Describe*',
            'kms:Enable*',
            'kms:List*',
            'kms:Put*',
            'kms:Update*',
            'kms:Revoke*',
            'kms:Disable*',
            'kms:Get*',
            'kms:Delete*',
            'kms:ScheduleKeyDeletion',
            'kms:CancelKeyDeletion',
            'kms:GenerateDataKey',
            'kms:TagResource',
            'kms:UntagResource',
        ];
        this.addToResourcePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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