# Copyright (c) 2024, qBraid Development Team
# All rights reserved.

"""
Unit tests for the create_qbraid_env_assets function in the envs app.

"""

import json
import os
import shutil
from unittest import mock  # pylint: disable=unused-variable
from unittest.mock import mock_open, patch

from qbraid_core.services.environments.create import create_qbraid_env_assets


def test_create_qbraid_env_assets(tmp_path):
    """Test creating the assets for a qbraid environment."""
    slug = "test_slug"
    alias = "test_alias"
    kernel_name = "test_kernel"
    slug_path = str(tmp_path)

    kernel_spec_mock = {
        "python3": {
            "spec": {
                "argv": ["python", "-m", "ipykernel_launcher", "-f", "{connection_file}"],
                "display_name": "Python 3",
                "language": "python",
            },
            "resource_dir": "/path/to/default/resources",
        }
    }

    with (
        patch("os.makedirs"),
        patch("os.path.isfile", return_value=True),
        patch("shutil.copy"),
        patch("json.dump"),
        patch("builtins.open", mock_open()),
        patch(
            "qbraid_core.services.environments.create.update_state_json"
        ) as mock_update_state_json,
        patch(
            "jupyter_client.kernelspec.KernelSpecManager.get_all_specs",
            return_value=kernel_spec_mock,
        ),
    ):
        create_qbraid_env_assets(slug, alias, kernel_name, slug_path)

        # Verify that update_state_json and create_venv are called with correct arguments
        mock_update_state_json.assert_called_once_with(slug_path, 0, 0, env_name=alias)

        # Verify kernel.json creation and contents
        expected_kernel_json_path = os.path.join(
            slug_path, "kernels", f"python3_{slug}", "kernel.json"
        )
        open.assert_any_call(expected_kernel_json_path, "w", encoding="utf-8")
        # Prepare the expected data for kernel.json,
        # modifying argv[0] to match the expected python_exec_path
        expected_kernel_data = kernel_spec_mock["python3"]["spec"]
        if os.name == "nt":
            expected_kernel_data["argv"][0] = os.path.join(
                slug_path, "pyenv", "Scripts", "python.exe"
            )
        else:
            expected_kernel_data["argv"][0] = os.path.join(slug_path, "pyenv", "bin", "python")
        expected_kernel_data["display_name"] = kernel_name

        # Assert json.dump was called with the expected kernel data
        json.dump.assert_called_with(  # pylint: disable=no-member
            expected_kernel_data, mock.ANY, indent=4
        )  # mock.ANY is used because we don't have the file object directly

        # Verify copying of logo files
        sys_resource_dir = kernel_spec_mock["python3"]["resource_dir"]
        for file_name in ["logo-32x32.png", "logo-64x64.png", "logo-svg.svg"]:
            sys_path = os.path.join(sys_resource_dir, file_name)
            loc_path = os.path.join(slug_path, "kernels", f"python3_{slug}", file_name)
            shutil.copy.assert_any_call(sys_path, loc_path)  # pylint: disable=no-member
