# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['datachunks']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'datachunks',
    'version': '0.0.2',
    'description': 'Data chunking for humans, particularly for data engineers',
    'long_description': '# datachunks\n\nData chunking for humans, particularly for data engineers. Makes batched data\nprocessing less painful and a little bit more joyful.\n\n**ATTENTION!** This is a pre-alpha version!\n\n## Install\ndatachunks requires Python 3.8 or newer. Install it from PyPI:\n```shell\n$ pip install datachunks\n```\n\n## Usage\ndatachunks implements two chunking strategy:\n1. "Pull" strategy. Wrap your source stream with <code>chunks</code> generator and consume a chunked data stream.\n2. "Push" strategy. Create a special "feeder" object that will send data chunks to a specified consumer function.\n\nFirst strategy is simple an sutable for most of applications, but second gives more flexibility in building non-trivial\nin-memory processing pipelines.\n\n### <code>chunks</code> and <code>achunks</code> functions\nThese functions implement the "pull" chunking strategy resectively for synchronous and async/await apllications.\n```python\nfrom datachunks import chunks\n\nfor chunk in chunks(range(12), 5):\n    print(chunk)\n```\nExpected output:\n```\n[0, 1, 2, 3, 4]\n[5, 6, 7, 8, 9]\n[10, 11]\n```\nAsynchronous version example:\n```python\nimport asyncio\nfrom datachunks import achunks\n\nasync def arange(*args, **kwargs):\n    for i in range(*args, **kwargs):\n        yield i\n\nasync def achunks_demo():\n    async for chunk in achunks(arange(12), 5):\n        print(chunk)\n\nasyncio.run(achunks_demo())\n```\nExpected output:\n```\n[0, 1, 2, 3, 4]\n[5, 6, 7, 8, 9]\n[10, 11]\n```\n\n### Using "push" strategy\n"Push" strategy is implemented in <code>ChunkingFeeder</code> and <code>AsyncChunkingFeeder</code> objects.\n\nConsider the situation we decided to process odd and even numbers separately. For this purpose we create two feeders\nand <code>put</code> into them odd and even values. \n```python\nfrom datachunks import ChunkingFeeder\n\nwith ChunkingFeeder(lambda c: print(f\'evens: {c}\'), 5) as print_evens_feeder, \\\n    ChunkingFeeder(lambda c: print(f\'odds: {c}\'), 5) as print_odds_feeder:\n        for i in range(25):\n            if i % 2 == 0:\n                print_evens_feeder.put(i)\n            else:\n                print_odds_feeder.put(i)\n```\nExpected output:\n```\nevens: [0, 2, 4, 6, 8]\nodds: [1, 3, 5, 7, 9]\nevens: [10, 12, 14, 16, 18]\nodds: [11, 13, 15, 17, 19]\nodds: [21, 23]\nevens: [20, 22, 24]\n```\nAdditional features:\n- It is guarandeed that all data is delivered to the callback functions after the context exit.\n- It is possible to produce additional items in callback function. It allows to build flexible and even recursive data processing, but of course it is your responsibility to avoid infinite recursion.\n- By default <code>ChunkingFeeder</code> calls its chunk consumer synchronously. To use multithreading specify the <code>workers_num</code> parameter.\n- To use multiprocessing set the <code>multiprocessing</code> parameter to <code>True</code> in addition to <code>workers_num</code> parameter.\n- The <code>AsyncChunkingFeeder</code> also supports the <code>workers_num</code> parameter, but does not support <code>multiprocessing</code>.\n\n## ETL example\nConsider a simple ETL task: we have an <code>orders.jsonl</code> file that we need to upload to some Mongo database. Sending objects one-by-one is too slow, and file is too big to opload it in one big batch. So we are going to split this data to chunks of reasonable size.\n\nFunction <code>read_jsonl</code> reads the file and yields objects one-by-one:\n```python\nimport json\n\ndef read_jsonl():\n    with open(\'orders.jsonl\', \'r\', encoding=\'utf-8\') as jsonl:\n        for jsoned_obj in jsonl:\n            if jsoned_obj:\n                yield json.loads(jsoned_obj)\n```\nThe following function pulls objects through chunks generator and send objects to some MongoDB:\n```python\nfrom datachunks import chunks\n\ndef transfer_orders(db_connection):\n    for chunk in chunks(read_jsonl(), chunk_size=200):\n        db_connection.orders.insert_many(chunk)\n```\nAfter a while we decided to store purchase and sales orders into different MongoDB collections, so let\'s use two chunking feeders:\n```python\nfrom datachunks import ChunkingFeeder\n\nclass TransferOrders():\n    def __init__(self, db_connection):\n        self.db_connection = db_connection\n\n    def store_purchase_orders(self, chunk):\n        self.db_connection.purchase_orders.insert_many(chunk)\n\n    def store_sales_orders(self, chunk):\n        self.db_connection.sales_orders.insert_many(chunk)\n\n    def do_transfer(self):\n        with ChunkingFeeder(self.store_purchase_orders, 100, workers_num=1) as purchase_feeder, \\\n            ChunkingFeeder(self.store_purchase_orders, 500, workers_num=1) as sales_feeder:\n            for order in read_jsonl():\n                if order.get(\'order_type\') == \'purchase\':\n                    purchase_feeder.put(order)\n                elif order.get(\'order_type\') == \'sales\':\n                    sales_feeder.put(order)\n\ndef transfer_orders(db_connection):\n    TransferOrders(db_connection).do_transfer()\n```\nThe <code>pymongo</code> library is thread-safe, so it makes sense to speed up our process by storing data in separate threads.\n',
    'author': 'Alexander Maslyeav',
    'author_email': 'maslyaev@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/amaslyaev/datachunks/',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
