import unittest

from tonclient.errors import TonException
from tonclient.test.helpers import async_core_client, sync_core_client


class TestTonBocAsyncCore(unittest.TestCase):
    def test_parse_message(self):
        message = 'te6ccgEBAQEAWAAAq2n+AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAE/zMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzSsG8DgAAAAAjuOu9NAL7BxYpA'
        parsed = async_core_client.boc.parse_message(boc=message)
        self.assertEqual(
            'dfd47194f3058ee058bfbfad3ea40cbbd9ad17ca77cd0904d4d9f18a48c2fbca',
            parsed['id'])
        self.assertEqual(
            '-1:0000000000000000000000000000000000000000000000000000000000000000',
            parsed['src'])
        self.assertEqual(
            '-1:3333333333333333333333333333333333333333333333333333333333333333',
            parsed['dst'])

        with self.assertRaises(TonException):
            async_core_client.boc.parse_message(boc='Wrong==')

    def test_parse_transaction(self):
        transaction = 'te6ccgECBwEAAZQAA7V75gA6WK5sEDTiHFGnH9ILOy2irjKLWTkWQMyMogsg40AAACDribjoE3gOAbYNpCaX4uLeXPQHt2Kw/Jp2OKkR2s+BASyeQM6wAAAg64IXyBX2DobAABRrMENIBQQBAhUEQojmJaAYazBCEQMCAFvAAAAAAAAAAAAAAAABLUUtpEnlC4z33SeGHxRhIq/htUa7i3D8ghbwxhQTn44EAJxC3UicQAAAAAAAAAAAdwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAgnJAnYEvIQY6SnQKc3lXk6x1Z/lyplGFRbwAuNtVBi9EeceU3Ojl0F3EkRdylowY5x2qlgHNv4lNZUjhq0WqrLMNAQGgBgC3aADLL4ChL2HyLHwOLub5Mep87W3xdnMW8BpxKyVoGe3RPQAvmADpYrmwQNOIcUacf0gs7LaKuMotZORZAzIyiCyDjQ5iWgAGFFhgAAAEHXC9CwS+wdDGKTmMFkA='
        parsed = async_core_client.boc.parse_transaction(boc=transaction)
        self.assertEqual(
            'd6315dbb2a741a2765da250bea4a186adf942469369c703c57c2050e2d6e9fe3',
            parsed['id'])
        self.assertEqual('0x20eb89b8e81', parsed['lt'])
        self.assertEqual(1600186476, parsed['now'])

        with self.assertRaises(TonException):
            async_core_client.boc.parse_transaction(boc='Wrong==')

    def test_parse_account(self):
        account = 'te6ccgECHQEAA/wAAnfAArtKDoOR5+qId/SCUGSSS9Qc4RD86X6TnTMjmZ4e+7EyOobmQvsHNngAAAg6t/34DgJWKJuuOehjU0ADAQFBlcBqp0PR+QAN1kt1SY8QavS350RCNNfeZ+ommI9hgd/gAgBToB6t2E3E7a7aW2YkvXv2hTmSWVRTvSYmCVdH4HjgZ4Z94AAAAAvsHNwwAib/APSkICLAAZL0oOGK7VNYMPShBgQBCvSkIPShBQAAAgEgCgcBAv8IAf5/Ie1E0CDXScIBn9P/0wD0Bfhqf/hh+Gb4Yo4b9AVt+GpwAYBA9A7yvdcL//hicPhjcPhmf/hh4tMAAY4SgQIA1xgg+QFY+EIg+GX5EPKo3iP4RSBukjBw3vhCuvLgZSHTP9MfNCD4I7zyuSL5ACD4SoEBAPQOIJEx3vLQZvgACQA2IPhKI8jLP1mBAQD0Q/hqXwTTHwHwAfhHbvJ8AgEgEQsCAVgPDAEJuOiY/FANAdb4QW6OEu1E0NP/0wD0Bfhqf/hh+Gb4Yt7RcG1vAvhKgQEA9IaVAdcLP3+TcHBw4pEgjjJfM8gizwv/Ic8LPzExAW8iIaQDWYAg9ENvAjQi+EqBAQD0fJUB1ws/f5NwcHDiAjUzMehfAyHA/w4AmI4uI9DTAfpAMDHIz4cgzo0EAAAAAAAAAAAAAAAAD3RMfijPFiFvIgLLH/QAyXH7AN4wwP+OEvhCyMv/+EbPCwD4SgH0AMntVN5/+GcBCbkWq+fwEAC2+EFujjbtRNAg10nCAZ/T/9MA9AX4an/4Yfhm+GKOG/QFbfhqcAGAQPQO8r3XC//4YnD4Y3D4Zn/4YeLe+Ebyc3H4ZtH4APhCyMv/+EbPCwD4SgH0AMntVH/4ZwIBIBUSAQm7Fe+TWBMBtvhBbo4S7UTQ0//TAPQF+Gp/+GH4Zvhi3vpA1w1/ldTR0NN/39cMAJXU0dDSAN/RVHEgyM+FgMoAc89AzgH6AoBrz0DJc/sA+EqBAQD0hpUB1ws/f5NwcHDikSAUAISOKCH4I7ubIvhKgQEA9Fsw+GreIvhKgQEA9HyVAdcLP3+TcHBw4gI1MzHoXwb4QsjL//hGzwsA+EoB9ADJ7VR/+GcCASAYFgEJuORhh1AXAL74QW6OEu1E0NP/0wD0Bfhqf/hh+Gb4Yt7U0fhFIG6SMHDe+EK68uBl+AD4QsjL//hGzwsA+EoB9ADJ7VT4DyD7BCDQ7R7tU/ACMPhCyMv/+EbPCwD4SgH0AMntVH/4ZwIC2hsZAQFIGgAs+ELIy//4Rs8LAPhKAfQAye1U+A/yAAEBSBwAWHAi0NYCMdIAMNwhxwDcIdcNH/K8UxHdwQQighD////9vLHyfAHwAfhHbvJ8'
        parsed = async_core_client.boc.parse_account(boc=account)
        self.assertEqual(
            '0:2bb4a0e8391e7ea8877f4825064924bd41ce110fce97e939d3323999e1efbb13',
            parsed['id'])
        self.assertEqual('0x20eadff7e03', parsed['last_trans_lt'])
        self.assertEqual('0x958a26eb8e7a18d', parsed['balance'])

        with self.assertRaises(TonException):
            async_core_client.boc.parse_account(boc='Wrong==')

    def test_parse_block(self):
        block = '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'
        parsed = async_core_client.boc.parse_block(boc=block)
        self.assertEqual(
            '048f59d5d652459939ea5c5e7b291155205696b71e0c556f641df69e70e1e725',
            parsed['id'])
        self.assertEqual(4296363, parsed['seq_no'])
        self.assertEqual(1600234696, parsed['gen_utime'])

        with self.assertRaises(TonException):
            async_core_client.boc.parse_block(boc='Wrong==')

    def test_get_blockchain_config(self):
        block = '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'
        config_boc = async_core_client.boc.get_blockchain_config(
            block_boc=block)
        self.assertEqual(
            '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',
            config_boc)


class TestTonBocSyncCore(unittest.TestCase):
    """ Sync core is not recommended to use, so make just a couple of tests """
    def test_parse_message(self):
        message = 'te6ccgEBAQEAWAAAq2n+AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAE/zMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzMzSsG8DgAAAAAjuOu9NAL7BxYpA'
        parsed = sync_core_client.boc.parse_message(boc=message)
        self.assertEqual(
            'dfd47194f3058ee058bfbfad3ea40cbbd9ad17ca77cd0904d4d9f18a48c2fbca',
            parsed['id'])
        self.assertEqual(
            '-1:0000000000000000000000000000000000000000000000000000000000000000',
            parsed['src'])
        self.assertEqual(
            '-1:3333333333333333333333333333333333333333333333333333333333333333',
            parsed['dst'])

        with self.assertRaises(TonException):
            sync_core_client.boc.parse_message(boc='Wrong==')

    def test_parse_block(self):
        block = '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'
        parsed = sync_core_client.boc.parse_block(boc=block)
        self.assertEqual(
            '048f59d5d652459939ea5c5e7b291155205696b71e0c556f641df69e70e1e725',
            parsed['id'])
        self.assertEqual(4296363, parsed['seq_no'])
        self.assertEqual(1600234696, parsed['gen_utime'])

        with self.assertRaises(TonException):
            sync_core_client.boc.parse_block(boc='Wrong==')
