#!/usr/bin/python3
"""DMC floss mapping."""
import pyxstitch.hex as hu


class FlossException(Exception):
    """Floss exception."""


class FlossType(object):
    """Floss type."""

    def __init__(self, dmc):
        """Init the instance."""
        self.floss_number = dmc[0]
        self.name = dmc[1]
        self.rgb = hu.to_rgb_string(dmc[2])


class Floss(object):
    """Floss definitions."""

    def __init__(self):
        """Init the instance."""
        self._colors = {}
        self._cache = {}
        self._load()
        self._map = {}

    def _key_rgb(self, red, green, blue):
        """Create a cache key."""
        return "{}.{}.{}".format(red, green, blue)

    def _create_floss(self, dmc):
        """Create floss, handle any mappings."""
        floss_type = FlossType(dmc)
        if floss_type.rgb in self._map:
            floss_type = FlossType(self._map[floss_type.rgb])
        return floss_type

    def lookup(self, rgb):
        """Lookup a code."""
        rgb_str = self._key_rgb(rgb[0], rgb[1], rgb[2])
        if rgb_str in self._cache:
            return self._create_floss(self._cache[rgb_str])
        # try harder...
        close = None
        closest = -1
        for tries in self._colors.keys():
            t = self._colors[tries][2]
            check = hu.rgb_close(rgb[0], rgb[1], rgb[2], t[0], t[1], t[2])
            if closest == -1 or check < closest:
                closest = check
                close = self._colors[tries]
        self._cache[rgb_str] = close
        return self._create_floss(close)

    def map(self, from_color, to_color):
        """Map one floss rgb to another."""
        if to_color in self._colors or to_color is None:
            if to_color is None:
                self._colors.pop(from_color, None)
            else:
                self._map[from_color] = self._colors[to_color]
            self._cache = {}
            return True
        raise FlossException("Unknown color(s): {} -> {}".format(from_color,
                                                                 to_color))

    def _add(self, number, desc, red, green, blue):
        """Add a color."""
        rgb = (red, green, blue)
        self._colors[hu.to_rgb_string(rgb)] = (number, desc, rgb)

    def _load(self):
        """Load all colors."""
        self._add('01', 'White Tin', 227, 227, 230)
        self._add('02', 'Tin', 215, 215, 216)
        self._add('03', 'Tin Medium', 184, 184, 187)
        self._add('04', 'Tin Dark', 174, 174, 177)
        self._add('05', 'Driftwood Light', 227, 204, 190)
        self._add('06', 'Driftwood Medium Light', 220, 198, 184)
        self._add('07', 'Driftwood', 143, 123, 110)
        self._add('08', 'Driftwood Dark', 106, 80, 70)
        self._add('09', 'Cocoa Very Dark', 85, 32, 14)
        self._add('10', 'Tender Green Very Light', 237, 254, 217)
        self._add('11', 'Tender Green Light', 226, 237, 181)
        self._add('12', 'Tender Green', 205, 217, 154)
        self._add('13', 'Nile Green Medium Light', 191, 246, 224)
        self._add('14', 'Apple Green Pale', 208, 251, 178)
        self._add('15', 'Apple Green', 209, 237, 164)
        self._add('150', 'Dusty Rose Ultra Very Dark', 171, 2, 73)
        self._add('151', 'Dusty Rose Very Light', 240, 206, 212)
        self._add('152', 'Shell Pink Medium Light', 226, 160, 153)
        self._add('153', 'Violet Very Light', 230, 204, 217)
        self._add('154', 'Grape Very Dark', 87, 36, 51)
        self._add('155', 'Blue Violet Medium Dark', 152, 145, 182)
        self._add('156', 'Blue Violet Medium Light', 163, 174, 209)
        self._add('157', 'Cornflower Blue Very Light', 187, 195, 217)
        self._add('158', 'Cornflower Blue  Very Dark', 76, 82, 110)
        self._add('159', 'Blue Gray Light', 199, 202, 215)
        self._add('16', 'Chartreuse Light', 201, 194, 88)
        self._add('160', 'Blue Gray Medium', 153, 159, 183)
        self._add('161', 'Blue Gray', 120, 128, 164)
        self._add('162', 'Blue Ultra Very Light', 219, 236, 245)
        self._add('163', 'Celadon Green Medium', 77, 131, 97)
        self._add('164', 'Forest Green Light', 200, 216, 184)
        self._add('165', 'Moss Green Very Light', 239, 244, 164)
        self._add('166', 'Moss Green Medium Light', 192, 200, 64)
        self._add('167', 'Yellow Beige Very Dark', 167, 124, 73)
        self._add('168', 'Pewter Very Light', 209, 209, 209)
        self._add('169', 'Pewter Light', 132, 132, 132)
        self._add('17', 'Yellow Plum Light', 229, 226, 114)
        self._add('18', 'Yellow Plum', 217, 213, 109)
        self._add('19', 'Autumn Gold medium Light', 247, 201, 95)
        self._add('20', 'Shrimp', 247, 175, 147)
        self._add('208', 'Lavender Very Dark', 131, 91, 139)
        self._add('209', 'Lavender Dark', 163, 123, 167)
        self._add('21', 'Alizarin Light', 215, 153, 130)
        self._add('210', 'Lavender Medium', 195, 159, 195)
        self._add('211', 'Lavender Light', 227, 203, 227)
        self._add('22', 'Alizarin', 188, 96, 78)
        self._add('221', 'Shell Pink Very Dark', 136, 62, 67)
        self._add('223', 'Shell Pink Light', 204, 132, 124)
        self._add('224', 'Shell Pink Very Light', 235, 183, 175)
        self._add('225', 'Shell Pink Ultra Very Light', 255, 223, 213)
        self._add('23', 'Apple Blossom', 237, 226, 237)
        self._add('24', 'White Lavender', 224, 215, 238)
        self._add('25', 'Lavender Ultra Light', 218, 210, 233)
        self._add('26', 'Lavender Pale', 215, 202, 230)
        self._add('27', 'White Violet', 240, 238, 249)
        self._add('28', 'Eggplant Medium Light', 144, 134, 169)
        self._add('29', 'Eggplant', 103, 64, 118)
        self._add('30', 'Blueberry Medium Light', 125, 119, 165)
        self._add('300', 'Mahogany Very Dark', 111, 47, 0)
        self._add('301', 'Mahogany Medium', 179, 95, 43)
        self._add('3011', 'Khaki Green Dark', 137, 138, 88)
        self._add('3012', 'Khaki Green Medium', 166, 167, 93)
        self._add('3013', 'Khaki Green Light', 185, 185, 130)
        self._add('3021', 'Brown Gray Very Dark', 79, 75, 65)
        self._add('3022', 'Brown Gray Medium', 142, 144, 120)
        self._add('3023', 'Brown Gray Light', 177, 170, 151)
        self._add('3024', 'Brown Gray Very Light', 235, 234, 231)
        self._add('3031', 'Mocha Brown Very Dark', 75, 60, 42)
        self._add('3032', 'Mocha Brown Medium', 179, 159, 139)
        self._add('3033', 'Mocha Brown Very Light', 227, 216, 204)
        self._add('304', 'Red Medium', 183, 31, 51)
        self._add('3041', 'Antique Violet Medium', 149, 111, 124)
        self._add('3042', 'Antique Violet Light', 183, 157, 167)
        self._add('3045', 'Yellow Beige Dark', 188, 150, 106)
        self._add('3046', 'Yellow Beige Medium', 216, 188, 154)
        self._add('3047', 'Yellow Beige Light', 231, 214, 193)
        self._add('3051', 'Green Gray Dark', 95, 102, 72)
        self._add('3052', 'Green Gray Medium', 136, 146, 104)
        self._add('3053', 'Green Gray', 156, 164, 130)
        self._add('3064', 'Desert Sand', 196, 142, 112)
        self._add('307', 'Lemon', 253, 237, 84)
        self._add('3072', 'Beaver Gray Very Light', 230, 232, 232)
        self._add('3078', 'Golden Yellow Very Light', 253, 249, 205)
        self._add('309', 'Rose Dark', 86, 74, 74)
        self._add('31', 'Blueberry', 80, 81, 141)
        self._add('310', 'Black', 0, 0, 0)
        self._add('311', 'Wedgewood Ultra Very Dark', 28, 80, 102)
        self._add('312', 'Baby Blue Very Dark', 53, 102, 139)
        self._add('315', 'Antique Mauve Medium Dark', 129, 73, 82)
        self._add('316', 'Antique Mauve Medium', 183, 115, 127)
        self._add('317', 'Pewter Gray', 108, 108, 108)
        self._add('318', 'Steel Gray Light', 171, 171, 171)
        self._add('319', 'Pistachio Grn Very Dark', 32, 95, 46)
        self._add('32', 'Blueberry Dark', 77, 46, 138)
        self._add('320', 'Pistachio Green Medium', 105, 136, 90)
        self._add('321', 'Red', 199, 43, 59)
        self._add('322', 'Baby Blue Dark', 90, 143, 184)
        self._add('326', 'Rose Very Dark', 179, 59, 75)
        self._add('327', 'Violet Dark', 99, 54, 102)
        self._add('33', 'Fuschia', 156, 89, 158)
        self._add('3325', 'Baby Blue Light', 184, 210, 230)
        self._add('3326', 'Rose Light', 251, 173, 180)
        self._add('3328', 'Salmon Dark', 227, 109, 109)
        self._add('333', 'Blue Violet Very Dark', 92, 84, 120)
        self._add('334', 'Baby Blue Medium', 115, 159, 193)
        self._add('3340', 'Apricot Medium', 255, 131, 111)
        self._add('3341', 'Apricot', 252, 171, 152)
        self._add('3345', 'Hunter Green Dark', 27, 89, 21)
        self._add('3346', 'Hunter Green', 64, 106, 58)
        self._add('3347', 'Yellow Green Medium', 113, 147, 92)
        self._add('3348', 'Yellow Green Light', 204, 217, 177)
        self._add('335', 'Rose', 238, 84, 110)
        self._add('3350', 'Dusty Rose Ultra Dark', 188, 67, 101)
        self._add('3354', 'Dusty Rose Light', 228, 166, 172)
        self._add('336', 'Navy Blue', 37, 59, 115)
        self._add('3362', 'Pine Green Dark', 94, 107, 71)
        self._add('3363', 'Pine Green Medium', 114, 130, 86)
        self._add('3364', 'Pine Green', 131, 151, 95)
        self._add('3371', 'Black Brown', 30, 17, 8)
        self._add('34', 'Fuschia Dark', 125, 48, 100)
        self._add('340', 'Blue Violet Medium', 173, 167, 199)
        self._add('341', 'Blue Violet Light', 183, 191, 221)
        self._add('347', 'Salmon Very Dark', 191, 45, 45)
        self._add('349', 'Coral Dark', 210, 16, 53)
        self._add('35', 'Fuschia Very Dark', 70, 5, 45)
        self._add('350', 'Coral Medium', 224, 72, 72)
        self._add('351', 'Coral', 233, 106, 103)
        self._add('352', 'Coral Light', 253, 156, 151)
        self._add('353', 'Peach', 254, 215, 204)
        self._add('355', 'Terra Cotta Dark', 152, 68, 54)
        self._add('356', 'Terra Cotta Medium', 197, 106, 91)
        self._add('3607', 'Plum Light', 197, 73, 137)
        self._add('3608', 'Plum Very Light', 234, 156, 196)
        self._add('3609', 'Plum Ultra Light', 244, 174, 213)
        self._add('367', 'Pistachio Green Dark', 97, 122, 82)
        self._add('368', 'Pistachio Green Light', 166, 194, 152)
        self._add('3685', 'Mauve Very Dark', 136, 21, 49)
        self._add('3687', 'Mauve', 201, 107, 112)
        self._add('3688', 'Mauve Medium', 231, 169, 172)
        self._add('3689', 'Mauve Light', 251, 191, 194)
        self._add('369', 'Pistachio Green Very Light', 215, 237, 204)
        self._add('370', 'Mustard Medium', 184, 157, 100)
        self._add('3705', 'Melon Dark', 255, 121, 146)
        self._add('3706', 'Melon Medium', 255, 173, 188)
        self._add('3708', 'Melon Light', 255, 203, 213)
        self._add('371', 'Mustard', 191, 166, 113)
        self._add('3712', 'Salmon Medium', 241, 135, 135)
        self._add('3713', 'Salmon Very Light', 255, 226, 226)
        self._add('3716', 'Dusty Rose Medium Very Light', 255, 189, 189)
        self._add('372', 'Mustard Light', 204, 183, 132)
        self._add('3721', 'Shell Pink Dark', 161, 75, 81)
        self._add('3722', 'Shell Pink Medium', 188, 108, 100)
        self._add('3726', 'Antique Mauve Dark', 155, 91, 102)
        self._add('3727', 'Antique Mauve Light', 219, 169, 178)
        self._add('3731', 'Dusty Rose Very Dark', 218, 103, 131)
        self._add('3733', 'Dusty Rose', 232, 135, 155)
        self._add('3740', 'Antique Violet Dark', 120, 87, 98)
        self._add('3743', 'Antique Violet Very Light', 215, 203, 211)
        self._add('3746', 'Blue Violet Dark', 119, 107, 152)
        self._add('3747', 'Blue Violet Very Light', 211, 215, 237)
        self._add('3750', 'Antique Blue Very Dark', 56, 76, 94)
        self._add('3752', 'Antique Blue Very Light', 199, 209, 219)
        self._add('3753', 'Antique Blue Ultra Very Light', 219, 226, 233)
        self._add('3755', 'Baby Blue', 147, 180, 206)
        self._add('3756', 'Baby Blue Ultra Very Light', 238, 252, 252)
        self._add('3760', 'Wedgewood Medium', 62, 133, 162)
        self._add('3761', 'Sky Blue Light', 172, 216, 226)
        self._add('3765', 'Peacock Blue Very Dark', 52, 127, 140)
        self._add('3766', 'Peacock Blue Light', 153, 207, 217)
        self._add('3768', 'Gray Green Dark', 101, 127, 127)
        self._add('3770', 'Tawny Vy Light', 255, 238, 227)
        self._add('3771', 'Terra Cotta Ultra Very Light', 244, 187, 169)
        self._add('3772', 'Desert Sand Very Dark', 160, 108, 80)
        self._add('3773', 'Desert Sand Dark', 182, 117, 82)
        self._add('3774', 'Desert Sand Very Light', 243, 225, 215)
        self._add('3776', 'Mahogany Light', 207, 121, 57)
        self._add('3777', 'Terra Cotta Very Dark', 134, 48, 34)
        self._add('3778', 'Terra Cotta Light', 217, 137, 120)
        self._add('3779', 'Rosewood Ultra Very Light', 248, 202, 200)
        self._add('3781', 'Mocha Brown Dark', 107, 87, 67)
        self._add('3782', 'Mocha Brown Light', 210, 188, 166)
        self._add('3787', 'Brown Gray Dark', 98, 93, 80)
        self._add('3790', 'Beige Gray Ultra Dark', 127, 106, 85)
        self._add('3799', 'Pewter Gray Very Dark', 66, 66, 66)
        self._add('3801', 'Melon Very Dark', 231, 73, 103)
        self._add('3802', 'Antique Mauve Very Darkv', 113, 65, 73)
        self._add('3803', 'Mauve Dark', 171, 51, 87)
        self._add('3804', 'Cyclamen Pink Dark', 224, 40, 118)
        self._add('3805', 'Cyclamen Pink', 243, 71, 139)
        self._add('3806', 'Cyclamen Pink Light', 255, 140, 174)
        self._add('3807', 'Cornflower Blue', 96, 103, 140)
        self._add('3808', 'Turquoise Ultra Very Dark', 54, 105, 112)
        self._add('3809', 'Turquoise Vy Dark', 63, 124, 133)
        self._add('3810', 'Turquoise Dark', 72, 142, 154)
        self._add('3811', 'Turquoise Very Light', 188, 227, 230)
        self._add('3812', 'Sea Green Very Dark', 47, 140, 132)
        self._add('3813', 'Blue Green Light', 178, 212, 189)
        self._add('3814', 'Aquamarine', 80, 139, 125)
        self._add('3815', 'Celadon Green Dark', 71, 119, 89)
        self._add('3816', 'Celadon Green', 101, 165, 125)
        self._add('3817', 'Celadon Green Light', 153, 195, 170)
        self._add('3818', 'Emerald Green Ultra Very Dark', 17, 90, 59)
        self._add('3819', 'Moss Green Light', 224, 232, 104)
        self._add('3820', 'Straw Dark', 223, 182, 95)
        self._add('3821', 'Straw', 243, 206, 117)
        self._add('3822', 'Straw Light', 246, 220, 152)
        self._add('3823', 'Yellow Ultra Pale', 255, 253, 227)
        self._add('3824', 'Apricot Light', 254, 205, 194)
        self._add('3825', 'Pumpkin Pale', 253, 189, 150)
        self._add('3826', 'Golden Brown', 173, 114, 57)
        self._add('3827', 'Golden Brown Pale', 247, 187, 119)
        self._add('3828', 'Hazelnut Brown', 183, 139, 97)
        self._add('3829', 'Old Gold Vy Dark', 169, 130, 4)
        self._add('3830', 'Terra Cotta', 185, 85, 68)
        self._add('3831', 'Raspberry Dark', 179, 47, 72)
        self._add('3832', 'Raspberry Medium', 219, 85, 110)
        self._add('3833', 'Raspberry Light', 234, 134, 153)
        self._add('3834', 'Grape Dark', 114, 55, 93)
        self._add('3835', 'Grape Medium', 148, 96, 131)
        self._add('3836', 'Grape Light', 186, 145, 170)
        self._add('3837', 'Lavender Ultra Dark', 108, 58, 110)
        self._add('3838', 'Lavender Blue Dark', 92, 114, 148)
        self._add('3839', 'Lavender Blue Medium', 123, 142, 171)
        self._add('3840', 'Lavender Blue Light', 176, 192, 218)
        self._add('3841', 'Baby Blue Pale', 205, 223, 237)
        self._add('3842', 'Wedgewood Very Dark', 50, 102, 124)
        self._add('3843', 'Electric Blue', 20, 170, 208)
        self._add('3844', 'Turquoise Bright Dark', 18, 174, 186)
        self._add('3845', 'Turquoise Bright Medium', 4, 196, 202)
        self._add('3846', 'Turquoise Bright Light', 6, 227, 230)
        self._add('3847', 'Teal Green Dark', 52, 125, 117)
        self._add('3848', 'Teal Green Medium', 85, 147, 146)
        self._add('3849', 'Teal Green Light', 82, 179, 164)
        self._add('3850', 'Green Bright Dark', 55, 132, 119)
        self._add('3851', 'Green Bright Light', 73, 179, 161)
        self._add('3852', 'Straw Very Dark', 205, 157, 55)
        self._add('3853', 'Autumn Gold Dark', 242, 151, 70)
        self._add('3854', 'Autumn Gold Medium', 242, 175, 104)
        self._add('3855', 'Autumn Gold Light', 250, 211, 150)
        self._add('3856', 'Mahogany Ultra Very Light', 255, 211, 181)
        self._add('3857', 'Rosewood Dark', 104, 37, 26)
        self._add('3858', 'Rosewood Medium', 150, 74, 63)
        self._add('3859', 'Rosewood Light', 186, 139, 124)
        self._add('3860', 'Cocoa', 125, 93, 87)
        self._add('3861', 'Cocoa Light', 166, 136, 129)
        self._add('3862', 'Mocha Beige Dark', 138, 110, 78)
        self._add('3863', 'Mocha Beige Medium', 164, 131, 92)
        self._add('3864', 'Mocha Beige Light', 203, 182, 156)
        self._add('3865', 'Winter White', 249, 247, 241)
        self._add('3866', 'Mocha Brown Ultra Very Light', 250, 246, 240)
        self._add('400', 'Mahogany Dark', 143, 67, 15)
        self._add('402', 'Mahogany Very Light', 247, 167, 119)
        self._add('407', 'Desert Sand Medium', 187, 129, 97)
        self._add('413', 'Pewter Gray Dark', 86, 86, 86)
        self._add('414', 'Steel Gray Dark', 140, 140, 140)
        self._add('415', 'Pearl Gray', 211, 211, 214)
        self._add('420', 'Hazelnut Brown Dark', 160, 112, 66)
        self._add('422', 'Hazelnut Brown Light', 198, 159, 123)
        self._add('433', 'Brown Medium', 122, 69, 31)
        self._add('434', 'Brown Light', 152, 94, 51)
        self._add('435', 'Brown Very Light', 184, 119, 72)
        self._add('436', 'Tan', 203, 144, 81)
        self._add('437', 'Tan Light', 228, 187, 142)
        self._add('444', 'Lemon Dark', 255, 214, 0)
        self._add('445', 'Lemon Light', 255, 251, 139)
        self._add('451', 'Shell Gray Dark', 145, 123, 115)
        self._add('452', 'Shell Gray Medium', 192, 179, 174)
        self._add('453', 'Shell Gray Light', 215, 206, 203)
        self._add('469', 'Avocado Green', 114, 132, 60)
        self._add('470', 'Avocado Grn Light', 148, 171, 79)
        self._add('471', 'Avocado Grn Very Light', 174, 191, 121)
        self._add('472', 'Avocado Green Ultra Light', 216, 228, 152)
        self._add('498', 'Red Dark', 167, 19, 43)
        self._add('500', 'Blue Green Very Dark', 4, 77, 51)
        self._add('501', 'Blue Green Dark', 57, 111, 82)
        self._add('502', 'Blue Green', 91, 144, 113)
        self._add('503', 'Blue Green Medium', 123, 172, 148)
        self._add('504', 'Blue Green Very Light', 196, 222, 204)
        self._add('505', 'Jade Green', 51, 131, 98)
        self._add('517', 'Wedgewood Dark', 59, 118, 143)
        self._add('518', 'Wedgewood Light', 79, 147, 167)
        self._add('519', 'Sky Blue', 126, 177, 200)
        self._add('520', 'Fern Green Dark', 102, 109, 79)
        self._add('522', 'Fern Green', 150, 158, 126)
        self._add('523', 'Fern Green Light', 171, 177, 151)
        self._add('524', 'Fern Green Very Light', 196, 205, 172)
        self._add('535', 'Ash Gray Very Light', 99, 100, 88)
        self._add('543', 'Beige Brown Ultra Very Light', 242, 227, 206)
        self._add('550', 'Violet Very Dark', 92, 24, 78)
        self._add('552', 'Violet  Medium', 128, 58, 107)
        self._add('553', 'Violet', 163, 99, 139)
        self._add('554', 'Violet Light', 219, 179, 203)
        self._add('561', 'Celadon Green VD', 44, 106, 69)
        self._add('562', 'Jade Medium', 83, 151, 106)
        self._add('563', 'Jade Light', 143, 192, 152)
        self._add('564', 'Jade Very Light', 167, 205, 175)
        self._add('580', 'Moss Green Dark', 136, 141, 51)
        self._add('581', 'Moss Green', 167, 174, 56)
        self._add('597', 'Turquoise', 91, 163, 179)
        self._add('598', 'Turquoise Light', 144, 195, 204)
        self._add('600', 'Cranberry Very Dark', 205, 47, 99)
        self._add('601', 'Cranberry Dark', 209, 40, 106)
        self._add('602', 'Cranberry Medium', 226, 72, 116)
        self._add('603', 'Cranberry', 255, 164, 190)
        self._add('604', 'Cranberry Light', 255, 176, 190)
        self._add('605', 'Cranberry Very Light', 255, 192, 205)
        self._add('606', 'Orange Red Bright', 250, 50, 3)
        self._add('608', 'Burnt Orange Bright', 253, 93, 53)
        self._add('610', 'Drab Brown Dark', 121, 96, 71)
        self._add('611', 'Drab Brown', 150, 118, 86)
        self._add('612', 'Drab Brown Light', 188, 154, 120)
        self._add('613', 'Drab Brown Very Light', 220, 196, 170)
        self._add('632', 'Desert Sand Ultra Very Dark', 135, 85, 57)
        self._add('640', 'Beige Gray Very Dark', 133, 123, 97)
        self._add('642', 'Beige Gray Dark', 164, 152, 120)
        self._add('644', 'Beige Gray Medium', 221, 216, 203)
        self._add('645', 'Beaver Gray Very Dark', 110, 101, 92)
        self._add('646', 'Beaver Gray Dark', 135, 125, 115)
        self._add('647', 'Beaver Gray Medium', 176, 166, 156)
        self._add('648', 'Beaver Gray Light', 188, 180, 172)
        self._add('666', 'Bright Red', 227, 29, 66)
        self._add('676', 'Old Gold Light', 229, 206, 151)
        self._add('677', 'Old Gold Very Light', 245, 236, 203)
        self._add('680', 'Old Gold Dark', 188, 141, 14)
        self._add('699', 'Green', 5, 101, 23)
        self._add('700', 'Green Bright', 7, 115, 27)
        self._add('701', 'Green Light', 63, 143, 41)
        self._add('702', 'Kelly Green', 71, 167, 47)
        self._add('703', 'Chartreuse', 123, 181, 71)
        self._add('704', 'Chartreuse Bright', 158, 207, 52)
        self._add('712', 'Cream', 255, 251, 239)
        self._add('718', 'Plum', 156, 36, 98)
        self._add('720', 'Orange Spice Dark', 229, 92, 31)
        self._add('721', 'Orange Spice Medium', 242, 120, 66)
        self._add('722', 'Orange Spice Light', 247, 151, 111)
        self._add('725', 'Topaz Medium Light', 255, 200, 64)
        self._add('726', 'Topaz Light', 253, 215, 85)
        self._add('727', 'Topaz Very Light', 255, 241, 175)
        self._add('728', 'Topaz', 228, 180, 104)
        self._add('729', 'Old Gold Medium', 208, 165, 62)
        self._add('730', 'Olive Green Very Dark', 130, 123, 48)
        self._add('731', 'Olive Green Dark', 147, 139, 55)
        self._add('732', 'Olive Green', 148, 140, 54)
        self._add('733', 'Olive Green Medium', 188, 179, 76)
        self._add('734', 'Olive Green Light', 199, 192, 119)
        self._add('738', 'Tan Very Light', 236, 204, 158)
        self._add('739', 'Tan Ultra Very Light', 248, 228, 200)
        self._add('740', 'Tangerine', 255, 139, 0)
        self._add('741', 'Tangerine Medium', 255, 163, 43)
        self._add('742', 'Tangerine Light', 255, 191, 87)
        self._add('743', 'Yellow Medium', 254, 211, 118)
        self._add('744', 'Yellow Pale', 255, 231, 147)
        self._add('745', 'Yellow Pale Light', 255, 233, 173)
        self._add('746', 'Off White', 252, 252, 238)
        self._add('747', 'Peacock Blue Very Light', 229, 252, 253)
        self._add('754', 'Peach Light', 247, 203, 191)
        self._add('758', 'Terra Cotta Very Light', 238, 170, 155)
        self._add('760', 'Salmon', 245, 173, 173)
        self._add('761', 'Salmon Light', 255, 201, 201)
        self._add('762', 'Pearl Gray Very Light', 236, 236, 236)
        self._add('772', 'Yellow Green Very Light', 228, 236, 212)
        self._add('775', 'Baby Blue Very Light', 217, 235, 241)
        self._add('776', 'Pink Medium', 252, 176, 185)
        self._add('777', 'Raspberry Very Dark', 145, 53, 70)
        self._add('778', 'Antique Mauve Very Light', 223, 179, 187)
        self._add('779', 'Cocoa Dark', 98, 75, 69)
        self._add('780', 'Topaz Ultra Very Dark', 148, 99, 26)
        self._add('781', 'Topaz Very Dark', 162, 109, 32)
        self._add('782', 'Topaz Dark', 174, 119, 32)
        self._add('783', 'Topaz Medium', 206, 145, 36)
        self._add('791', 'Cornflower Blue Very Dark', 70, 69, 99)
        self._add('792', 'Cornflower Blue Dark', 85, 91, 123)
        self._add('793', 'Cornflower Blue Medium', 112, 125, 162)
        self._add('794', 'Cornflower Blue Light', 143, 156, 193)
        self._add('796', 'Royal Blue Dark', 17, 65, 109)
        self._add('797', 'Royal Blue', 19, 71, 125)
        self._add('798', 'Delft Blue Dark', 70, 106, 142)
        self._add('799', 'Delft Blue Medium', 116, 142, 182)
        self._add('800', 'Delft Blue Pale', 192, 204, 222)
        self._add('801', 'Coffee Brown Dark', 101, 57, 25)
        self._add('803', 'Baby Blue Ultra Very Dark', 44, 89, 124)
        self._add('806', 'Peacock Blue Dark', 61, 149, 165)
        self._add('807', 'Peacock Blue', 100, 171, 186)
        self._add('809', 'Delft Blue', 148, 168, 198)
        self._add('813', 'Blue Light', 161, 194, 215)
        self._add('814', 'Garnet Dark', 123, 0, 27)
        self._add('815', 'Garnet Medium', 135, 7, 31)
        self._add('816', 'Garnet', 151, 11, 35)
        self._add('817', 'Coral Red Very Dark', 187, 5, 31)
        self._add('818', 'Baby Pink', 255, 223, 217)
        self._add('819', 'Baby Pink Light', 255, 238, 235)
        self._add('820', 'Royal Blue Very Dark', 14, 54, 92)
        self._add('822', 'Beige Gray Light', 231, 226, 211)
        self._add('823', 'Navy Blue Dark', 33, 48, 99)
        self._add('824', 'Blue Very Dark', 57, 105, 135)
        self._add('825', 'Blue Dark', 71, 129, 165)
        self._add('826', 'Blue Medium', 107, 158, 191)
        self._add('827', 'Blue Very Light', 189, 221, 237)
        self._add('828', 'Sky Blue Very Light', 197, 232, 237)
        self._add('829', 'Golden Olive Very Dark', 126, 107, 66)
        self._add('830', 'Golden Olive Dark', 141, 120, 75)
        self._add('831', 'Golden Olive Medium', 170, 143, 86)
        self._add('832', 'Golden Olive', 189, 155, 81)
        self._add('833', 'Golden Olive Light', 200, 171, 108)
        self._add('834', 'Golden Olive Very Light', 219, 190, 127)
        self._add('838', 'Beige Brown Very Dark', 89, 73, 55)
        self._add('839', 'Beige Brown Dark', 103, 85, 65)
        self._add('840', 'Beige Brown Medium', 154, 124, 92)
        self._add('841', 'Beige Brown Light', 182, 155, 126)
        self._add('842', 'Beige Brown Very Light', 209, 186, 161)
        self._add('844', 'Beaver Gray Ultra Dark', 72, 72, 72)
        self._add('869', 'Hazelnut Brown Very Dark', 131, 94, 57)
        self._add('890', 'Pistachio Grn Ultra Very Dark', 23, 73, 35)
        self._add('891', 'Carnation Dark', 255, 87, 115)
        self._add('892', 'Carnation Medium', 255, 121, 140)
        self._add('893', 'Carnation Light', 252, 144, 162)
        self._add('894', 'Carnation Very Light', 255, 178, 187)
        self._add('895', 'Hunter Green Very Dark', 27, 83, 0)
        self._add('898', 'Coffee Brown Very Dark', 73, 42, 19)
        self._add('899', 'Rose Medium', 242, 118, 136)
        self._add('900', 'Burnt Orange Dark', 209, 88, 7)
        self._add('902', 'Garnet Very Dark', 130, 38, 55)
        self._add('904', 'Parrot Green Very Dark', 85, 120, 34)
        self._add('905', 'Parrot Green Dark', 98, 138, 40)
        self._add('906', 'Parrot Green Medium', 127, 179, 53)
        self._add('907', 'Parrot Green Light', 199, 230, 102)
        self._add('909', 'Emerald Green Very Dark', 21, 111, 73)
        self._add('910', 'Emerald Green Dark', 24, 126, 86)
        self._add('911', 'Emerald Green Medium', 24, 144, 101)
        self._add('912', 'Emerald Green Light', 27, 157, 107)
        self._add('913', 'Nile Green Medium', 109, 171, 119)
        self._add('915', 'Plum Dark', 130, 0, 67)
        self._add('917', 'Plum Medium', 155, 19, 89)
        self._add('918', 'Red Copper Dark', 130, 52, 10)
        self._add('919', 'Red Copper', 166, 69, 16)
        self._add('920', 'Copper Medium', 172, 84, 20)
        self._add('921', 'Copper', 198, 98, 24)
        self._add('922', 'Copper Light', 226, 115, 35)
        self._add('924', 'Gray Green Vy Dark', 86, 106, 106)
        self._add('926', 'Gray Green Medium', 152, 174, 174)
        self._add('927', 'Gray Green Light', 189, 203, 203)
        self._add('928', 'Gray Green Very Light', 221, 227, 227)
        self._add('930', 'Antique Blue Dark', 69, 92, 113)
        self._add('931', 'Antique Blue Medium', 106, 133, 158)
        self._add('932', 'Antique Blue Light', 162, 181, 198)
        self._add('934', 'Avocado Grn Black', 49, 57, 25)
        self._add('935', 'Avocado Green Dark', 66, 77, 33)
        self._add('936', 'Avocado Green Very Dark', 76, 88, 38)
        self._add('937', 'Avocado Green Medium', 98, 113, 51)
        self._add('938', 'Coffee Brown Ultra Dark', 54, 31, 14)
        self._add('939', 'Navy Blue Very Dark', 27, 40, 83)
        self._add('943', 'Green Bright Medium', 61, 147, 132)
        self._add('945', 'Tawny', 251, 213, 187)
        self._add('946', 'Burnt Orange Medium', 235, 99, 7)
        self._add('947', 'Burnt Orange', 255, 123, 77)
        self._add('948', 'Peach Very Light', 254, 231, 218)
        self._add('950', 'Desert Sand Light', 238, 211, 196)
        self._add('951', 'Tawny Light', 255, 226, 207)
        self._add('954', 'Nile Green', 136, 186, 145)
        self._add('955', 'Nile Green Light', 162, 214, 173)
        self._add('956', 'Geranium', 255, 145, 145)
        self._add('957', 'Geranium Pale', 253, 181, 181)
        self._add('958', 'Sea Green Dark', 62, 182, 161)
        self._add('959', 'Sea Green Medium', 89, 199, 180)
        self._add('961', 'Dusty Rose Dark', 207, 115, 115)
        self._add('962', 'Dusty Rose Medium', 230, 138, 138)
        self._add('963', 'Dusty Rose Ultra Very Light', 255, 215, 215)
        self._add('964', 'Sea Green Light', 169, 226, 216)
        self._add('966', 'Jade Ultra Very Light', 185, 215, 192)
        self._add('967', 'Apricot Very Light', 255, 222, 213)
        self._add('970', 'Pumpkin Light', 247, 139, 19)
        self._add('971', 'Pumpkin', 246, 127, 0)
        self._add('972', 'Canary Deep', 255, 181, 21)
        self._add('973', 'Canary Bright', 255, 227, 0)
        self._add('975', 'Golden Brown Dark', 145, 79, 18)
        self._add('976', 'Golden Brown Medium', 194, 129, 66)
        self._add('977', 'Golden Brown Light', 220, 156, 86)
        self._add('986', 'Forest Green Very Dark', 64, 82, 48)
        self._add('987', 'Forest Green Dark', 88, 113, 65)
        self._add('988', 'Forest Green Medium', 115, 139, 91)
        self._add('989', 'Forest Green ', 141, 166, 117)
        self._add('991', 'Aquamarine Dark', 71, 123, 110)
        self._add('992', 'Aquamarine Light', 111, 174, 159)
        self._add('993', 'Aquamarine Very Light', 144, 192, 180)
        self._add('995', 'Electric Blue Dark', 38, 150, 182)
        self._add('996', 'Electric Blue Medium', 48, 194, 236)
        self._add('B5200', 'Snow White', 255, 255, 255)
        self._add('Ecru', 'Ecru', 240, 234, 218)
        self._add('White', 'White', 252, 251, 248)
