# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['betterletter']

package_data = \
{'': ['*'],
 'betterletter': ['resources/*',
                  'resources/dicts/*',
                  'resources/dicts/filtered/README.md']}

install_requires = \
['pyperclip>=1.8.2,<2.0.0']

entry_points = \
{'console_scripts': ['betterletter = betterletter.__main__:main']}

setup_kwargs = {
    'name': 'betterletter',
    'version': '1.0.0',
    'description': 'Substitute alternative spellings of native characters (e.g. German umlauts [ae, oe, ue] etc. [ss]) with their correct versions (ä, ö, ü, ß).',
    'long_description': "# betterletter\n\nIn a given text, replaces alternative spellings of native characters with their proper spellings.\n\nFor example, German native characters and their corresponding alternative spellings (e.g. when no proper keyboard layout is at hand, or ASCII is used) are:\n\n| Native Character | Alternative Spelling |\n| :--------------: | :------------------: |\n|       Ä/ä        |        Ae/ae         |\n|       Ö/ö        |        Oe/oe         |\n|       Ü/ü        |        Ue/ue         |\n|       ẞ/ß        |        SS/ss         |\n\nThese pairings are recorded [here](betterletter/resources/languages.json).\n\nGoing from left to right is simple: replace all native characters with their alternative spellings, minding case.\nThat use case is also supported by this tool (`reverse` flag).\n\nThe other direction is much less straightforward: there exist countless words for which alternative spellings occur somewhere as a pattern, yet replacing them with the corresponding native character would be wrong:\n\n| Character | Correct Spelling  | Wrong Spelling |\n| --------- | ----------------- | -------------- |\n| *Ä*       | **Ae**rodynamik   | Ärodynamik     |\n| *Ä*       | Isr**ae**l        | Isräl          |\n| *Ä*       | Schuf**ae**intrag | Schufäintrag   |\n| *Ö*       | K**oe**ffizient   | Köffizient     |\n| *Ö*       | Domin**oe**ffekt  | Dominöffekt    |\n| *Ö*       | P**oet**          | Pöt            |\n| *Ü*       | Abente**ue**r     | Abenteür       |\n| *Ü*       | Ma**ue**r         | Maür           |\n| *Ü*       | Ste**ue**rung     | Steürung       |\n| *ß*       | Me**ss**gerät     | Meßgerät       |\n| *ß*       | Me**ss**e         | Meße           |\n| *ß*       | Abschlu**ss**     | Abschluß       |\n\njust to name a few, pretty common examples.\n\nAs such, this tool is based on a dictionary lookup, see also the [containing directory](betterletter/resources/dicts/).\n\n## Examples\n\nSee also the [tests](tests/).\n\n### de\n\nThe input:\n\n> Ueberhaupt braeuchte es mal einen Teststring.\n> Saetze ohne Bedeutung, aber mit vielen Umlauten.\n> DRPFA-Angehoerige gehoeren haeufig nicht dazu.\n> Bindestrich-Woerter spraechen Baende ueber Fehler.\n> Doppelgaenger-Doppelgaenger sind doppelt droelfzig.\n> Oder Uemlaeuten? Auslaeuten? Leute gaebe es, wuerde man meinen.\n> Ueble Nachrede ist naechtens nicht erlaubt.\n> Erlaube man dieses, waere es schoen uebertrieben.\n> Busse muesste geloest werden, bevor Gruesse zum Gruss kommen.\n> Busse sind Geraete, die womoeglich schnell fuehren.\n> Voegel sind aehnlich zu Oel.\n> Hierfuer ist fuer den droegen Poebel zu beachten, dass Anmassungen zu Gehoerverlust fuehren koennen.\n> Stroemelschnoesseldaemel!\n\nis turned into:\n\n> Überhaupt bräuchte es mal einen Teststring.\n> Sätze ohne Bedeutung, aber mit vielen Umlauten.\n> DRPFA-Angehörige gehören häufig nicht dazu.\n> Bindestrich-Wörter sprächen Bände über Fehler.\n> Doppelgänger-Doppelgänger sind doppelt droelfzig.\n> Oder Uemlaeuten? Auslaeuten? Leute gäbe es, würde man meinen.\n> Üble Nachrede ist nächtens nicht erlaubt.\n> Erlaube man dieses, wäre es schön übertrieben.\n> Buße müsste gelöst werden, bevor Grüße zum Gruß kommen.\n> Buße sind Geräte, die womöglich schnell führen.\n> Vögel sind ähnlich zu Öl.\n> Hierfür ist für den drögen Pöbel zu beachten, dass Anmaßungen zu Gehörverlust führen können.\n> Stroemelschnoesseldaemel!\n\n---\n\nNote that some corrections are out of scope for this little script, e.g.:\n\n> Busse\n\nIn German, *Busse* and *Buße* are two words of vastly different meaning (*busses* and *penance*, respectively).\nUnfortunately, they map to the same alternative spelling of *Busse*.\nThe tool sees *Busse* (meaning *just that*, with no intent of changing it), notices *Buße* is a legal substitution, and therefore makes it.\nThe tool has no awareness of context.\n\nTurning substitutions like these off would mean the tool would no longer emit *Buße*, ever.\nThis could be as undesirable as the current behaviour.\nThere seems to be no easy resolve.\n\n## Running\n\n### Prerequisites\n\nIdeally, run the project (as is good, albeit annoying practice) in its own virtual environment.\nThis project uses [poetry](https://python-poetry.org/) for dependency management.\nRefer to the [poetry config file](pyproject.toml) for more info (e.g. the required Python modules to install if you don't want to deal with `poetry`).\n\nUsing poetry, from the project root, run:\n\n```bash\n# Installs virtual environment according to lock file (if available in repo),\n# otherwise pyproject.toml:\npoetry install\n# Run command within that environment:\npoetry run python -m betterletter -h\n```\n\n### Usage\n\nUsage help (invoke from this project's root) will display all options:\n\n```bash\npoetry run python -m betterletter -h\n```\n\nThe tool can read from STDIN (outputting to STDOUT), or work with the clipboard (overwriting its contents with a corrected version).\nThis allows for example:\n\n```bash\n$ cat test.txt\nHoeflich fragen!\n$ cat test.txt | poetry run python -m betterletter de\nHöflich fragen!\n# Reverse mode:\n$ cat test.txt | poetry run python -m betterletter de | poetry run python -m betterletter -r de\nHoeflich fragen!\n```\n\nor\n\n```bash\npoetry run python -m betterletter -c de\n# Nothing happens: clipboard is read and written to silently.\n```\n\nAfter installing (see below) the package, these calls should work system-wide.\n\n## Development\n\nDevelopment tasks are all run through `poetry`, within the context of the virtual environment.\nThe latter is created through\n\n```bash\npoetry install\n```\n\nand then accessed through either `poetry run <command>` or `poetry shell`.\n\nRun `make` (without arguments) for more available commands.\n\n## AutoHotKey\n\nThis tool can be integrated with [AutoHotKey](https://www.autohotkey.com/), allowing you to use it at the touch of a button.\nThis can be used to setup a keyboard shortcut to run this tool in-place, quickly replacing what you need without leaving your text editing environment.\n\nI was unable to use `poetry` commands for this, getting\n\n> The system cannot find the file specified.\n\nIt works with plain `python` invocations.\nThanks to `SetWorkingDir`, we do *not* have to install the module system-wide.\n*However*, the requirements need to be installed and available to the plain `python` command.\n\nThe AutoHotKey file is [here](betterletter.ahk).\n\nFollow [this guide](https://www.autohotkey.com/docs/FAQ.htm#Startup) to have the script launch on boot automatically.\n",
    'author': 'Alex Povel',
    'author_email': 'python@alexpovel.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/alexpovel/betterletter/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
