#!/usr/bin/env python
"""
Prisma SASE Python SDK - WebSocket Functions

**Author:** Palo Alto Networks

**Copyright:** © 2023 Palo Alto Networks. All rights reserved

**License:** MIT
"""
import logging

__author__ = "Prisma SASE Developer Support <prisma-sase-developers@paloaltonetworks.com>"
__email__ = "prisma-sase-developers@paloaltonetworks.com"
__copyright__ = "Copyright © 2023 Palo Alto Networks. All rights reserved"
__license__ = """
    MIT License

    Copyright © 2023 Palo Alto Networks. All rights reserved

    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all
    copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
    SOFTWARE.
"""

# Set logging to function name
api_logger = logging.getLogger(__name__)
"""`logging.getlogger` object to enable debug printing via `prisma_sase.API.set_debug`"""


class WebSockets(object):
    """
    Prisma SASE Python SDK - WebSocket Functions

    Object to handle WebSocket operations.
    """

    # placeholder for parent class namespace
    _parent_class = None

    def toolkit_session(self, element_id, tenant_id=None, api_version="v2.0", cols=207, rows=53, **kwargs):
        """
        Open a Toolkit Session WebSocket

          **Parameters:**:

          - **element_id**: Element ID
          - **tenant_id**: Tenant ID
          - **api_version**: API version to use (default v2.0)
          - **cols**: Optional: Integer, Number of columns for terminal (default 207)
          - **rows**: Optional: Integer, Number of rows for terminal (default 53)
          - **&ast;&ast;kwargs**: Optional: Additional Keyword Arguments to pass to `websockets.client.Connect()`

        **Returns:** `websockets.client.Connect` object.
        """

        if tenant_id is None and self._parent_class.tenant_id:
            # Pull tenant_id from parent namespace cache.
            tenant_id = self._parent_class.tenant_id

        # set controller, converting protocol to wss
        wss_ctlr = self._parent_class.controller.replace('https://', 'wss://', 1)

        url = str(wss_ctlr) + "/sdwan/{}/api/elements/{}/ws/toolkitsessions?cols={}&rows={}" \
                              "".format(api_version, element_id, cols, rows)

        api_logger.debug("URL = %s", url)
        return self._parent_class.websocket_call(url, **kwargs)

    def default(self, tenant_id=None, api_version="v2.0", **kwargs):
        """
        Open the default Tenant WebSocket for use in multiple functions.

          **Parameters:**:

          - **tenant_id**: Tenant ID
          - **api_version**: API version to use (default v2.0)
          - **&ast;&ast;kwargs**: Optional: Additional Keyword Arguments to pass to `websockets.client.Connect()`

        **Returns:** `websockets.client.Connect` object.
        """

        if tenant_id is None and self._parent_class.tenant_id:
            # Pull tenant_id from parent namespace cache.
            tenant_id = self._parent_class.tenant_id

        # set controller, converting protocol to wss
        wss_ctlr = self._parent_class.controller.replace('https://', 'wss://', 1)

        url = str(wss_ctlr) + "/sdwan/{}/api/ws" \
                              "".format(api_version)

        api_logger.debug("URL = %s", url)
        return self._parent_class.websocket_call(url)
