# -*- coding:utf-8 -*-
# /usr/bin/env python
"""
Date: 2021/6/15 22:26
Desc: 东方财富网-行情首页-上证 A 股-每日行情
"""
import requests
import pandas as pd


def stock_zh_a_spot_em() -> pd.DataFrame:
    """
    东方财富-A股-实时行情
    http://quote.eastmoney.com/center/gridlist.html#hs_a_board
    :return: 实时行情
    :rtype: pandas.DataFrame
    """
    url = "http://35.push2.eastmoney.com/api/qt/clist/get"
    params = {
        "pn": "1",
        "pz": "5000",
        "po": "1",
        "np": "1",
        "ut": "bd1d9ddb04089700cf9c27f6f7426281",
        "fltt": "2",
        "invt": "2",
        "fid": "f3",
        "fs": "m:0 t:6,m:0 t:80,m:1 t:2,m:1 t:23",
        "fields": "f1,f2,f3,f4,f5,f6,f7,f8,f9,f10,f12,f13,f14,f15,f16,f17,f18,f20,f21,f23,f24,f25,f22,f11,f62,f128,f136,f115,f152",
        "_": "1623833739532",
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["data"]["diff"])
    temp_df.columns = ["_",
                       "最新价",
                       "涨跌幅",
                       "涨跌额",
                       "成交量",
                       "成交额",
                       "振幅",
                       "换手率",
                       "市盈率-动态",
                       "量比",
                       "_",
                       "代码",
                       "_",
                       "名称",
                       "最高",
                       "最低",
                       "今开",
                       "昨收",
                       "_",
                       "_",
                       "_",
                       "市净率",
                       "_",
                       "_",
                       "_",
                       "_",
                       "_",
                       "_",
                       "_",
                       "_",
                       "_",
                       ]
    temp_df.reset_index(inplace=True)
    temp_df['index'] = range(1, len(temp_df)+1)
    temp_df.rename(columns={'index': "序号"}, inplace=True)
    temp_df = temp_df[[
        "序号",
        "代码",
        "名称",
        "最新价",
        "涨跌幅",
        "涨跌额",
        "成交量",
        "成交额",
        "振幅",
        "最高",
        "最低",
        "今开",
        "昨收",
        "量比",
        "换手率",
        "市盈率-动态",
        "市净率",
    ]]
    temp_df['最新价'] = pd.to_numeric(temp_df['最新价'], errors='coerce')
    temp_df['涨跌幅'] = pd.to_numeric(temp_df['涨跌幅'], errors='coerce')
    temp_df['涨跌额'] = pd.to_numeric(temp_df['涨跌额'], errors='coerce')
    temp_df['成交量'] = pd.to_numeric(temp_df['成交量'], errors='coerce')
    temp_df['成交额'] = pd.to_numeric(temp_df['成交额'], errors='coerce')
    temp_df['振幅'] = pd.to_numeric(temp_df['振幅'], errors='coerce')
    temp_df['最高'] = pd.to_numeric(temp_df['最高'], errors='coerce')
    temp_df['最低'] = pd.to_numeric(temp_df['最低'], errors='coerce')
    temp_df['今开'] = pd.to_numeric(temp_df['今开'], errors='coerce')
    temp_df['昨收'] = pd.to_numeric(temp_df['昨收'], errors='coerce')
    temp_df['量比'] = pd.to_numeric(temp_df['量比'], errors='coerce')
    temp_df['换手率'] = pd.to_numeric(temp_df['换手率'], errors='coerce')
    temp_df['市盈率-动态'] = pd.to_numeric(temp_df['市盈率-动态'], errors='coerce')
    temp_df['市净率'] = pd.to_numeric(temp_df['市净率'], errors='coerce')
    return temp_df


def _code_id_map() -> dict:
    """
    东方财富-股票和市场代码
    http://quote.eastmoney.com/center/gridlist.html#hs_a_board
    :return: 股票和市场代码
    :rtype: dict
    """
    url = "http://80.push2.eastmoney.com/api/qt/clist/get"
    params = {
        "pn": "1",
        "pz": "5000",
        "po": "1",
        "np": "1",
        "ut": "bd1d9ddb04089700cf9c27f6f7426281",
        "fltt": "2",
        "invt": "2",
        "fid": "f3",
        "fs": "m:1 t:2,m:1 t:23",
        "fields": "f12",
        "_": "1623833739532",
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["data"]["diff"])
    temp_df["market_id"] = 1
    temp_df.columns = ["sh_code", "sh_id"]
    code_id_dict = dict(zip(temp_df["sh_code"], temp_df["sh_id"]))
    params = {
        "pn": "1",
        "pz": "5000",
        "po": "1",
        "np": "1",
        "ut": "bd1d9ddb04089700cf9c27f6f7426281",
        "fltt": "2",
        "invt": "2",
        "fid": "f3",
        "fs": "m:0 t:6,m:0 t:80",
        "fields": "f12",
        "_": "1623833739532",
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df_sz = pd.DataFrame(data_json["data"]["diff"])
    temp_df_sz["sz_id"] = 0
    code_id_dict.update(dict(zip(temp_df_sz["f12"], temp_df_sz["sz_id"])))
    return code_id_dict


def stock_zh_a_hist(
    symbol: str = "600070",
    start_date: str = "19700101",
    end_date: str = "22220101",
    adjust: str = "",
) -> pd.DataFrame:
    """
    东方财富网-行情首页-上证A股-每日行情
    http://quote.eastmoney.com/concept/sh603777.html?from=classic
    :param symbol: 股票代码
    :type symbol: str
    :param start_date: 开始日期
    :type start_date: str
    :param end_date: 结束日期
    :type end_date: str
    :param adjust: choice of {"qfq": "1", "hfq": "2", "": "不复权"}
    :type adjust: str
    :return: 每日行情
    :rtype: pandas.DataFrame
    """
    code_id_dict = _code_id_map()
    adjust_dict = {"qfq": "1", "hfq": "2", "": "0"}
    url = "http://push2his.eastmoney.com/api/qt/stock/kline/get"
    params = {
        "fields1": "f1,f2,f3,f4,f5,f6",
        "fields2": "f51,f52,f53,f54,f55,f56,f57,f58,f59,f60,f61",
        "ut": "7eea3edcaed734bea9cbfc24409ed989",
        "klt": "101",
        "fqt": adjust_dict[adjust],
        "secid": f"{code_id_dict[symbol]}.{symbol}",
        "beg": "0",
        "end": "20500000",
        "_": "1623766962675",
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame([item.split(",") for item in data_json["data"]["klines"]])
    temp_df.columns = [
        "日期",
        "开盘",
        "收盘",
        "最高",
        "最低",
        "成交量",
        "成交额",
        "振幅",
        "涨跌幅",
        "涨跌额",
        "换手率",
    ]
    temp_df.index = pd.to_datetime(temp_df["日期"])
    temp_df = temp_df[start_date:end_date]
    temp_df.reset_index(inplace=True, drop=True)
    temp_df = temp_df.astype(
        {
            "开盘": float,
            "收盘": float,
            "最高": float,
            "最低": float,
            "成交量": int,
            "成交额": float,
            "振幅": float,
            "涨跌幅": float,
            "涨跌额": float,
            "换手率": float,
        }
    )
    return temp_df


def stock_hk_spot_em() -> pd.DataFrame:
    """
    东方财富-港股-实时行情
    http://quote.eastmoney.com/center/gridlist.html#hk_stocks
    :return: 港股-实时行情
    :rtype: pandas.DataFrame
    """
    url = "http://72.push2.eastmoney.com/api/qt/clist/get"
    params = {
        'pn': '1',
        'pz': '5000',
        'po': '1',
        'np': '1',
        'ut': 'bd1d9ddb04089700cf9c27f6f7426281',
        'fltt': '2',
        'invt': '2',
        'fid': 'f3',
        'fs': 'm:128 t:3,m:128 t:4,m:128 t:1,m:128 t:2',
        'fields': "f1,f2,f3,f4,f5,f6,f7,f8,f9,f10,f12,f13,f14,f15,f16,f17,f18,f20,f21,f23,f24,f25,f22,f11,f62,f128,f136,f115,f152",
        '_': '1624010056945'
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["data"]["diff"])
    temp_df.columns = [
        "_",
        "最新价",
        "涨跌幅",
        "涨跌额",
        "成交量",
        "成交额",
        "振幅",
        "换手率",
        "市盈率-动态",
        "量比",
        "_",
        "代码",
        "_",
        "名称",
        "最高",
        "最低",
        "今开",
        "昨收",
        "_",
        "_",
        "_",
        "市净率",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
                       ]
    temp_df.reset_index(inplace=True)
    temp_df['index'] = range(1, len(temp_df)+1)
    temp_df.rename(columns={'index': "序号"}, inplace=True)
    temp_df = temp_df[[
        "序号",
        "代码",
        "名称",
        "最新价",
        "涨跌额",
        "涨跌幅",
        "今开",
        "最高",
        "最低",
        "昨收",
        "成交量",
        "成交额",
    ]]
    return temp_df


def stock_hk_hist(
        symbol: str = "00593",
        start_date: str = "19700101",
        end_date: str = "22220101",
        adjust: str = "",
) -> pd.DataFrame:
    """
    东方财富网-行情首页-港股-每日行情
    http://quote.eastmoney.com/hk/08367.html
    :param symbol: 港股-每日行情
    :type symbol: str
    :param start_date: 开始日期
    :type start_date: str
    :param end_date: 结束日期
    :type end_date: str
    :param adjust: choice of {"qfq": "1", "hfq": "2", "": "不复权"}
    :type adjust: str
    :return: 每日行情
    :rtype: pandas.DataFrame
    """
    adjust_dict = {"qfq": "1", "hfq": "2", "": "0"}
    url = "http://33.push2his.eastmoney.com/api/qt/stock/kline/get"
    params = {
        "secid": f"116.{symbol}",
        "ut": "fa5fd1943c7b386f172d6893dbfba10b",
        "fields1": "f1,f2,f3,f4,f5,f6",
        "fields2": "f51,f52,f53,f54,f55,f56,f57,f58,f59,f60,f61",
        "klt": "101",
        "fqt": adjust_dict[adjust],
        "end": "20500000",
        'lmt': '1000000',
        "_": "1623766962675",
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame([item.split(",") for item in data_json["data"]["klines"]])
    temp_df.columns = [
        "日期",
        "开盘",
        "收盘",
        "最高",
        "最低",
        "成交量",
        "成交额",
        "振幅",
        "涨跌幅",
        "涨跌额",
        "换手率",
    ]
    temp_df.index = pd.to_datetime(temp_df["日期"])
    temp_df = temp_df[start_date:end_date]
    temp_df.reset_index(inplace=True, drop=True)
    temp_df['开盘'] = pd.to_numeric(temp_df['开盘'])
    temp_df['收盘'] = pd.to_numeric(temp_df['收盘'])
    temp_df['最高'] = pd.to_numeric(temp_df['最高'])
    temp_df['最低'] = pd.to_numeric(temp_df['最低'])
    temp_df['成交量'] = pd.to_numeric(temp_df['成交量'])
    temp_df['成交额'] = pd.to_numeric(temp_df['成交额'])
    temp_df['振幅'] = pd.to_numeric(temp_df['振幅'])
    temp_df['涨跌幅'] = pd.to_numeric(temp_df['涨跌幅'])
    temp_df['涨跌额'] = pd.to_numeric(temp_df['涨跌额'])
    temp_df['换手率'] = pd.to_numeric(temp_df['换手率'])
    return temp_df


def stock_us_spot_em() -> pd.DataFrame:
    """
    东方财富-美股-实时行情
    http://quote.eastmoney.com/center/gridlist.html#us_stocks
    :return: 美股-实时行情; 延迟 15 min
    :rtype: pandas.DataFrame
    """
    url = "http://72.push2.eastmoney.com/api/qt/clist/get"
    params = {
        'pn': '1',
        'pz': '20000',
        'po': '1',
        'np': '1',
        'ut': 'bd1d9ddb04089700cf9c27f6f7426281',
        'fltt': '2',
        'invt': '2',
        'fid': 'f3',
        'fs': 'm:105,m:106,m:107',
        'fields': "f1,f2,f3,f4,f5,f6,f7,f8,f9,f10,f12,f13,f14,f15,f16,f17,f18,f20,f21,f23,f24,f25,f26,f22,f33,f11,f62,f128,f136,f115,f152",
        '_': '1624010056945'
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json["data"]["diff"])
    temp_df.columns = [
        "_",
        "最新价",
        "涨跌幅",
        "涨跌额",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "简称",
        "编码",
        "名称",
        "最高价",
        "最低价",
        "开盘价",
        "昨收价",
        "总市值",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "_",
        "市盈率",
        "_",
        "_",
        "_",
        "_",
        "_",
                       ]
    temp_df.reset_index(inplace=True)
    temp_df['index'] = range(1, len(temp_df)+1)
    temp_df.rename(columns={'index': "序号"}, inplace=True)
    temp_df["代码"] = temp_df["编码"].astype(str) + '.' + temp_df["简称"]
    temp_df = temp_df[[
        "序号",
        "名称",
        "最新价",
        "涨跌额",
        "涨跌幅",
        "开盘价",
        "最高价",
        "最低价",
        "昨收价",
        "总市值",
        "市盈率",
        "代码",
    ]]
    return temp_df


def stock_us_hist(
        symbol: str = "105.LI",
        start_date: str = "19700101",
        end_date: str = "22220101",
        adjust: str = "",
) -> pd.DataFrame:
    """
    东方财富网-行情首页-美股-每日行情
    http://quote.eastmoney.com/us/ENTX.html#fullScreenChart
    :param symbol: 股票代码; 此股票代码需要通过调用 ak.stock_us_spot_em 的 `代码` 字段获取
    :type symbol: str
    :param start_date: 开始日期
    :type start_date: str
    :param end_date: 结束日期
    :type end_date: str
    :param adjust: choice of {"qfq": "1", "hfq": "2", "": "不复权"}
    :type adjust: str
    :return: 每日行情
    :rtype: pandas.DataFrame
    """
    adjust_dict = {"qfq": "1", "hfq": "2", "": "0"}
    url = "http://63.push2his.eastmoney.com/api/qt/stock/kline/get"
    params = {
        "secid": f"{symbol}",
        "ut": "fa5fd1943c7b386f172d6893dbfba10b",
        "fields1": "f1,f2,f3,f4,f5,f6",
        "fields2": "f51,f52,f53,f54,f55,f56,f57,f58,f59,f60,f61",
        "klt": "101",
        "fqt": adjust_dict[adjust],
        "end": "20500000",
        'lmt': '1000000',
        "_": "1623766962675",
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame([item.split(",") for item in data_json["data"]["klines"]])
    temp_df.columns = [
        "日期",
        "开盘",
        "收盘",
        "最高",
        "最低",
        "成交量",
        "成交额",
        "振幅",
        "涨跌幅",
        "涨跌额",
        "换手率",
    ]
    temp_df.index = pd.to_datetime(temp_df["日期"])
    temp_df = temp_df[start_date:end_date]
    temp_df.reset_index(inplace=True, drop=True)
    temp_df = temp_df.astype(
        {
            "开盘": float,
            "收盘": float,
            "最高": float,
            "最低": float,
            "成交量": int,
            "成交额": float,
            "振幅": float,
            "涨跌幅": float,
            "涨跌额": float,
            "换手率": float,
        }
    )
    return temp_df


if __name__ == "__main__":
    stock_hk_spot_em_df = stock_hk_spot_em()
    print(stock_hk_spot_em_df)

    stock_hk_hist_df = stock_hk_hist(symbol="01246", start_date="19700101", end_date="22220101", adjust="")
    print(stock_hk_hist_df)

    stock_hk_hist_qfq_df = stock_hk_hist(symbol="00593", start_date="19700101", end_date="22220101", adjust="qfq")
    print(stock_hk_hist_qfq_df)

    stock_hk_hist_hfq_df = stock_hk_hist(symbol="00593", start_date="19700101", end_date="22220101", adjust="hfq")
    print(stock_hk_hist_hfq_df)

    stock_us_spot_em_df = stock_us_spot_em()
    print(stock_us_spot_em_df)

    stock_us_hist_df = stock_us_hist(symbol='105.MTP', start_date="19700101", end_date="22220101", adjust="qfq")
    print(stock_us_hist_df)

    stock_zh_a_spot_em_df = stock_zh_a_spot_em()
    print(stock_zh_a_spot_em_df)

    stock_zh_a_hist_df = stock_zh_a_hist(symbol="002415", start_date="20190301", end_date="20210616", adjust="hfq")
    print(stock_zh_a_hist_df)
