"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 * @stability stable
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     * @stability stable
     */
    static custom(name) {
        return new OAuthScope(name);
    }
    /**
     * Adds a custom scope that's tied to a resource server in your stack.
     *
     * @stability stable
     */
    static resourceServer(server, scope) {
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_IUserPoolResourceServer(server);
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_ResourceServerScope(scope);
        return new OAuthScope(`${server.userPoolResourceServerId}/${scope.scopeName}`);
    }
}
exports.OAuthScope = OAuthScope;
_a = JSII_RTTI_SYMBOL_1;
OAuthScope[_a] = { fqn: "@aws-cdk/aws-cognito.OAuthScope", version: "1.140.0" };
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 *
 * Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client.
 *
 * @stability stable
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
 *
 * @stability stable
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
 *
 * @stability stable
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient.
 *
 * @stability stable
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     * @stability stable
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
_b = JSII_RTTI_SYMBOL_1;
UserPoolClientIdentityProvider[_b] = { fqn: "@aws-cdk/aws-cognito.UserPoolClientIdentityProvider", version: "1.140.0" };
/**
 * Allow users to sign in using 'Sign In With Apple'.
 *
 * A `UserPoolIdentityProviderApple` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.APPLE = new UserPoolClientIdentityProvider('SignInWithApple');
/**
 * Allow users to sign in using 'Facebook Login'.
 *
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Google Login'.
 *
 * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.GOOGLE = new UserPoolClientIdentityProvider('Google');
/**
 * Allow users to sign in using 'Login With Amazon'.
 *
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool.
 *
 * @stability stable
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client.
 *
 * @stability stable
 */
class UserPoolClient extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_cognito_UserPoolClientProps(props);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_e = (_d = props.oAuth) === null || _d === void 0 ? void 0 : _d.flows) !== null && _e !== void 0 ? _e : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_f = props.oAuth) === null || _f === void 0 ? void 0 : _f.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 && !props.disableOAuth ? callbackUrls : undefined,
            logoutUrLs: (_g = props.oAuth) === null || _g === void 0 ? void 0 : _g.logoutUrls,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
            readAttributes: (_h = props.readAttributes) === null || _h === void 0 ? void 0 : _h.attributes(),
            writeAttributes: (_j = props.writeAttributes) === null || _j === void 0 ? void 0 : _j.attributes(),
            enableTokenRevocation: props.enableTokenRevocation,
        });
        this.configureTokenValidity(resource, props);
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     *
     * @stability stable
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     *
     * @stability stable
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        if (!props.authFlows)
            return undefined;
        const authFlows = [];
        if (props.authFlows.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if (props.authFlows.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if (props.authFlows.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        // refreshToken should always be allowed if authFlows are present
        if (authFlows.length > 0) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _d;
        const scopes = (_d = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _d !== void 0 ? _d : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
    configureTokenValidity(resource, props) {
        this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.idTokenValidity);
        this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), core_1.Duration.days(1), props.accessTokenValidity);
        this.validateDuration('refreshTokenValidity', core_1.Duration.minutes(60), core_1.Duration.days(10 * 365), props.refreshTokenValidity);
        if (props.refreshTokenValidity) {
            this.validateDuration('idTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.idTokenValidity);
            this.validateDuration('accessTokenValidity', core_1.Duration.minutes(5), props.refreshTokenValidity, props.accessTokenValidity);
        }
        if (props.accessTokenValidity || props.idTokenValidity || props.refreshTokenValidity) {
            resource.tokenValidityUnits = {
                idToken: props.idTokenValidity ? 'minutes' : undefined,
                accessToken: props.accessTokenValidity ? 'minutes' : undefined,
                refreshToken: props.refreshTokenValidity ? 'minutes' : undefined,
            };
        }
        ;
        resource.idTokenValidity = props.idTokenValidity ? props.idTokenValidity.toMinutes() : undefined;
        resource.refreshTokenValidity = props.refreshTokenValidity ? props.refreshTokenValidity.toMinutes() : undefined;
        resource.accessTokenValidity = props.accessTokenValidity ? props.accessTokenValidity.toMinutes() : undefined;
    }
    validateDuration(name, min, max, value) {
        if (value === undefined) {
            return;
        }
        if (value.toMilliseconds() < min.toMilliseconds() || value.toMilliseconds() > max.toMilliseconds()) {
            throw new Error(`${name}: Must be a duration between ${min.toHumanString()} and ${max.toHumanString()} (inclusive); received ${value.toHumanString()}.`);
        }
    }
}
exports.UserPoolClient = UserPoolClient;
_c = JSII_RTTI_SYMBOL_1;
UserPoolClient[_c] = { fqn: "@aws-cdk/aws-cognito.UserPoolClient", version: "1.140.0" };
//# sourceMappingURL=data:application/json;base64,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